﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  サーバーに対して発行する API です。
 */

#include <nn/fs/fs_SaveDataTransferForCloudBackUp.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/util/util_Execution.h>

#include <curl/curl.h>

namespace nn { namespace olsc { namespace srv { namespace transfer {

    /**
    *       @brief      URL を含んだ ComponentFile の情報を取得します。
    *
    *       @param[out]  pOut   サーバーから返ってくる ComponentFileInfo。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   id   ComponentFileInfo の ID。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result GetComponentFileSignedUrlForUpload(ComponentFileInfo* pOut, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、ComponentFile の新規作成を要求します。
    *
    *       @param[out]  outComponentFile   サーバーから返ってくる ComponentFile。
    *       @param[in]   SaveDataArchiveId   SaveDataArchive の ID。
    *       @param[in]   clientArgument   作成する ComponentFile に与えるクライアントが管理する情報。
    *       @param[in]   type   ComponentFile のタイプ。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result CreateComponentFile(ComponentFileInfo* out, SaveDataArchiveId saveDataArchiveId, const ClientArgument& clientArgument, ComponentFileType type, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、ComponentFile の更新を要求します。
    *
    *       @param[out]  out   サーバーから返ってくる ComponentFileInfo。
    *       @param[in]   id   ComponentFile の ID。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result UpdateComponentFile(ComponentFileInfo* out, ComponentFileId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      S3 に対して、指定した URL にデータをアップロードします。
    *
    *       @param[out]  pOutSize   アップロードしたサイズ。
    *       @param[out]  pOutDigest   アップロードしたデータのダイジェスト。
    *       @param[in]   url   アップロード先のサーバの URL。
    *       @param[in]   exporter   アップロードするデータの exporter。
    *       @param[in]   id   アップロードするデータの SaveDataChunkId。
    *       @param[in]   nsaIdToken   使用する NSA ID トークン。
    *       @param[in]   curlHandle   内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer   ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result ExportComponentFile(size_t* pOutSize, SaveDataChunkDigest* pOutDigest, const char* url, fs::ISaveDataDivisionExporter* exporter, fs::SaveDataChunkId id, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

    /**
    *       @brief      ストレージサーバに対して、 ComponentFile 保存完了処理を要求します。
    *
    *       @param[in]   id   ComponentFile の ID。
    *       @param[in]   archiveDataSize   S3 にアップロードするデータのサイズ。
    *       @param[in]   saveDataChunkDigest   S3 にアップロードしたデータのダイジェスト。
    *       @param[in]   nsaIdToken    使用する NSA ID トークン。
    *       @param[in]   curlHandle    内部の通信処理に使用する libcURL のハンドル。
    *       @param[in]   workBuffer    ワークバッファ。
    *       @param[in]   workBufferSize   ワークバッファのサイズ。
    *       @param[in]   pCancelable   中止を検知するためのオブジェクト。
    *
    *       @return     処理結果を返します。
    *
    */
    Result CompleteComponentFile(ComponentFileId id, size_t archiveDataSize, const SaveDataChunkDigest& saveDataChunkDigest, const NsaIdToken& nsaIdToken, CURL* curlHandle, void* workBuffer, size_t workBufferSize, const nn::util::Cancelable* pCancelable) NN_NOEXCEPT;

}}}} // namespace nn::olsc::srv::transfer
