﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <mutex>
#include <nn/nn_Result.h>
#include <nn/olsc/detail/olsc_Log.h>
#include <nn/olsc/olsc_ResultPrivate.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_Out.h>

namespace nn { namespace olsc { namespace srv {

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
class SystemEventManager
{
public:
    SystemEventManager() NN_NOEXCEPT;
    ~SystemEventManager() NN_NOEXCEPT;

    void Signal() NN_NOEXCEPT;
    Result AcquireSystemEvent(os::SystemEventType* outValue) NN_NOEXCEPT;
    Result ReleaseSystemEvent(os::SystemEventType* systemEvent) NN_NOEXCEPT;


    class NativeHandleHolder
    {
    public:
        NN_IMPLICIT NativeHandleHolder(SystemEventManager& systemEventManager) NN_NOEXCEPT;
        ~NativeHandleHolder() NN_NOEXCEPT;
        Result Initialize() NN_NOEXCEPT;
        Result GetNativeHandle(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT;
    private:
        SystemEventManager& m_SystemEventManager;
        os::SystemEventType m_SystemEvent;
        bool m_IsInitialized;
    };

private:
    os::SdkRecursiveMutex m_Lock;
    std::array<os::SystemEventType*, MaxEventCount> m_Storage;
    int m_CurrentCount;
};

using TransferTaskCompleteEventManager = SystemEventManager<5, os::EventClearMode::EventClearMode_ManualClear, true>;
using TransferTaskStartEventManager = SystemEventManager<5, os::EventClearMode::EventClearMode_ManualClear, true>;

}}} // ~namespace nn::olsc::srv


// -----------------------------------------------------------------------
// 実装

#include <nn/olsc/olsc_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace olsc { namespace srv {

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::SystemEventManager() NN_NOEXCEPT
    : m_CurrentCount(0)
{}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::~SystemEventManager() NN_NOEXCEPT
{
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
Result SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::AcquireSystemEvent(os::SystemEventType* outValue) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_RESULT_THROW_UNLESS(m_CurrentCount < MaxEventCount, olsc::ResultOutOfSystemEvent());

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateSystemEvent(outValue, EventClearMode, InterProcess));
    m_Storage[m_CurrentCount++] = outValue;

    NN_RESULT_SUCCESS;
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
Result SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::ReleaseSystemEvent(os::SystemEventType* systemEvent) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (int i = 0; i < m_CurrentCount; ++i)
    {
        if (m_Storage[i] == systemEvent)
        {
            os::DestroySystemEvent(systemEvent);
            m_Storage[i] = m_Storage[m_CurrentCount - 1];
            m_CurrentCount--;
            NN_RESULT_SUCCESS;
        }
    }

    NN_ABORT("Not come here.\n");
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
void SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::Signal() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (int i = 0; i < m_CurrentCount; ++i)
    {
        os::SignalSystemEvent(m_Storage[i]);
    }
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::NativeHandleHolder::NativeHandleHolder(SystemEventManager& systemEventManager) NN_NOEXCEPT : m_SystemEventManager(systemEventManager)
{
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::NativeHandleHolder::~NativeHandleHolder() NN_NOEXCEPT
{
    if (m_IsInitialized)
    {
        m_SystemEventManager.ReleaseSystemEvent(&m_SystemEvent);
    }
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
Result SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::NativeHandleHolder::Initialize() NN_NOEXCEPT
{
    NN_RESULT_DO(m_SystemEventManager.AcquireSystemEvent(&m_SystemEvent));
    m_IsInitialized = true;
    NN_RESULT_SUCCESS;
}

template<int MaxEventCount, os::EventClearMode EventClearMode, bool InterProcess>
Result SystemEventManager<MaxEventCount, EventClearMode, InterProcess>::NativeHandleHolder::GetNativeHandle(sf::Out<sf::NativeHandle> outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    *outValue = sf::NativeHandle(os::GetReadableHandleOfSystemEvent(&m_SystemEvent), false);
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::olsc::srv
