﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/nn_ApplicationId.h>
#include <nn/nn_Result.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_SaveData.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class TransferTaskDatabaseManagerBase
{
    NN_DISALLOW_COPY(TransferTaskDatabaseManagerBase);

protected:
    NN_IMPLICIT TransferTaskDatabaseManagerBase(util::DefaultMountManager& mountManager) NN_NOEXCEPT
        :   m_TaskDatabase(mountManager),
            m_ContextDatabase(mountManager), m_MountManager(mountManager)
    {}


    using WriteTransaction = std::function<Result(TransferTaskDatabase& taskDatabase, TransferTaskContextDatabase& contextDatabase)>;
    Result PerformWriteTransaction(const WriteTransaction& transaction) NN_NOEXCEPT;

    using ReadTransaction = std::function<Result(const TransferTaskDatabase& taskDatabase, const TransferTaskContextDatabase& contextDatabase)>;
    void PerformReadTransaction(const ReadTransaction& transaction) const NN_NOEXCEPT;

public:
    os::SystemEvent& GetExecutionListUpdatedEvent() NN_NOEXCEPT;

private:

    TransferTaskDatabase            m_TaskDatabase;
    TransferTaskContextDatabase     m_ContextDatabase;
    mutable os::SdkRecursiveMutex   m_Lock;
    util::DefaultMountManager&      m_MountManager;

};

class TransferTaskDatabaseManager : public TransferTaskDatabaseManagerBase
{
public:
    NN_IMPLICIT TransferTaskDatabaseManager(util::DefaultMountManager& mountManager) NN_NOEXCEPT
        : TransferTaskDatabaseManagerBase(mountManager)
    {}

    using FindTaskPredicate = TransferTaskDatabase::FindTaskPredicate;
    nn::util::optional<TransferTaskDetailInfo> FindDetailInfo(const FindTaskPredicate& predicate) const NN_NOEXCEPT;
    Result GetTransferTaskCount(uint32_t* out) const NN_NOEXCEPT;

    Result RemoveTransferTask(TransferTaskId id) NN_NOEXCEPT;
    Result RemoveAutoUploadTaskByUser(const account::Uid& uid) NN_NOEXCEPT;
    Result RegisterTransferTask(TransferTaskId* out, const account::Uid& uid, ApplicationId appId, const TransferTaskConfig& config, TransferTaskRank rank) NN_NOEXCEPT;

    using TransferTaskContext = database::TransferTaskContextDatabase::TransferTaskContext;

    nn::util::optional<TransferTaskContext> GetTransferTaskContext(TransferTaskId id) NN_NOEXCEPT;
    nn::util::optional<TransferTaskDetailInfo> GetTransferTaskDetailInfo(TransferTaskId id) NN_NOEXCEPT;

    Result RemoveTransferTaskContext(TransferTaskId id) NN_NOEXCEPT;

    // TODO: この辺はインターフェース要検討
    Result SetCompleted(TransferTaskId id) NN_NOEXCEPT;
    Result SetFailed(TransferTaskId id, Result result) NN_NOEXCEPT;
    Result SetSuspended(TransferTaskId id, const TransferTaskContext& context) NN_NOEXCEPT;
    Result SetRetry(TransferTaskId id) NN_NOEXCEPT;

    Result MoveTransferTaskToHead(TransferTaskId id) NN_NOEXCEPT;
    int ListDetailInfo(TransferTaskDetailInfo out[], int maxOutCount, int offset) NN_NOEXCEPT;

    // デバッグ用
    Result RemoveAllTransferTask() NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database

