﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  ポリシー情報キャッシュ
 */

#include <functional>
#include <mutex>
#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_SortedDataArray.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/util/util_LockGuard.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

struct PolicyInfoCacheEntry
{
    int launchRequiredVersion;
    int latestVersion;
    PolicyInfo policyInfo;
};

inline bool operator==(const PolicyInfoCacheEntry& lhs, const PolicyInfoCacheEntry& rhs) NN_NOEXCEPT
{
    return lhs.launchRequiredVersion == rhs.launchRequiredVersion
        && lhs.latestVersion == rhs.latestVersion
        && lhs.policyInfo == rhs.policyInfo;
}

inline bool operator!=(const PolicyInfoCacheEntry& lhs, const PolicyInfoCacheEntry& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

class PolicyInfoCache
{
    NN_DISALLOW_COPY(PolicyInfoCache);
    NN_DISALLOW_MOVE(PolicyInfoCache);

private:
    static const int MaxEntryCount = MaxApplicationCount;

public:

    NN_IMPLICIT PolicyInfoCache(util::DefaultMountManager& mountManager) NN_NOEXCEPT
        :   m_Database(mountManager)
    {}

    util::WriteMount AcquireWriteMount() NN_NOEXCEPT
    {
        return m_Database.AcquireWriteMount();
    }

    nn::util::optional<PolicyInfoCacheEntry> Get(ApplicationId appId) const NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Database);
        return m_Database.Find(appId);
    }
    void Add(ApplicationId appId, const PolicyInfoCacheEntry& entry) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Database);
        if (!m_Database.Add(appId, entry))
        {
            // TODO: 削除するものを一番昔に追加したものにする
            //       m_Database とは別にリングバッファ的なものを永続化しておき
            //       順番を管理するなど
            m_Database.Remove(m_Database[0].key);
            NN_ABORT_UNLESS(m_Database.Add(appId, entry));
        }
    }
    void Remove(ApplicationId appId) NN_NOEXCEPT
    {
        NN_UTIL_LOCK_GUARD(m_Database);
        m_Database.Remove(appId);
    }

private:
    static const int ReadBufferCount = 32;
    class Database : public SortedDataArray<ApplicationId, PolicyInfoCacheEntry, ReadBufferCount>
    {
    public:
        NN_IMPLICIT Database(util::DefaultMountManager& mountManager) :
            SortedDataArray(CompareKey, &m_ReadBuffer, MaxEntryCount), m_MountManager(mountManager)
        {}
        virtual util::WriteMount AcquireWriteMount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireDeviceSaveForWrite();
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "policy_info_meta";
        }
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "policy_info_entry";
        }
        virtual util::ReadMount AcquireReadMount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireDeviceSaveForRead();
        }
    private:
        static int CompareKey(const ApplicationId& lhs, const ApplicationId& rhs) NN_NOEXCEPT
        {
            if (lhs.value < rhs.value)
            {
                return -1;
            }
            else if (lhs.value > rhs.value)
            {
                return 1;
            }
            return 0;
        }
        ReadBuffer m_ReadBuffer;
        util::DefaultMountManager& m_MountManager;
    };

    mutable Database m_Database;
};

}}}} // namespace nn::olsc::srv::database
