﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <mutex>
#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_DataArray.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace olsc { namespace srv { namespace database {

class ExecutionList
{
    NN_DISALLOW_COPY(ExecutionList);
    NN_DISALLOW_MOVE(ExecutionList);

private:
    struct ExecutionInfoEntry
    {
        TransferTaskId      id;
        account::Uid        uid;

        inline bool operator==(const ExecutionInfoEntry& rhs) const NN_NOEXCEPT
        {
            return this->id == rhs.id
                && this->uid == rhs.uid;
        }
        inline bool operator!=(const ExecutionInfoEntry& rhs) const NN_NOEXCEPT
        {
            return !(*this == rhs);
        }

        inline static ExecutionInfoEntry Make(TransferTaskId id, const account::Uid& uid) NN_NOEXCEPT
        {
            return ExecutionInfoEntry{
                id, uid
            };
        }
    };

public:
    struct ExecutionInfo
    {
        TransferTaskId      id;
        account::Uid        uid;
        TransferTaskRank    rank;

        inline bool operator==(const ExecutionInfo& rhs) const NN_NOEXCEPT
        {
            return this->id == rhs.id
                && this->uid == rhs.uid
                && this->rank == rhs.rank;
        }
        inline bool operator!=(const ExecutionInfo& rhs) const NN_NOEXCEPT
        {
            return !(*this == rhs);
        }

        inline static ExecutionInfo Make(TransferTaskId id, const account::Uid& uid, TransferTaskRank rank) NN_NOEXCEPT
        {
            return ExecutionInfo{
                id, uid, rank
            };
        }

        inline static ExecutionInfo Make(const ExecutionInfoEntry& entry, TransferTaskRank rank) NN_NOEXCEPT
        {
            return Make(entry.id, entry.uid, rank);
        }
    };


    NN_IMPLICIT ExecutionList(util::DefaultMountManager& mountManager) NN_NOEXCEPT;

    int GetCount() const NN_NOEXCEPT;
    nn::util::optional<ExecutionInfo> FindIf(const std::function<bool(const ExecutionInfo&)>& pred) const NN_NOEXCEPT;
    Result At(ExecutionInfo* out, int index) const NN_NOEXCEPT;
    bool HasSpace(TransferTaskRank rank) const NN_NOEXCEPT;

    Result PushFront(const ExecutionInfo& info) NN_NOEXCEPT;
    Result PushBack(const ExecutionInfo& info) NN_NOEXCEPT;

    void RemoveAll(const std::function<bool(const ExecutionInfo&)>& pred) NN_NOEXCEPT;
    nn::util::optional<ExecutionInfo> Remove(TransferTaskId tid) NN_NOEXCEPT;
    nn::util::optional<ExecutionInfo> RemoveIf(const std::function<bool(const ExecutionInfo&)>& pred) NN_NOEXCEPT;

    Result Cleanup() NN_NOEXCEPT;

    void lock() NN_NOEXCEPT;
    void lock() const NN_NOEXCEPT;
    void unlock() NN_NOEXCEPT;
    void unlock() const NN_NOEXCEPT;

    void ForEach(const std::function<bool(const ExecutionInfo&)>& pred) NN_NOEXCEPT;
private:
    static const int CacheCount = 1024;
    static const int ReadBufferCount = 256;

    class ListImpl : public DataArray<util::DefaultMountManager, ExecutionInfoEntry, MaxTransferTaskCount, CacheCount, ReadBufferCount>
    {
    public:
        NN_IMPLICIT ListImpl(const char* prefix, TransferTaskRank rank, util::DefaultMountManager& mountManager) NN_NOEXCEPT;
        TransferTaskRank GetRank() const NN_NOEXCEPT;

    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::ReadMount AcquireReadMount(util::DefaultMountManager& mountManager) const NN_NOEXCEPT NN_OVERRIDE;
        virtual util::WriteMount AcquireWriteMount(util::DefaultMountManager& mountManager) NN_NOEXCEPT NN_OVERRIDE;

    private:
        CacheBuffer m_CacheBuffer;
        ReadBuffer m_ReadBuffer;
        char m_MetadataPath[64];
        char m_EntryPath[64];
        TransferTaskRank m_Rank;
    };


    ListImpl m_BasicList;
    ListImpl m_PostponedList;

    void ForEachList(const std::function<bool(ListImpl& list)>& pred) NN_NOEXCEPT;
    void ForEachList(const std::function<bool(const ListImpl& list)>& pred) const NN_NOEXCEPT;
};

}}}} // namespace nn::olsc::srv::database
