﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  エラー履歴データベース
 */

#include <functional>
#include <mutex>
#include <nn/account/account_Types.h>
#include <nn/nn_ApplicationId.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <nn/olsc/srv/database/olsc_SortedDataArray.h>
#include <nn/olsc/srv/util/olsc_MountContext.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_PosixTime.h>
#include <nn/util/util_Optional.h>
#include <utility>

namespace nn { namespace olsc { namespace srv { namespace database {

class ErrorHistoryDatabase
{
    NN_DISALLOW_COPY(ErrorHistoryDatabase);
    NN_DISALLOW_MOVE(ErrorHistoryDatabase);

public:
    struct ErrorInfo
    {
        TransferTaskKind taskKind;
        bool isRetryable;
        time::PosixTime registeredTime;
        Result result;
    };

    NN_IMPLICIT ErrorHistoryDatabase(util::DefaultMountManager& mountManager) NN_NOEXCEPT
        :   m_Database(mountManager)
    {}

    nn::util::optional<ErrorInfo> GetLastErrorInfo(const account::Uid& uid, ApplicationId appId) const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Database)> lock(m_Database);

        return m_Database.Find({ appId, uid });
    }

    bool SetLastError(const account::Uid& uid, ApplicationId appId, TransferTaskKind taskKind, bool isRetryable, Result result) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Database)> lock(m_Database);
        time::PosixTime currentTime;
        auto getCurrentTimeResult = time::StandardNetworkSystemClock::GetCurrentTime(&currentTime);
        if (getCurrentTimeResult.IsFailure())
        {
            NN_DETAIL_OLSC_TRACE("SetLastError() uses StandardUserSystemClock instead of StandardNetworkSystemClock because network clock has troubled(%08x).\n", getCurrentTimeResult.GetInnerValueForDebug());
            NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardUserSystemClock::GetCurrentTime(&currentTime));
        }

        ErrorInfo errorInfo = {
            taskKind,
            isRetryable,
            currentTime,
            result
        };

        return m_Database.Add({ appId, uid }, errorInfo);
    }

    bool SetLastError(const TransferTaskErrorInfo& errorInfo) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Database)> lock(m_Database);

        ErrorInfo errorInfoValue = {
            errorInfo.kind,
            errorInfo.isRetryable,
            errorInfo.registeredTime,
            errorInfo.lastResult,
        };

        return m_Database.Add({ errorInfo.applicationId, errorInfo.uid }, errorInfoValue);
    }

    void RemoveLastErrorInfo(const account::Uid& uid, ApplicationId appId) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Database)> lock(m_Database);

        m_Database.Remove({ appId, uid });
    }

    int ListLastErrorInfoAsTransferTaskErrorInfo(TransferTaskErrorInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT
    {
        return ListImpl<TransferTaskErrorInfo>(out, maxOutCount, offset, [](const std::pair<ApplicationId, account::Uid>& key, const ErrorInfo& ei)-> TransferTaskErrorInfo
        {
            return {
                key.second,
                key.first,
                ei.taskKind,
                ei.isRetryable,
                ei.registeredTime,
                ei.result,
            };
        });
    }

    int ListLastErrorInfo(ErrorInfo out[], int maxOutCount, int offset) const NN_NOEXCEPT
    {
        return ListImpl<ErrorInfo>(out, maxOutCount, offset, [](const std::pair<ApplicationId, account::Uid>&, const ErrorInfo& ei) -> ErrorInfo
        {
            return ei;
        });
    }

    void Cleanup() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Database)> lock(m_Database);

        m_Database.Cleanup();
    }

    int GetLastErrorInfoCount() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Database)> lock(m_Database);

        return m_Database.GetCount();
    }

    util::WriteMount AcquireWriteMount() NN_NOEXCEPT
    {
        return m_Database.AcquireWriteMount();
    }

    // TODO: ユーザ削除時等で呼ぶ uid 指定で全部消すやつ、CleanupByUserId 的なやつが必要

private:

    static const int ReadBufferCount = 32;
    static const int MaxEntryCount = MaxApplicationCount * account::UserCountMax;
    class Database : public SortedDataArray<std::pair<ApplicationId, account::Uid> , ErrorInfo, ReadBufferCount>
    {
    public:
        Database(util::DefaultMountManager& mountManager) :
            SortedDataArray(CompareKey, &m_ReadBuffer, MaxEntryCount), m_MountManager(mountManager)
        {}
        virtual util::WriteMount AcquireWriteMount() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireDeviceSaveForWrite();
        }
    protected:
        virtual const char* GetMetadataFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "err_history_meta";
        }
        virtual const char* GetEntryFileRelativePath() const NN_NOEXCEPT NN_OVERRIDE
        {
            return "err_history_entry";
        }
        virtual util::ReadMount AcquireReadMount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_MountManager.AcquireDeviceSaveForRead();
        }
    private:
        static int CompareUid(const account::Uid& lhs, const account::Uid& rhs) NN_NOEXCEPT
        {
            NN_STATIC_ASSERT(sizeof(account::Uid::_data) == sizeof(Bit64) * 2);
            for (int i = 0; i < 2; ++i)
            {
                if (lhs._data[i] < rhs._data[i])
                {
                    return -1;
                }
                else if (lhs._data[i] > rhs._data[i])
                {
                    return 1;
                }
            }
            return 0;
        }

        static int CompareKey(const std::pair<ApplicationId, account::Uid>& lhs, const std::pair<ApplicationId, account::Uid>& rhs) NN_NOEXCEPT
        {
            if (lhs.first.value < rhs.first.value)
            {
                return -1;
            }
            else if (lhs.first.value > rhs.first.value)
            {
                return 1;
            }

            // appId が等しい場合は uid を比較
            return CompareUid(lhs.second, rhs.second);
        }
        ReadBuffer m_ReadBuffer;
        util::DefaultMountManager& m_MountManager;
    };

    template<typename OutputType>
    int ListImpl(OutputType out[], int maxOutCount, int offset,
        std::function<OutputType(const std::pair<ApplicationId, account::Uid>&, const ErrorInfo&)> converter) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(offset >= 0);

        std::lock_guard<decltype(m_Database)> lock(m_Database);

        int listedCount;
        for (listedCount = 0; listedCount < maxOutCount && offset + listedCount < m_Database.GetCount(); ++listedCount)
        {
            auto kv = m_Database[listedCount + offset];
            out[listedCount] = converter(kv.key, kv.value);
        }
        return listedCount;
    }

    mutable Database m_Database;
};

}}}} // namespace nn::olsc::srv::database
