﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief  システムサービス向けの API
 */

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/olsc/olsc_DaemonController.h>
#include <nn/olsc/olsc_TransferTaskTypes.h>

namespace nn { namespace olsc {

    class TransferTaskListController;
    class RemoteStorageController;

    /**
    *   @brief      有効な DaemonController を取得します。
    *   @return     DaemonController のインスタンス。
    */
    DaemonController OpenDaemonController() NN_NOEXCEPT;

    /**
    *   @brief      有効な TransferTaskListController を取得します。
    *   @return     TransferTaskListController のインスタンス。
    */
    TransferTaskListController OpenTransferTaskListController() NN_NOEXCEPT;

    /**
    *   @brief      有効な RemoteStorageController を取得します。
    *   @return     RemoteStorageController のインスタンス。
    */
    RemoteStorageController OpenRemoteStorageController(const account::Uid& uid) NN_NOEXCEPT;

    /**
    *   @brief      転送タスクのエラー情報の数を取得します。
    *
    *   @return     本体に記録されている転送タスクのエラー履歴の数。
    */
    int GetTransferTaskErrorInfoCount() NN_NOEXCEPT;

    /**
    *   @brief      転送タスクのエラー情報を列挙します。
    *
    *   @param[out] out             エラー情報の列挙先。
    *   @param[in]  maxOutCount     最大列挙数。
    *   @param[in]  offset          列挙を開始するオフセット。
    *
    *   @return     列挙した数。
    *
    *   @details    uid, applicationId 毎に最新の1件のエラーが記録されています。
    *               この API ではそれらのエラーをリストアップすることができます。
    *               エラー情報は uid と applicationId が一致するタスクが成功した時に自動的に削除されます。
    *               @ref RemoveTransferTaskErrorInfo() を使用することで明示的に削除することもできます。
    *
    */
    int ListTransferTaskErrorInfo(TransferTaskErrorInfo out[], int maxOutCount, int offset) NN_NOEXCEPT;

    /**
    *   @brief      転送タスクのエラー情報を削除します。
    *
    *   @param[in]  uid             対象のユーザ。
    *   @param[in]  applicationId   対象のアプリケーション。
    *
    *   @return     なし。
    */
    void RemoveTransferTaskErrorInfo(const account::Uid& uid, ApplicationId applicationId) NN_NOEXCEPT;

    /**
    *   @brief 廃止予定の API です。代わりに @ref OpenTransferTaskListController() を使用するようにしてください。
    */
    void OpenTransferTaskListController(TransferTaskListController* out) NN_NOEXCEPT;
    /**
    *   @brief 廃止予定の API です。代わりに @ref OpenRemoteStorageController(const account::Uid& uid) を使用するようにしてください。
    */
    void OpenRemoteStorageController(RemoteStorageController* out, const account::Uid& uid) NN_NOEXCEPT;


}} // namespace nn::olsc
