﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   HDCP 制御に関する API の宣言
 *
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>

namespace nn { namespace oe {

/**
 * @brief       HDCP 認証状態を表す列挙体です。
 */
enum HdcpAuthenticationState
{
    HdcpAuthenticationState_Unauthenticated,              //!< 未認証状態
    HdcpAuthenticationState_Processing,                   //!< 認証処理中
    HdcpAuthenticationState_Authenticated,                //!< 認証完了
};

//-----------------------------------------------------------------------------

//! @name   HDCP 制御に関する API
//! @{

/**
* @brief   HDCP の認証状態を取得します。
*
* @return  HDCP の認証状態を返します。
*
* @details
*  HDCP の認証状態を取得します。
*  この状態はシステムで管理されています。
*/
HdcpAuthenticationState GetHdcpAuthenticationState() NN_NOEXCEPT;

/**
* @brief   HDCP の認証状態が変更されたときのシステムイベントを取得します。
*
* @param[out]   pEvent   取得したシステムイベントの格納先
*
* @details
*  HDCP の認証状態が変更されたときにシグナルされるシステムイベントを取得します。
*  こちらのシステムイベントのクリアモードは nn::os::EventClearMode_AutoClear に設定されています。
*
*  アプリケーションの実行中、HDCP 認証を有効にするには、
*  @confluencelink{211655259, 「アプリケーションの実行中、HDCP 認証を有効にする」} に従って nmeta ファイルを編集して下さい。
*  以下では HDCP 認証を有効に設定されたアプリケーションを VOD アプリケーションと呼びます。
*
*  VOD アプリケーションが起動していないとき、 HdcpAuthenticationState_Unauthenticated になっています。
*  以下のいずれかの場合になったとき、 HdcpAuthenticationState_Unauthenticated から HdcpAuthenticationState_Processing へ移行します。
*
*  - 動作モードが据置モードのとき、VOD アプリケーションを起動した
*  - 動作モードが携帯モードでかつ VOD アプリケーションが前面で動作中のとき、動作モードが据置モードへ切り替わった
*
*  HdcpAuthenticationState_Processing から HdcpAuthenticationState_Authenticated へは表示先のディスプレイにもよりますが、
*  約 3 秒程度経過したのちに移行します。
*
*  以下のいずれかの場合になったとき、 HdcpAuthenticationState_Unauthenticated へ移行します。
*
*  - VOD アプリケーションが終了した
*  - 動作モードが携帯モードへと切り替わった
*  - HDCP 認証に失敗し、システムが認証失敗画面を表示した
*
*  HDCP 認証に失敗したときは即座にシステムでハンドリングし、失敗した旨を画面に表示します。
*  このため基本的には本 API や GetHdcpAuthenticationState() を使用する必要はありません。
*  HDCP 認証を開始してから完了するまでの間に映像コンテンツなどを表示したくない場合、
*  これらの API を使用してください。
*
*  なお、動作モードの取得については nn::oe::GetOperationMode() を使用してください。
*
*/
void GetHdcpAuthenticationStateChangeEvent(os::SystemEvent* pEvent) NN_NOEXCEPT;

//! @}

}} // namespace nn::oe
