﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ntc/detail/service/ntc_Common.h>
#include <nn/util/util_IntrusiveList.h>
#include <functional>

namespace nn { namespace ntc { namespace detail { namespace service { namespace core {

class TaskExecutor
{
NN_DISALLOW_COPY(TaskExecutor);
NN_DISALLOW_MOVE(TaskExecutor);
public:
    TaskExecutor() NN_NOEXCEPT {}
    virtual nn::Result Execute() NN_NOEXCEPT = 0;
};

class TaskResource
{
NN_DISALLOW_COPY(TaskResource);
NN_DISALLOW_MOVE(TaskResource);

public:
    explicit TaskResource(nn::Result initialResult, TaskExecutor* pTaskExecutor):
        m_SystemEvent(nn::os::EventClearMode_AutoClear, true),
        InitialResult(initialResult),
        m_Result(InitialResult),
        m_pTaskExecutor(pTaskExecutor)
    {
    }

    //!< タスク完了通知イベント
    nn::os::SystemEvent m_SystemEvent;

    //!< タスク再登録時の Result 初期化用のため保持
    const nn::Result InitialResult;

    //!< タスク結果(イベントシグナル後に正しい値が入る)
    std::atomic<nn::Result> m_Result;

    //!< タスク実行クラス
    TaskExecutor* const m_pTaskExecutor;
};

class TaskResourceNode : public nn::util::IntrusiveListBaseNode<TaskResourceNode>
{
private:
    TaskResource* m_pTaskResource;

public:
    TaskResourceNode() NN_NOEXCEPT:
        m_pTaskResource(nullptr)
    {
    }

    explicit TaskResourceNode(TaskResource* pTaskResource) NN_NOEXCEPT:
        m_pTaskResource(pTaskResource)
    {
        NN_SDK_ASSERT_NOT_NULL(pTaskResource);
    }

    bool operator==(const TaskResourceNode& node) NN_NOEXCEPT
    {
        return m_pTaskResource == node.m_pTaskResource;
    }

    bool operator!=(const TaskResourceNode& node) NN_NOEXCEPT
    {
        return !(this->m_pTaskResource == node.m_pTaskResource);
    }

    TaskResourceNode& operator=(const TaskResourceNode& node) NN_NOEXCEPT
    {
        this->m_pTaskResource = node.m_pTaskResource;
        return *this;
    }

    TaskResource* GetTaskResourcePtr() const NN_NOEXCEPT
    {
        return m_pTaskResource;
    }
};

typedef nn::util::IntrusiveList<TaskResourceNode, nn::util::IntrusiveListBaseNodeTraits<TaskResourceNode>> TaskResourceIntrusiveList;
class SerializeTaskThread;

class TaskResourceList
{
NN_DISALLOW_COPY(TaskResourceList);
NN_DISALLOW_MOVE(TaskResourceList);

public:
    TaskResourceList() NN_NOEXCEPT;

    //!< タスクリストが空かどうか
    bool IsEmpty() const NN_NOEXCEPT;

    //!< 先頭タスクの実行
    void ExecuteFrontTask() NN_NOEXCEPT;

    //!< 指定ノードを result で完了状態に遷移させる
    bool Done(nn::Result result, TaskResourceNode& node) NN_NOEXCEPT;

    //!< すべてのノードを result で完了状態に遷移させる
    void DoneAll(nn::Result result) NN_NOEXCEPT;

    //!< タスクをリスト末尾に追加
    bool PushBack(TaskResourceNode& node) NN_NOEXCEPT;

    //!< タスクが追加済であるかどうか
    bool IsPushed(TaskResourceNode& node) NN_NOEXCEPT;

    //!< タスクを削除
    bool Remove(TaskResourceNode& node) NN_NOEXCEPT;

    nn::os::Event& GetTaskEvent() NN_NOEXCEPT
    {
        return m_TaskEvent;
    }

private:
    TaskResourceIntrusiveList m_List;
    mutable nn::os::Mutex m_Mutex;
    nn::os::Event m_TaskEvent;
};


/**
 * @brief   1 スレッドでタスク実行を行うクラスです
 * @details
 *  RegisterTask() で1つでもタスクが登録されると、TaskResource の m_pTaskExecutor->Execute() がスレッドで実行されます。
 *
 *  StartThread(), StopThread() はスレッドアンセーフです。
 */
class SerializeTaskThread
{
    NN_DISALLOW_COPY(SerializeTaskThread);
    NN_DISALLOW_MOVE(SerializeTaskThread);

private:
    nn::os::ThreadType m_Thread;
    nn::os::Event m_StopEvent; //!< スレッド終了要求があればシグナルするイベント
    TaskResourceList m_TaskResourceList; //!< タスク実行待ちリスト

    static void ThreadFunction(void* p) NN_NOEXCEPT
    {
        reinterpret_cast<SerializeTaskThread*>(p)->ThreadFunctionImpl();
    }

    void ThreadFunctionImpl() NN_NOEXCEPT;

public:
    SerializeTaskThread() NN_NOEXCEPT;

    //!< スレッド開始.
    void StartThread(char* stack, size_t stackSize, int priority, const char* threadName) NN_NOEXCEPT;

    //! スレッド終了.登録済タスクリソースに渡す nn::Result を指定する.スレッド完了まで待機.
    void StopThread(nn::Result remainTaskResult) NN_NOEXCEPT;

    //!< タスク登録. タスク処理終了時に taskResource.m_SystemEvent がシグナルする.
    bool RegisterTask(TaskResourceNode& taskResourceNode) NN_NOEXCEPT;

    //!< 登録済タスクリソースの削除.登録されていない場合は何もしない.
    bool UnregisterTaskResource(TaskResourceNode& taskResourceNode) NN_NOEXCEPT;

    //!< 指定タスクが登録されているかどうか
    bool IsRegisteredTask(TaskResourceNode& taskResourceNode) NN_NOEXCEPT;

    //!< タスクを指定の Result で終了.(キャンセルに使えます)
    bool DoneTask(nn::Result doneResult, TaskResourceNode& taskResourceNode) NN_NOEXCEPT;
};

}}}}} // nn::ntc::detail::service::core
