﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace nsd { namespace detail { namespace util {

    /**
     * @brief   文字列が同じかチェック
     * @tparam  T   T::Size, char value[Size] をメンバに持つ型
     * @param[in]   rhs
     * @param[in]   lhs
     * @return  true    同じ
     * @return  false   違う
     */
    template <typename T>
    bool IsEqualString(const T& rhs, const char* lhs) NN_NOEXCEPT
    {
        return nn::util::Strncmp(rhs.value, lhs, static_cast<int>(T::Size)) == 0;
    }

    /**
     * @brief   文字列が同じかチェック
     * @tparam  T   T::Size, char value[Size] をメンバに持つ型
     * @param[in]   rhs
     * @param[in]   lhs
     * @return  true    同じ
     * @return  false   違う
     */
    template <typename T>
    bool IsEqualString(const T& rhs, const T& lhs) NN_NOEXCEPT
    {
        return nn::util::Strncmp(rhs.value, lhs.value, static_cast<int>(T::Size)) == 0;
    }

    /**
     * @brief   文字列セット
     * @tparam  T   T::Size, char value[Size] をメンバに持つ型
     * @param[out]  pOut
     * @param[in]   str
     * @retval  true    pOut にすべて格納された
     * @retval  false   pOut に切れて格納された
     */
    template <typename T>
    bool SetString(T* pOut, const char* str) NN_NOEXCEPT
    {
        return nn::util::Strlcpy(pOut->value, str, static_cast<int>(T::Size)) < static_cast<int>(T::Size);
    }

    /**
     * @brief   文字列クリア
     * @tparam  T   T::Size, char value[Size] をメンバに持つ型
     * @param[out]  pOut
     */
    template <typename T>
    void ClearString(T* pOut) NN_NOEXCEPT
    {
        std::memset(pOut->value, 0, T::Size);
    }

    /**
     * @brief   文字列長の取得
     * @tparam  T   T::Size, char value[Size] をメンバに持つ型
     * @param[in]   str
     * @return  文字列長
     */
    template <typename T>
    int GetLength(const T& str) NN_NOEXCEPT
    {
        return nn::util::Strnlen(str.value, static_cast<int>(T::Size));
    }

    /**
     * @brief   空の文字列かどうか
     * @tparam  T   T::Size, char value[Size] をメンバに持つ型
     * @param[in]   str
     * @retval  true    空である
     * @retval  false   空でない
     */
    template <typename T>
    bool IsEmptyString(const T& str) NN_NOEXCEPT
    {
        return str.value[0] == '\0';
    }

    /*
     * @brief   begin から end の直前まで検索し c と一致するポインタを返す. NULL 終端が見つかった時点で検索終了.
     * @param[in]   begin   検索開始のポインタ
     * @param[in]   end     検索終了のポインタ(end 直前のポインタまで検索)
     * @param[in]   c       検索対象の文字
     *
     * @return 検索ヒットした要素を指すポインタ
     * @retval nullptr        見つからなかった
     * @retval nullptr以外    見つかった
     */
    const char* SearchCharWithEndPtr(const char* begin, const char* end, char c) NN_NOEXCEPT;

    /*
     * @brief   src 中の c のポインタを返す
     * @param[in]   src     検索をかけたい文字列
     * @param[in]   length  文字列の長さ
     * @param[in]   c       検索対象の文字
     *
     * @return 検索ヒットした src の要素を指すポインタ
     * @retval nullptr        見つからなかった
     * @retval nullptr以外    見つかった
     */
    const char* SearchCharWithLenght(const char* src, int length, char c) NN_NOEXCEPT;

}}}} // nn::nsd::detail::util
