﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 雑多な内部実装 API のヘッダ
 */

#pragma once

#include <nn/nsd/nsd_Common.h>
#include <nn/nsd/nsd_Types.h>
#include <nn/nsd/nsd_TypesPrivate.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace nsd { namespace detail {

    /**
     * @brief FQDN の解決
     * @param[out]  pOut
     * @param[in]   fqdn
     * @param[in]   saveData
     * @param[in]   currentTime
     */
    Result Resolve(
        Fqdn *pOut,
        const Fqdn& fqdn,
        const SaveData& saveData,
        const nn::time::PosixTime& currentTime) NN_NOEXCEPT;

    /**
     * @brief FQDN に含まれる '%' を環境識別子へ変換します。
     * @param[out]  pOut
     * @param[in]   fqdn
     * @param[in]   identifier
     *
     * @pre
     *  - pOut != nullptr
     *
     * @return
     * @retval  ResultSuccess           成功しました。
     * @retval  ResultFqdnTooLong       処理結果の FQDN が長すぎます。
     * @retval  ResultInvalidArgument   入力が不正です。(入力が NULL 終端していない場合などに返ります)
     */
    Result InsertEnvironmentIdentifier(Fqdn* pOut, const Fqdn& fqdn, const EnvironmentIdentifier& identifier) NN_NOEXCEPT;

    /**
     * @brief FQDN に不正な文字列が含まれているかを判定
     * @param[in] fqdn
     * @retval  true    不正な文字列が存在しました
     * @retval  false   不正な文字列が存在しませんでした
     * @details
     *  true が返っても正しい FQDN であるとは限りません。（ラベルの長さなどのチェックは行っていません）
     */
    bool HasBadChar(const Fqdn& fqdn) NN_NOEXCEPT;

    /**
     * @brief サーバーレスポンスのエラーコードを nn::Result 型へ変換します。
     * @param[in]   errorCode   エラーコード文字列
     * @return  エラーコードに対応する Result
     */
    Result HandleServerError(const char* errorCode) NN_NOEXCEPT;

    /**
     * @brief セーブデータが初期状態かどうか
     */
    bool IsInitialSaveData(const SaveData& saveData) NN_NOEXCEPT;

    /**
     * @brief セーブデータの正当性検証
     * @param[in]   saveData                検証対象のセーブデータ
     * @param[in]   ownDeviceId             自分のデバイス ID
     * @param[in]   isDeviceIdCheckEnabled  デバイス ID チェックを行うかどうか
     * @details
     *  製品機かどうかをハンドリングしで、適切な環境識別子を返します。
     *
     *  以下をチェックしています.@n
     *  - 必須の設定値が空になっていないか
     *  - デバイス ID が自身のものと同じか(isDeviceIdCheckEnabled == true の場合のみ)
     *  - FQDN として許可していない文字がないか
     */
    Result VerifySaveData(
        const SaveData& saveData,
        const DeviceId& ownDeviceId,
        bool isDeviceIdCheckEnabled) NN_NOEXCEPT;

    /**
     * @brief デフォルトの設定値を取得
     */
    void GetDefaultSettings(SaveData* pOut, const DeviceId& ownDeviceId, const EnvironmentIdentifier& env) NN_NOEXCEPT;

    /**
     * @brief   NAS の認証・トークン取得リクエスト用の FQDN 取得
     */
    Result GetNasRequestFqdn(Fqdn* pOut, const SaveData& saveData, const nn::time::PosixTime& currentTime) NN_NOEXCEPT;

    /**
     * @brief   NAS の API 用の FQDN 取得
     */
    Result GetNasApiFqdn(Fqdn* pOut, const SaveData& saveData, const nn::time::PosixTime& currentTime) NN_NOEXCEPT;

    /**
     * @brief   NAS サービスを利用するための設定取得
     */
    Result GetNasServiceSetting(
        nn::nsd::NasServiceSetting* pOutNasServiceSetting,
        const nn::nsd::NasServiceName& nasServiceName,
        const SaveData& saveData,
        const nn::time::PosixTime& currentTime ) NN_NOEXCEPT;
}}} // nn::nsd::detail
