﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nsd/nsd_Common.h>
#include <nn/nsd/detail/json/nsd_JsonParser.h>
#include <nn/nsd/detail/nsd_Log.h>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Base64.h>

#include <rapidjson/document.h>

namespace nn { namespace nsd { namespace detail { namespace jwt {
    /**
     * @brief   JWT ドキュメントの要素を表す構造体
     */
    struct JwtElements
    {
        const char* pHeader; //!< ヘッダ要素ポインタ
        size_t headerSize; //!< ヘッダサイズ

        const char* pClaim; //!< クレーム要素ポインタ
        size_t claimSize; //!< クレームサイズ

        const char* pSignature; //!< 署名要素ポインタ
        size_t signatureSize; //!< 署名サイズ

        const char* pSignTarget; //!< 署名対象ポインタ
        size_t signTargetSize; //!< 署名対象サイズ
    };

    /**
     * @brief   JWT ドキュメントの解析後のヘッダを表す構造体
     */
    struct JwtHeaderData
    {
        /**
         * @brief ヘッダの "typ" 要素を表す型
         */
        struct TypData
        {
            static const size_t Size = 8;
            char value[Size];
        };
        TypData typ; //!< 'typ' を示す文字列

        /**
         * @brief ヘッダの "alg" 要素を表す型
         */
        struct AlgData
        {
            static const size_t Size = 8;
            char value[Size];
        };
        AlgData alg; //!< 'alg' を示す文字列
    };

    /**
     * @brief   JWT ドキュメントから各要素を取り出す
     * @param[out]  pOut    各要素の出力
     * @param[in]   pJwtDocument        JWT ドキュメントへのポインタ
     * @param[in]   jwtDocumentSize     JWT ドキュメントのサイズ
     */
    Result GetJwtElements(
        JwtElements* pOut,
        const char* pJwtDocument, size_t jwtDocumentSize) NN_NOEXCEPT;

    /**
     * @brief   JWT ドキュメントのヘッダを解析します
     * @param[out]  pOut
     * @param[in]   pHeader
     * @param[in]   headerSize
     */
    Result ParseHeader(JwtHeaderData* pOut, const JwtElements& jwtElements) NN_NOEXCEPT;

    /**
     * @brief   JWT ドキュメントのクレーム要素を解析します。
     * @param[in]   rapidJsonEventHandlerForJwtClaims   Claim 要素の JSON 解析用 rapidjson イベントハンドラ
     * @param[in]   pClaimBegin                         Claim 要素の開始ポインタ
     * @param[in]   claimSize                           Claim 要素のサイズ
     */
    Result ParseClaim(
        json::JsonEventAccepter* rapidJsonEventHandlerForJwtClaims,
        const JwtElements& jwtElements) NN_NOEXCEPT;

    /**
     * @brief   JWT ドキュメントの署名データをデコードします
     * @param[out]  pOutActualSize      署名データの実際のサイズ
     * @param[out]  pOutBuffer          署名データを入れるバッファ
     * @param[in]   outBufferSize       pOutBuffer のサイズ
     * @param[in]   pJwtDocument
     * @param[in]   jwtDocumentSize
     */
    Result DecodeSignatureBuffer(
        size_t* pOutActualSize, void* pOutBuffer, size_t outBufferSize,
        const JwtElements& jwtElements) NN_NOEXCEPT;

    /**
     * @brief   署名データを検証します。
     * @tparam  VerifierType    署名検証を行うクラス(nn::crypto::Rsa2048Pkcs1Sha256Verifier など)
     * @param[in]   verifier
     * @param[in]   pSignature
     * @param[in]   signatureSize
     * @param[in]   pDataBegin
     * @param[in]   dataSize
     */
    template <typename VerifierType>
    Result VerifySignature(
        VerifierType& verifier,
        const char* pSignature, size_t signatureSize,
        const JwtElements& jwtElements) NN_NOEXCEPT
    {
        if(signatureSize != VerifierType::SignatureSize)
        {
            NN_DETAIL_NSD_ERROR("[NSD][DETAIL][JWT] Invalid signature size. expected:%zu actual:%zu\n",
                VerifierType::SignatureSize, jwtElements.signatureSize);
            NN_RESULT_THROW(ResultJwtInvalidSignatureSize());
        }

        NN_SDK_ASSERT_NOT_NULL(jwtElements.pSignTarget);
        verifier.Update(jwtElements.pSignTarget, jwtElements.signTargetSize);

        if(!verifier.Verify(pSignature, signatureSize))
        {
            NN_DETAIL_NSD_ERROR("[NSD][DETAIL][JWT] Signature verify error.\n");
            NN_RESULT_THROW(ResultJwtSignatureVerifyError());
        }
        NN_RESULT_SUCCESS;
    };


}}}} // nn::nsd::detail::jwt
