﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/ns/detail/ns_UserResourceManagementInternalTypes.h>

#include <mutex>

#include <nn/nn_ApplicationId.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/account/account_Types.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>

namespace nn { namespace ns { namespace srv {

class ProgressMonitorForDeleteUserSaveDataAll
{
private:
    mutable os::Mutex m_Lock;

    ns::detail::ProgressForDeleteUserSaveDataAll m_Progress;

    enum class State
    {
        NotInitialized,
        Initialized,
        Finished,
    } m_State;
    bool m_Canceled;
    os::SystemEvent m_Event;
    Result m_Result;

    bool IsInitializedUnsafe() const NN_NOEXCEPT;
    bool IsCanceledUnsafe() const NN_NOEXCEPT;

public:
    ProgressMonitorForDeleteUserSaveDataAll() NN_NOEXCEPT;

    // UserResourceManager 呼び出し側
    void Initialize() NN_NOEXCEPT;
    void Cancel() NN_NOEXCEPT;
    void MarkAsFinished(const Result& result) NN_NOEXCEPT;

    // UserResourceManager 内部向け
    bool IsInitialized() const NN_NOEXCEPT;
    bool IsCanceled() const NN_NOEXCEPT;
    void Update(const UserSaveDataStatistics& stat) NN_NOEXCEPT;
    void Update(const UserSaveDataStatistics& stat, const ncm::ApplicationId& appId) NN_NOEXCEPT;

    // 進捗監視用
    detail::ProgressForDeleteUserSaveDataAll GetProgress() const NN_NOEXCEPT;
    const os::SystemEvent& GetEventRef() NN_NOEXCEPT;
    bool IsFinished() const NN_NOEXCEPT;
    Result GetResult() const NN_NOEXCEPT;
};

class UserResourceManager
{
public:
    static Result DeleteUserSystemSaveData(const account::Uid& uid, const fs::SystemSaveDataId& sysSaveId) NN_NOEXCEPT;
    static Result CalculateUserSaveDataStatistics(UserSaveDataStatistics* pOut, const account::Uid& uid) NN_NOEXCEPT;
    static Result DeleteUserSaveDataAll(ProgressMonitorForDeleteUserSaveDataAll& progress, const account::Uid& uid) NN_NOEXCEPT;

    static Result UnregisterNetworkServiceAccount(const account::Uid& uid) NN_NOEXCEPT;
    static Result UnregisterNetworkServiceAccountWithUserSaveDataDeletion(const account::Uid& uid) NN_NOEXCEPT;
};

}}} // ~namespace nn::ns::srv


// 実装 ------------------------------------------------------------------------------------------

namespace nn { namespace ns { namespace srv {

inline ProgressMonitorForDeleteUserSaveDataAll::ProgressMonitorForDeleteUserSaveDataAll() NN_NOEXCEPT
    : m_Lock(false)
    , m_State(State::NotInitialized)
    , m_Canceled(false)
    , m_Event(os::EventClearMode_ManualClear, true)
    , m_Result(ResultSuccess())
{
}
inline bool ProgressMonitorForDeleteUserSaveDataAll::IsInitializedUnsafe() const NN_NOEXCEPT
{
    return m_State == State::Initialized;
}
inline bool ProgressMonitorForDeleteUserSaveDataAll::IsCanceledUnsafe() const NN_NOEXCEPT
{
    return m_Canceled;
}

inline void ProgressMonitorForDeleteUserSaveDataAll::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_ASSERT(m_State == State::NotInitialized);

    m_Progress._startedAt = os::GetSystemTick().GetInt64Value();
    m_Progress._count = 0u;
    m_Progress._sizeInBytes = 0ull;
    m_Progress._cursor.isSystem = true;
    m_State = State::Initialized;
}
inline bool ProgressMonitorForDeleteUserSaveDataAll::IsInitialized() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return IsInitializedUnsafe();
}
inline void ProgressMonitorForDeleteUserSaveDataAll::Cancel() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_ASSERT(IsInitialized());
    m_Canceled = true;
}
inline bool ProgressMonitorForDeleteUserSaveDataAll::IsCanceled() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return IsCanceledUnsafe();
}
inline void ProgressMonitorForDeleteUserSaveDataAll::Update(const UserSaveDataStatistics& stat) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_ASSERT(IsInitializedUnsafe());

    m_Progress._count = stat.count;
    m_Progress._sizeInBytes = stat.sizeInBytes;
    m_Progress._cursor.isSystem = true;
}
inline void ProgressMonitorForDeleteUserSaveDataAll::Update(const UserSaveDataStatistics& stat, const ncm::ApplicationId& appId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_ASSERT(IsInitializedUnsafe());

    m_Progress._count = stat.count;
    m_Progress._sizeInBytes = stat.sizeInBytes;
    m_Progress._cursor.isSystem = false;
    m_Progress._cursor.identity.applicationId.value = appId.value;
}
inline void ProgressMonitorForDeleteUserSaveDataAll::MarkAsFinished(const Result& result) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_ASSERT(IsInitializedUnsafe());

    m_Result = result;
    m_State = State::Finished;
    m_Event.Signal();
}

inline ns::detail::ProgressForDeleteUserSaveDataAll ProgressMonitorForDeleteUserSaveDataAll::GetProgress() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Progress;
}
inline const os::SystemEvent& ProgressMonitorForDeleteUserSaveDataAll::GetEventRef() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Event;
}
inline bool ProgressMonitorForDeleteUserSaveDataAll::IsFinished() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_State == State::Finished;
}
inline Result ProgressMonitorForDeleteUserSaveDataAll::GetResult() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    return m_Result;
}

}}} // ~namespace nn::ns::srv

