﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_Mount.h>
#include <nn/kvdb/kvdb_FileKeyValueCache.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationControlDataSystemApi.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_Language.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace ns { namespace srv {
    struct ApplicationControlKey
    {
        char data[sizeof(ncm::ApplicationId) + sizeof(uint32_t) + sizeof(settings::LanguageCode)];

        ncm::ApplicationId GetId() const;

        uint32_t GetVersion() const;

        settings::LanguageCode GetLanguageCode() const;

        bool operator == (const ApplicationControlKey& key) const;

        static ApplicationControlKey Make(ncm::ApplicationId id, uint32_t version, settings::LanguageCode languageCode) NN_NOEXCEPT;
    };

    template<class KeyValueCacheT>
    class KeyValueCacheDatabase : private KeyValueCacheT
    {
    private:
        typedef kvdb::BoundedString<32> Path;
        Result EnsureDirectory(const Path& path) NN_NOEXCEPT
        {
            NN_RESULT_TRY(fs::CreateDirectory(path))
                NN_RESULT_CATCH(fs::ResultPathAlreadyExists) {}
            NN_RESULT_END_TRY

            NN_RESULT_SUCCESS;
        }

        Result EnsureDirectoryRecursively(const Path& path) NN_NOEXCEPT
        {
            for (size_t i = 0; i < path.GetLength(); i++)
            {
                if (path.Get()[i] == '/' && path.Get()[i - 1] != ':')
                {
                    NN_RESULT_DO(EnsureDirectory(path.MakeSubString(0, i)));
                }
            }
            NN_RESULT_DO(EnsureDirectory(path));

            NN_RESULT_SUCCESS;
        }

    public:
        typedef typename KeyValueCacheT::KeyType KeyType;
        static const size_t RequiredBufferSize = KeyValueCacheT::RequiredBufferSize;

        Result Initialize(fs::SystemSaveDataId id, const char* mountName, void* buffer, size_t bufferSize, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags) NN_NOEXCEPT
        {
            fs::DisableAutoSaveDataCreation();
            NN_RESULT_TRY(fs::MountSystemSaveData(mountName, id))
                NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                    NN_RESULT_DO(fs::CreateSystemSaveData(id, saveDataSize, saveDataJournalSize, saveDataFlags));
                    NN_RESULT_DO(fs::MountSystemSaveData(mountName, id));
                }
            NN_RESULT_END_TRY
            m_MountName.Assign(mountName);

            Path rootPath;
            rootPath.AssignFormat("%s:/cache", mountName);

            auto result = KeyValueCacheT::Verify(rootPath);
            if (result.IsFailure())
            {
                NN_SDK_LOG("[ApplicationManager] Failed to initialize key value cache as 0x%08x. Destroy and create.\n", result.GetInnerValueForDebug());
                fs::DeleteDirectoryRecursively(rootPath);
                NN_RESULT_DO(EnsureDirectoryRecursively(rootPath));
                NN_RESULT_DO(KeyValueCacheT::Create(rootPath));
                NN_RESULT_DO(Commit());
            }

            NN_RESULT_DO(KeyValueCacheT::Initialize(rootPath, buffer, bufferSize));

            NN_RESULT_SUCCESS;
        }

        void Finalize() NN_NOEXCEPT
        {
            fs::Unmount(m_MountName);
        }

        Result Get(size_t* outValue, void* buffer, size_t bufferSize, const KeyType& key) NN_NOEXCEPT
        {
            return KeyValueCacheT::Get(outValue, buffer, bufferSize, key);
        }

        Result GetSize(size_t* outValue, const KeyType& key) NN_NOEXCEPT
        {
            return KeyValueCacheT::GetSize(outValue, key);
        }

        bool Has(const KeyType& key) NN_NOEXCEPT
        {
            return KeyValueCacheT::Has(key);
        }

        int Count() NN_NOEXCEPT
        {
            return KeyValueCacheT::Count();
        }

        KeyType GetKey(int index) NN_NOEXCEPT
        {
            return KeyValueCacheT::GetKey(index);
        }

        Result Flush() NN_NOEXCEPT
        {
            NN_RESULT_DO(KeyValueCacheT::Flush());
            NN_RESULT_DO(Commit());
            NN_RESULT_SUCCESS;
        }

        Result Put(const KeyType& key, const void* value, size_t valueSize) NN_NOEXCEPT
        {
            NN_RESULT_DO(KeyValueCacheT::Put(key, value, valueSize));
            NN_RESULT_DO(Commit());
            NN_RESULT_SUCCESS;
        }

        Result Delete(const KeyType& key) NN_NOEXCEPT
        {
            NN_RESULT_DO(KeyValueCacheT::Delete(key));
            NN_RESULT_DO(Commit());
            NN_RESULT_SUCCESS;
        }

        Result DeleteAll() NN_NOEXCEPT
        {
            NN_RESULT_DO(KeyValueCacheT::DeleteAll());
            NN_RESULT_DO(Commit());
            NN_RESULT_SUCCESS;
        }

    private:
        Result Commit() NN_NOEXCEPT
        {
            return fs::CommitSaveData(m_MountName);
        }

        Path m_MountName;
    };

#if defined( NN_BUILD_CONFIG_OS_WIN )
    typedef kvdb::FileKeyValueCache<ApplicationControlKey, 2048> ApplicationControlKeyValueCache;
#else
    typedef kvdb::FileKeyValueCache<ApplicationControlKey, 2048, 384> ApplicationControlKeyValueCache;
#endif

    typedef KeyValueCacheDatabase<ApplicationControlKeyValueCache> ApplicationControlDatabase;

    class ApplicationControlDatabaseAccessor
    {
        NN_DISALLOW_COPY(ApplicationControlDatabaseAccessor);

    public:
        // NN_DISALLOW_COPY を宣言するため、デフォルトコンストラクタが必要
        ApplicationControlDatabaseAccessor() {}

        Result Initialize(fs::SystemSaveDataId id, const char* mountName, void* buffer, size_t bufferSize, int64_t saveDataSize, int64_t saveDataJournalSize, int saveDataFlags) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Initialize(id, mountName, buffer, bufferSize, saveDataSize, saveDataJournalSize, saveDataFlags);
        }

        void Finalize() NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            m_Db.Finalize();
        }

        Result Get(size_t* outValue, void* buffer, size_t bufferSize, const ApplicationControlDatabase::KeyType& key) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Get(outValue, buffer, bufferSize, key);
        }

        Result GetSize(size_t* outValue, const ApplicationControlDatabase::KeyType& key) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.GetSize(outValue, key);
        }

        bool Has(const ApplicationControlDatabase::KeyType& key) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Has(key);
        }

        int Count() NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Count();
        }

        ApplicationControlDatabase::KeyType GetKey(int index) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.GetKey(index);
        }

        Result Flush() NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Flush();
        }

        Result Put(const ApplicationControlDatabase::KeyType& key, const void* value, size_t valueSize) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Put(key, value, valueSize);
        }

        Result Delete(const ApplicationControlDatabase::KeyType& key) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.Delete(key);
        }

        Result DeleteAll() NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);
            return m_Db.DeleteAll();
        }

        util::optional<ApplicationControlDatabase::KeyType> FindKeyById(ncm::ApplicationId id) NN_NOEXCEPT
        {
            util::optional<ApplicationControlDatabase::KeyType> foundKey;
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

            ApplicationControlDatabase::KeyType key;
            auto index = FindKeyByIdImpl(&key, id);

            return (index >= 0) ? util::optional<ApplicationControlDatabase::KeyType>(key) : util::nullopt;
        }

        Result GetLatest(size_t* outValue, void* buffer, size_t bufferSize, ncm::ApplicationId id) NN_NOEXCEPT
        {
            util::optional<ApplicationControlDatabase::KeyType> foundKey;
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

            int nextIndex = 0;
            while (NN_STATIC_CONDITION(true))
            {
                ApplicationControlDatabase::KeyType key;
                auto foundIndex = FindKeyByIdImpl(&key, id, nextIndex);
                if (foundIndex < 0)
                {
                    break;
                }
                else if (!foundKey || key.GetVersion() > foundKey->GetVersion())
                {
                    foundKey = key;
                }
                nextIndex = foundIndex + 1;
            }
            NN_RESULT_THROW_UNLESS(foundKey, ResultApplicationControlDataNotFound());

            NN_RESULT_DO(m_Db.Get(outValue, buffer, bufferSize, *foundKey));

            NN_RESULT_SUCCESS;
        }

        Result DeleteById(ncm::ApplicationId id) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

            while (NN_STATIC_CONDITION(true))
            {
                ApplicationControlDatabase::KeyType key;
                auto index = FindKeyByIdImpl(&key, id);
                if (index < 0)
                {
                    break;
                }

                NN_RESULT_DO(m_Db.Delete(key));
            }

            NN_RESULT_SUCCESS;
        }

        Result ListEntryInfo(int* outCount, ApplicationControlCacheEntryInfo outList[], int count) NN_NOEXCEPT
        {
            std::lock_guard<NonRecursiveMutex> guard(m_Mutex);

            auto entryCount = m_Db.Count();

            int readCount = 0;
            for (int i = 0; i < count && i < entryCount; i++)
            {
                auto key = m_Db.GetKey(i);
                size_t size;
                NN_RESULT_DO(m_Db.GetSize(&size, key));

                ApplicationControlCacheEntryInfo info = {};
                info.applicationId = key.GetId();
                info.version = key.GetVersion();
                info.language = key.GetLanguageCode();
                info.iconSize = static_cast<int64_t>(size - sizeof(ApplicationControlProperty));

                outList[readCount] = info;
                readCount++;
            }

            *outCount = readCount;
            NN_RESULT_SUCCESS;
        }

    private:
        int FindKeyByIdImpl(ApplicationControlDatabase::KeyType* outValue, ncm::ApplicationId id, int offset = 0) NN_NOEXCEPT
        {
            int count = m_Db.Count();

            for (int i = offset; i < count; i++)
            {
                auto key = m_Db.GetKey(count - i - 1);
                if (key.GetId().value == id.value)
                {
                    *outValue = key;
                    return i;
                }
            }

            return -1;
        }

    private:
        ApplicationControlDatabase m_Db;
        NonRecursiveMutex m_Mutex;
    };

}}}
