﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ns/ns_ApplicationControlDataSystemApi.h>
#include <nn/ns/ns_ApplicationLaunchInfo.h>
#include <nn/ns/detail/ns_IApplicationManagerInterface.sfdl.h>
#include <nn/ns/srv/ns_ApplicationKeyValueCache.h>
#include <nn/ns/srv/ns_IntegratedContentManager.h>
#include <nn/ns/srv/ns_OsUtil.h>
#include <nn/ns/srv/ns_RequestServer.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace ns { namespace srv {

    class ApplicationRecordDatabase;

    class ApplicationControlDataManager
    {
    public:
        ApplicationControlDataManager() NN_NOEXCEPT : m_RecordDb(), m_Integrated() {}
        Result Initialize(ApplicationRecordDatabase* recordDb, IntegratedContentManager* integrated, RequestServer* requestServer) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        Result GetApplicationControlData(sf::Out<std::uint32_t> outValue, const sf::OutBuffer& buffer, ns::ApplicationControlSource source, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetApplicationControlProperty(ApplicationControlProperty* outValue, const ApplicationLaunchInfo& info, uint8_t programIndex) NN_NOEXCEPT;
        Result InvalidateAllApplicationControlCache() NN_NOEXCEPT;
        Result InvalidateApplicationControlCache(ncm::ApplicationId id) NN_NOEXCEPT;
        Result RequestDownloadApplicationControlData(sf::Out<sf::NativeHandle> outHandle, sf::Out<sf::SharedPointer<ns::detail::IAsyncResult>> outAsync, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetMaxApplicationControlCacheCount(sf::Out<std::int32_t> outValue) NN_NOEXCEPT;

        // alongWithIcon = false は、source に StorageOnly を指定した場合のみ現状利用可能
        Result Get(size_t* outValue, void* buffer, size_t bufferSize, ApplicationControlSource source, ncm::ApplicationId id, bool alongWithIcon = true) NN_NOEXCEPT;
        Result GetByProgramIndex(ApplicationControlProperty* outValue, ncm::ApplicationId id, uint8_t programIndex) NN_NOEXCEPT;

        Result PutPath(ncm::ApplicationId id, uint32_t version, uint8_t programIndex, const ncm::Path& path, bool notifyParentalControl = true) NN_NOEXCEPT;
        bool Has(ncm::ApplicationId id, uint32_t version) NN_NOEXCEPT;
        bool HasAny(ncm::ApplicationId id) NN_NOEXCEPT;

        Result ListApplicationControlCacheEntryInfo(sf::Out<std::int32_t> outCount, const sf::OutArray<ns::ApplicationControlCacheEntryInfo>& outList) NN_NOEXCEPT;

    private:
        Result GetByKey(size_t* outValue, void* buffer, size_t bufferSize, ApplicationControlSource source, ApplicationControlKey cacheKey, util::optional<ncm::ContentMetaKey> storageKey, bool alongWithIcon) NN_NOEXCEPT;
        Result GetFromStorage(size_t* outValue, void* buffer, size_t bufferSize, const ncm::ContentMetaKey& key, const ApplicationControlKey& appKey, uint8_t programIndex, bool alongWithIcon) NN_NOEXCEPT;
        Result GetLatestFromCache(size_t* outValue, void* buffer, size_t bufferSize, ncm::ApplicationId id) NN_NOEXCEPT;
        Result GetFromCache(size_t* outValue, void* buffer, size_t bufferSize, const ApplicationControlKey& appKey) NN_NOEXCEPT;

        NonRecursiveMutex m_ControDataBufferMutex;
        char m_ControlDataBuffer[512 * 1024];
        Bit64 m_CacheBufferBuffer[ApplicationControlDatabase::RequiredBufferSize / sizeof(Bit64)];

        ApplicationControlDatabaseAccessor m_Cache;
        ApplicationRecordDatabase* m_RecordDb;
        IntegratedContentManager* m_Integrated;
        RequestServer* m_RequestServer;
    };
}}}
