﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaKey.h>

namespace nn { namespace ns {

    class AsyncResult;

    enum class ApplicationDownloadTaskStatus : Bit8
    {
        NotRunning,       //!< 実行していません。
        Running,          //!< 実行中です。
        Downloaded,       //!< ダウンロード済みです。ダウンロードしたものを反映するためには再起動してください。
        Fatal,            //!< ダウンロードが失敗しました。CancelApplicationDownload を呼んでタスクをキャンセルし、再起動してください。
    };

    /**
    * @brief    ダウンロードタスクの状態を取得します。
    *
    * @detail   この関数は試遊台向けに提供されます。
    *           この関数の互換性は維持される必要があります。
    *
    * @return   処理の結果が返ります。
    */
    Result GetApplicationDownloadTaskStatus(ApplicationDownloadTaskStatus* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーション本体およびの最新のパッチのダウンロードを要求します。
    *           パッチのダウンロードは必ず直接更新によって行われます。
    *
    * @details  処理が成功すると、バックグラウンドでアプリケーションの更新が始まります。
    *           存在しない実体も含めてアプリケーションの本体およびパッチがダウンロードされます。
    *
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *
    *           この関数は試遊台向けに提供されます。
    *           この関数の互換性は維持される必要があります。
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultAlreadyDownloading                        指定されたアプリケーションが既にダウンロード中、またはパッチ間差分適用中です。
    * @retval   ResultOutOfMaxRunningTask                       同時に発行できるリクエスト数が最大に達しています。
    */
    Result DownloadApplication(ncm::ApplicationId applicationId, ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;

    /**
    * @brief    アプリケーション本体およびの最新のパッチのダウンロードを非同期処理で要求します。
    *           パッチのダウンロードは必ず直接更新によって行われます。
    *
    * @details  非同期処理をリクエストして AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功すると、バックグラウンドでアプリケーションの更新が始まります。
    *           存在しない実体も含めてアプリケーションの本体およびパッチがダウンロードされます。
    *
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
    *           - RequestUpdateApplication
    *           - RequestApplicationUpdateInfo
    *           - RequestDownloadAddOnContent
    *           - RequestDownloadApplicationControlData
    *           - RequestVerifyApplication
    *
    *           この関数は試遊台向けに提供されます。
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultAlreadyDownloading                        指定されたアプリケーションが既にダウンロード中、またはパッチ間差分適用中です。
    * @retval   ResultOutOfMaxRunningTask                       同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultAlreadyUpToDate                           （AsyncResult から返される Result）既に対象のコンテンツが最新です。
    * @retval   ResultSystemUpdateRequiredForContentDelivery    （AsyncResult から返される Result）コンテンツ配信に本体更新が必要です。
    * @retval   ResultCanceled                                  （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestDownloadApplication(AsyncResult* outValue, ncm::ApplicationId applicationId, ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;

    /**
    * @brief    最新の追加コンテンツのダウンロードを非同期処理で要求します。
    *
    * @details  非同期処理をリクエストして AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功すると、バックグラウンドでアプリケーションの更新が始まります。
    *
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
    *           - RequestUpdateApplication
    *           - RequestApplicationUpdateInfo
    *           - RequestDownloadApplication
    *           - RequestDownloadApplicationControlData
    *           - RequestVerifyApplication
    *
    *           この関数は試遊台向けに提供されます。
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationRecordNotFound                 指定されたアプリケーション記録が見つかりません。
    * @retval   ResultAlreadyDownloading                        指定されたアプリケーションが既にダウンロード中、またはパッチ間差分適用中です。
    * @retval   ResultOutOfMaxRunningTask                       同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultAlreadyUpToDate                           （AsyncResult から返される Result）既に対象のコンテンツが最新です。
    * @retval   ResultSystemUpdateRequiredForContentDelivery    （AsyncResult から返される Result）コンテンツ配信に本体更新が必要です。
    * @retval   ResultCanceled                                  （AsyncResult から返される Result）処理がキャンセルされました。
    */
    Result RequestDownloadAddOnContent(AsyncResult* outValue, ncm::ApplicationId id, ncm::ContentMetaKey* keyList, int count, ncm::StorageId storageId = ncm::StorageId::Any) NN_NOEXCEPT;

    /**
    * @brief    指定したコンテンツの実体を削除します。
    *
    * @details  コンテンツの実体を削除できます。
    *           実体を削除してもアプリケーション記録は削除されずに列挙されます。
    *           この関数は試遊台向けに提供されます。
    *
    * @return   処理の結果が返ります。メディアアクセスエラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    */
    Result DeleteApplicationContentEntity(const ncm::ContentMetaKey& key) NN_NOEXCEPT;

    // これ以降の API の呼び出しには ns:rid 権限が必要になります。
    namespace detail
    {
        const char PortNameForRetailInteractiveDisplay[] = "ns:rid";
    }

    /**
    * @brief        ユーザーアカウントを新規に作成します。
    *
    * @details      ユーザーのニックネームとプロフィール画像を指定してアカウントを作成します。
    *               プロフィール画像は下記の形式に従う必要があります。
    *               ただしこのメソッドはその正当性を評価しません。
    *               - 画像解像度は縦横各256px
    *               - 色空間はsRGB
    *               - ベースライン方式のJPEGフォーマット
    *               - YCbCr サンプリング比は 4:2:0, 4:2:2, 4:4:4 のいずれか
    *               - 大きさ (バイト数) は128KiB (131,072バイト) を上限とする
    *
    * @param[in]    name        ユーザーのニックネーム
    * @param[in]    image       適用するプロフィール画像の先頭を指すアドレス
    * @param[in]    imageSize   image から読み込み可能な大きさ (バイト数)
    *
    * @return       処理の結果が返ります。
    */
    Result CreateUserAccount(account::Nickname name, void* image, size_t imageSize) NN_NOEXCEPT;
}}
