﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/ncm/ncm_ContentMetaId.h>

namespace nn { namespace ns {

    /*
    * @brief    カード内に含められるアプリケーションの最大数
    */
    static const int MaxApplicationCountOnGameCard = 8;

    /*
    * @brief    ゲームカードアップデート検知のためのイベントを取得します。
    *
    * @details  当該イベントは、ゲームカードアップデートが必要な状態では常にシグナルされています。
    *           ゲームカードが抜去されるなど、ゲームカードアップデートが不要な状態になるとクリアされます。
    *           当該イベントをユーザ側でクリアしないでください。
    *
    */
    void GetGameCardUpdateDetectionEvent(os::SystemEvent* outValue) NN_NOEXCEPT;

    /*
    * @brief    ゲームカードのマウント処理に失敗したことを検知するためのイベントを取得します。
    *
    * @return   クリアモードが nn::os::EventClearMode_ManualClear のイベントが返ります。
    */
    void GetGameCardMountFailureEvent(os::SystemEvent* outValue) NN_NOEXCEPT;

    /*
    * @brief    ゲームカードが挿入されているかを取得します。
    *
    * @details  当該 API はゲームカードの挿入状態のみを考慮し、ゲームカードの利用可能状態は考慮しません。
    *           したがって、「挿入されているけど使えない」状態であっても True を返します。
    */
    bool IsGameCardInserted() NN_NOEXCEPT;

    /*
    * @brief    ゲームカードへの空アクセスを行います。
    *
    * @return   処理の結果が返ります。  Result は実際に処理を行っている他のモジュールの Result が返ってきます。
    *           Result が Failure の場合はゲームカードが想定外の状態になっています。
    *           エラーレポートに登録する際は、エラーコードに ResultEnsureGameCardAccessFailed を登録し、
    *           Backtrace Result に当該 API で取得した Result を登録してください。
    */
    Result EnsureGameCardAccess() NN_NOEXCEPT;

    /*
    * @brief    ゲームカードのマウント処理に失敗した時の Result を取得します。
    *
    * @details  一番最後に発生したマウント処理の失敗要因を取得します。
    *           Result は実際に処理を行っている他のモジュールの Result が返ってきます。
    *           もし、起動中に一度もマウントに失敗していない場合、 ResultSuccess が返ります。
    *           エラーレポートに登録する際は、エラーコードに ResultGameCardMountFailed を登録し、
    *           Backtrace Result に当該 API で取得した Result を登録してください。
    *
    * @return  ゲームカードマウント処理における失敗要因の Result
    */
    Result GetLastGameCardMountFailureResult() NN_NOEXCEPT;

    /*
    * @brief    ゲームカード内のアプリケーションの ID を列挙します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultGameCardNotInserted   ゲームカードが利用できない状態です。
    */
    Result ListApplicationIdOnGameCard(int* outCount, ncm::ApplicationId* outList, int numList) NN_NOEXCEPT;
}}  // namespace ns
