﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace ns {

struct ApplicationViewFlagTag {};
typedef util::BitFlagSet<32, ApplicationViewFlagTag> ApplicationViewFlag;

enum class ApplicationDownloadState : Bit8
{
    Runnable = 0,
    Suspended,
    NotEnoughSpace,
    Fatal,
    Finished,
    SystemUpdateRequired,  // コミットするためには、先に本体更新が必要
};

enum class ApplicationApplyDeltaState : Bit8
{
    Applying,         //!< 適用中です
    Suspended,        //!< 内部的な状態で、View 取得時には返ってきません
    NotEnoughSpace,   //!< 容量不足です
    Fatal,            //!<
    NoTask,           //!< 内部的な状態で、View 取得時には返ってきません
    WaitApply,        //!< 内部的な状態で、View 取得時には返ってきません
    Applied,          //!< 内部的な状態で、View 取得時には返ってきません
};

/**
* @brief    アプリケーションのダウンロード状況です。
*
* @details  remainingTime は以下の条件をすべて満たす時のみ有効な値が入っています。無効値は 0 です。
*           @li isRunning == true
*           @li ダウンロード必要容量の計算が完了している
*           @li ダウンロードが始まって一定時間が経過している
*/
struct ApplicationDownloadProgress
{
    int64_t                     downloaded;     //!< ダウンロードが完了したサイズです。プログレスバー表示に利用します。
    int64_t                     total;          //!< ダウンロードが必要なサイズです。プログレスバー表示に利用します。
    Result                      lastResult;     //!< ダウンロード状態が Suspended もしくは Fatal の時に有効な原因です。
    ApplicationDownloadState    state;          //!< ダウンロード状態です。
    bool                        isRunning;      //!< ダウンロード実行中に true になります。
    Bit8                        reserved[2];
    TimeSpanType                remainingTime;  //!< ダウンロードが完了するまでの予想残り時間です。
};

/**
 * @brief  パッチ間差分の適用状況です。
 */
struct ApplicationApplyDeltaProgress
{
    int64_t      applied;    //!< 適用済みのサイズです。プログレスバー表示に利用します。
    int64_t      total;      //!< 適用が必要な総サイズです。プログレスバー表示に利用します。
    Result       lastResult; //!< 適用状態が Suspended あるいは Fatal の時に有効な、Suspended/Fatal になった原因です。
    ApplicationApplyDeltaState state; //!< パッチの適用状態です。
    bool         isRunning;  //!< 適用処理実行中に true になります。
    Bit8         reserved[2];
    TimeSpanType remainingTime; //!< パッチ適用が完了するまでの予想残り時間です。
};

/**
* @brief    アプリケーションのアイコンを表示するためのアプリケーションビュー構造体です。
*/
struct ApplicationView
{
    ncm::ApplicationId          id;             //!< アプリケーションの ID です。
    uint32_t                    version;        //!< アプリケーションの新旧比較に使用できるバージョンです。リリースバージョンとは異なります。
    ApplicationViewFlag         flag;           //!< システムが使用するフラグです。
    ApplicationDownloadProgress progress;       //!< アプリケーションのダウンロード状況です。IsDownloading() 時に有効です。
    ApplicationApplyDeltaProgress applyProgress; //!< パッチ間差分の適用状況です。

    /**
    *   @brief    アプリケーションの記録が存在します。
    */
    bool HasRecord() const NN_NOEXCEPT;

    /**
    *   @brief    アプリケーション本体の記録が存在します。
    */
    bool HasMainRecord() const NN_NOEXCEPT;

    /**
    *   @brief    パッチの記録が存在します。
    */
    bool HasPatchRecord() const NN_NOEXCEPT;

    /**
    *   @brief    追加コンテンツの記録が存在します。
    */
    bool HasAddOnContentRecord() const NN_NOEXCEPT;

    /**
    * @brief    アプリケーション本体がインストールされた記録が存在します。
    */
    bool HasMainInstallRecord() const NN_NOEXCEPT;

    /**
    * @brief    アプリケーション（本体・パッチ・追加コンテンツ含む）をダウンロード中です。
    */
    bool IsDownloading() const NN_NOEXCEPT;

    /**
    * @brief    アプリケーションはゲームカードとして扱われます。
    */
    bool IsGameCard() const NN_NOEXCEPT;

    /**
    * @brief    ゲームカードの実体が存在します。
    */
    bool HasGameCardEntity() const NN_NOEXCEPT;

    /**
    * @brief    アプリケーションが起動可能です。
    */
    bool IsLaunchable() const NN_NOEXCEPT;

    /**
    * @brief    全ての実体が存在します。記録が存在しない場合も true を返します。
    */
    bool HasAllEntity() const NN_NOEXCEPT;

    /**
    * @brief    アプリケーション本体の実体が存在します。
    */
    bool HasMainEntity() const NN_NOEXCEPT;

    /**
    * @brief    全ての追加コンテンツの実体が存在します。追加コンテンツの記録が存在しない場合も true を返します。
    */
    bool HasAllAddOnContentEntity() const NN_NOEXCEPT;

    /**
    * @brief    追加コンテンツの実体が少なくとも一つは存在します。
    */
    bool HasAnyAddOnContentEntity() const NN_NOEXCEPT;

    /**
    * @brief    パッチの実体が存在します。
    */
    bool HasPatchEntity() const NN_NOEXCEPT;

    /**
    * @brief    アプリケーションが破損している可能性があります。起動もしくは実体の削除でフラグが無効になります。
    */
    bool MaybeCorrupted() const NN_NOEXCEPT;

    /**
    * @brief    コミット待ちです。
    */
    bool IsWaitingCommit() const NN_NOEXCEPT;

    /**
    * @brief    コミット待ちのアプリケーション本体があります。
    */
    bool IsWaitingApplicationCommit() const NN_NOEXCEPT;

    /**
    * @brief    コミット待ちの追加コンテンツがあります。
    */
    bool IsWaitingAocCommit() const NN_NOEXCEPT;

    /**
    * @brief    インストール待ちのパッチがあります。
    */
    bool IsWaitingPatchInstall() const NN_NOEXCEPT;

    /**
    * @brief    自動削除が無効化されています。
    */
    bool IsAutoDeleteDisabled() const NN_NOEXCEPT;

    /**
    * @brief    パッチ間差分適用中です。
    */
    bool IsApplyingDelta() const NN_NOEXCEPT;

    /**
    * @brief    追加コンテンツのクリーンナップが要求されています。
    */
    bool IsCleanupAddOnContentWithNoRightsRecommended() const NN_NOEXCEPT;

    /**
    * @brief    コミットするには本体更新が必要です。
    */
    bool IsSystemUpdateRequiredToCommit() const NN_NOEXCEPT;

    /**
    * @brief    プリインストールのアプリケーションです。
    */
    bool IsPreInstalledApplication() const NN_NOEXCEPT;
};

/**
* @brief    アプリケーションビューを取得します。
*
* @details  アプリケーション記録が存在するアプリケーションのアプリケーションビューを取得できます。
*
* @pre
*           - outList, idList には count 分以上の配列が確保されている。
*
* @post
*           - outList に取得したアプリケーションビューが書き込まれている。
*
* @retval   ResultApplicationRecordNotFound   指定されたアプリケーション記録が１つ以上見つかりません。
*/
Result GetApplicationView(ApplicationView outList[], const ncm::ApplicationId idList[], int count) NN_NOEXCEPT;

/**
* @brief    アプリケーションビューのダウンロードエラー文脈を取得します。
*
* @details  ApplicationDownloadState が Suspended もしくは Fatal の時に取得してください。
*
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
* @retval   ResultApplicationAlreadyDownloading 指定されたアプリケーションはダウンロード中です。
*/
Result GetApplicationViewDownloadErrorContext(err::ErrorContext* outValue, const ncm::ApplicationId id) NN_NOEXCEPT;

struct ApplicationDownloadProgressDeprecated
{
    int64_t                     downloaded;
    int64_t                     total;
    Result                      lastResult;
    ApplicationDownloadState    state;
    bool                        isRunning;
    Bit8                        reserved[2];
};

struct ApplicationApplyDeltaProgressDeprecated
{
    int64_t      applied;
    int64_t      total;
    Result       lastResult;
    ApplicationApplyDeltaState state;
    Bit8         reserved[3];
};

struct ApplicationViewDeprecated
{
    ncm::ApplicationId          id;
    uint32_t                    version;
    ApplicationViewFlag         flag;
    ApplicationDownloadProgressDeprecated progress;
    ApplicationApplyDeltaProgressDeprecated applyProgress;
};

}}  // namespace nn::ns
