﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_ApplicationDownloadApi.h>
#include <nn/ns/ns_ApplicationRecordApi.h>
#include <nn/ns/ns_ApplicationManagerInitializeApi.h>
#include <nn/ns/ns_ApplicationViewApi.h>
#include <nn/ns/ns_ApplicationEntityApi.h>
#include <nn/ns/ns_DocumentApi.h>
#include <nn/ns/ns_GameCardApi.h>
#include <nn/ns/ns_SdCardApi.h>

namespace nn { namespace ns {

/**
* @brief    アプリケーションが一つのストレージを占有するサイズ情報です。
*/
struct ApplicationOccupiedSizeEntity
{
    ncm::StorageId  storageId; //!< 対象となるストレージの ID です。
    Bit8            reserved[7];
    int64_t         appSize;   //!< アプリケーションがストレージを占有しているバイト数です。
    int64_t         patchSize; //!< パッチがストレージを占有しているバイト数です。
    int64_t         aocSize;   //!< 追加コンテンツがストレージを占有しているバイト数です。
};

/**
* @brief    アプリケーションがストレージを占有するサイズ情報です。
*/
struct ApplicationOccupiedSize
{
    static const int MaxStorageCount = 4;
    ApplicationOccupiedSizeEntity storage[MaxStorageCount];
};

/**
* @brief    アプリケーションの占有サイズを計算します。
*
* @details  アプリケーション記録が存在するアプリケーション実体の占有サイズを計算できます。
*           本体が認識している全てのストレージから実体を探索して占有サイズを計算します。
*           ストレージ毎の占有サイズの情報は ApplicationOccupiedSize.storage の先頭から埋められていきます。
*           ApplicationOccupiedSizeEntity.storageId が None の要素がある場合には、
*           それ以降の配列の要素に有効な値は入っていません。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound     指定されたアプリケーション記録が見つかりません。
*
* @post
*           - outValue にアプリケーションの占有サイズが書き込まれている
*/
Result CalculateApplicationOccupiedSize(ApplicationOccupiedSize* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定されたアプリケーションの起動に対して、本体更新もしくはアプリケーション更新の必要性を調べます。
*
* @details  この API はシステム更新とアプリケーション更新の必要性を判断し、その結果を返します。
*
* @return   処理の結果が返ります。
* @retval   ResultSystemUpdateRequired              アプリケーションを起動するためにはシステム更新が必要です。
* @retval   ResultApplicationUpdateRequired         アプリケーションを起動するためには最新の更新データの適用が必要です。
* @retval   ResultApplicationUpdateRequiredByRequiredVersion                ResultApplicationUpdateRequired に含まれる Result です。必須バージョンが満たされていません。
* @retval   ResultApplicationUpdateRequiredByRequiredApplicationVersion     ResultApplicationUpdateRequired に含まれる Result です。 AddOnContent の必須アプリケーションリリースバージョンが満たされていません。
* @retval   ResultApplicationUpdateRecommended      アプリケーションは起動可能ですが最新の更新データの適用が推奨されます。
* @retval   ResultApplicationRecordNotFound         アプリケーション記録が存在しません。
* @retval   ResultMainApplicationNotFound           指定したアプリケーションが存在しません。
*
* @pre
*           - InitializeForApplicationManager が呼び出されている
*
*/
Result CheckApplicationLaunchVersion(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定されたアプリケーションを起動する権利があるかを調べます。
*
* @details
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationLaunchRightsNotFound   アプリケーションを起動するための権利がありません。
* @retval   ResultInactiveNintendoAccount           アプリケーションを起動する権利を持っている NA が無効状態です。
* @retval   ResultApplicationRecordNotFound         アプリケーション記録が見つかりません。
* @retval   ResultApplicationContentNotFound        アプリケーションの実体が見つかりません。
* @retval   ResultApplicationPrepurchased           アプリケーションが予約購入状態です。
*
*
* @pre
*           - InitializeForApplicationManager が呼び出されている
*
*/
Result CheckApplicationLaunchRights(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    指定されたアプリケーションを再開する権利があるかを調べます。
*
* @details
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationLaunchRightsNotFound   アプリケーションを再開するための権利がありません。
* @retval   ResultInactiveNintendoAccount           アプリケーションを再開する権利を持っている NA が無効状態です。
* @retval   ResultApplicationRecordNotFound         アプリケーション記録が見つかりません。
* @retval   ResultApplicationContentNotFound        アプリケーションの実体が見つかりません。
*
*
* @pre
*           - InitializeForApplicationManager が呼び出されている
*
*/
Result CheckApplicationResumeRights(ncm::ApplicationId id) NN_NOEXCEPT;

/**
* @brief    ロゴデータを取得します。
*
* @details  ロゴデータパーティションから、logoPath で指定したファイルの内容を読み取ります。
*
* @return   処理の結果が返ります。
* @retval   ResultApplicationRecordNotFound         アプリケーション記録が存在しません。
* @retval   ResultBufferNotEnough                   与えられたバッファがロゴデータより小さいです。
* @retval   ResultLogoPartitionNotFound             ロゴデータのパーティションが存在しません。
* @retval   ResultLogoPathNotFound                  logoPath で指定したファイルが存在しません。
* @retval   ResultApplicationLogoCorrupted          ロゴデータが破損しています。
*
* @pre
*           - InitializeForApplicationManager が呼び出されている
*
*/
Result GetApplicationLogoData(size_t* outSize, void* buffer, size_t bufferSize, ncm::ApplicationId id, const char* logoPath) NN_NOEXCEPT;

/**
* @brief    対象ストレージの全体容量を取得します。
*
* @return   処理の結果が返ります。
* @retval   ResultUnsupportedStorage    未対応のストレージが指定されました。
* @retval   ResultStorageAccessFailed   指定されたストレージにアクセス出来ませんでした。
*/
Result GetTotalSpaceSize(int64_t* pOutSize, ncm::StorageId storage) NN_NOEXCEPT;

/**
* @brief    対象ストレージの空き容量を取得します。
*
* @return   処理の結果が返ります。
* @retval   ResultUnsupportedStorage    未対応のストレージが指定されました。
* @retval   ResultStorageAccessFailed   指定されたストレージにアクセス出来ませんでした。
*/
Result GetFreeSpaceSize(int64_t* pOutSize, ncm::StorageId storage) NN_NOEXCEPT;

}}  // namespace nn::ns
