﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ns/ns_Async.h>
#include <nn/settings/settings_Language.h>

namespace nn { namespace ns {
    /**
    * @brief    アプリケーションの言語ごとのタイトルです。
    */
    struct ApplicationTitle
    {
        char    name[512];          //!< タイトル名です。
        char    publisher[256];     //!< パブリッシャ名です。
    };

    /**
    * @brief    アプリケーションが起動時に要求するアカウントです。
    */
    enum class StartupUserAccount : Bit8
    {
        None,                                       //!< アカウントを要求しません。
        Required,                                   //!< ローカルアカウントにログインしている必要があります。
        RequiredWithNetworkServiceAccountAvailable  //!< ニンテンドーアカウントにログインしている必要があります。
    };

    enum class AttributeFlag : Bit32
    {
        Demo                        = 0x1u << 0,
        RetailInteractiveDisplay    = 0x1u << 1
    };

    /**
    * @brief    アプリケーションが使用するペアレンタルコントロール設定対象の項目です。
    */
    enum class ParentalControlFlag : Bit32
    {
        FreeCommunication = 0x1u,     //!< FreeCommunication を使用します。
    };

    /**
    * @brief    アプリケーション起動中のスクリーンショット撮影の可否フラグです。
    */
    enum class Screenshot : Bit8
    {
        Allow,      //!< スクリーンショットを撮影することができます。
        Deny        //!< スクリーンショットを撮影することはできません。
    };

    /**
    * @brief    アプリケーション起動中の動画撮影の可否フラグです。
    */
    enum class VideoCapture : Bit8
    {
        Disable = 0x0u, //!< 動画を撮影することはできません。
        Manual  = 0x1u, //!< 動画を撮影することができます。アプリからメモリを渡します。
        Enable  = 0x2u, //!< 動画を撮影することができます。アプリのメモリが自動的に使用されます。
    };

    /**
    * @brief   アプリケーションのロゴデータに利用するライセンス種別です。
    */
    enum class LogoType : Bit8
    {
        LicensedByNintendo,     //!< 一般のアプリケーション向けのライセンスです。
        ObsoletedLogoType,
        Nintendo,               //!< 任天堂タイトルのライセンスです。
    };

    /**
    * @brief    複数アプリケーション間でフレンドプレゼンスを共有するために使用されるID です。
    */
    struct PresenceGroupId
    {
        Bit64 value;
    };

    /**
    * @brief    レーティング団体です。
    */
    enum class RatingOrganization : Bit8
    {
        CERO = 0,
        GRACGCRB,
        GSRMR,
        ESRB,
        ClassInd,
        USK,
        PEGI,
        PEGIPortugal,
        PEGIBBFC,
        Russian,
        ACB,
        OFLC
    };

    /**
    * @brief    システムバージョンの表記です。
    */
    struct SystemVersionSignage
    {
        char signage[32];
    };

    /**
    * @brief    ソフト終了時のデータ消失確認の有無です。
    */
    enum class DataLossConfirmation : Bit8
    {
        None,        //!< データ消失確認を行いません。
        Required,    //!< データ消失確認を行います。
    };

    /**
    * @brief    アプリケーションの起動に関するログの記録方法です。
    */
    enum class PlayLogPolicy : uint8_t
    {
        All = 0, //!< 起動・終了の記録を行い、統計情報の作成対象にします。主に通常のアプリケーション向けの設定です。
        LogOnly = 1, //!< 起動・終了の記録を行いますが、統計情報の作成対象から除外します。主にシステムアプレットやライブラリアプレット向けの設定です。
        None = 2, //!< 起動・終了の記録自体を行いません。主にデバッグツール、修理ツール向けの設定です。
    };

    /**
    * @brief    アプリケーション独自のエラーコードのカテゴリです。
    */
    struct ApplicationErrorCodeCategory
    {
        char value[8];
    };

    /**
    * @brief    アプリケーションのロゴ表示管理方法です。
    */
    enum class LogoHandling : uint8_t
    {
        Auto = 0,           //!< 自動でロゴ表示の管理を行います。
        Manual = 1,         //!< アプリケーションがロゴ表示の管理を行います。
    };

    /**
    * @brief    追加コンテンツ登録の方法です。
    */
    enum class AddOnContentRegistrationType : uint8_t
    {
        AllOnLaunch = 0,    //!< アプリケーション起動時に全て登録します。(0.12 系でビルドしたものはこの値)
        OnDemand = 1,       //!< マウント時に登録します。(1.1.0 以降)
    };

    /**
    * @brief    アプリケーションの HDCP の使用有無です。
    */
    enum class Hdcp : uint8_t
    {
        None = 0, //!< HDCP を利用しません。
        Required = 1, //!< HDCP を利用します。
    };

    /**
    * @brief    クラッシュレポートの保存可否フラグです。
    */
    enum class CrashReport : uint8_t
    {
        Deny = 0,  //!< 保存を許可しません。
        Allow = 1, //!< 保存を許可します。
    };

    /**
    * @brief    アプリ実行中の、追加コンテンツのコミット制御方法です。
    */
    enum class RuntimeAddOnContentInstall : uint8_t
    {
        Deny = 0,        //!< コミットは行いません。
        AllowAppend = 1, //!< 新規追加のみ行います。
    };

    /**
    * @brief    アプリのプレイ履歴の取得権限です。
    */
    enum class PlayLogQueryCapability : uint8_t
    {
        None      = 0, //!< プレイ履歴を取得できません。
        WhiteList = 1, //!< PlayLogQueryableApplicationId に指定したアプリのプレイ履歴を取得できます。
        All       = 2, //!< 全てのアプリのプレイ履歴を取得できます。
    };

    /**
    * @brief    PlayLogQueryableApplicationId に指定できるアプリの最大数です。
    */
    const int PlayLogQueryableApplicationCountMax = 16;

    /**
    * @brief    修理ツールの動作を指定するフラグです。
    */
    enum class RepairFlag : Bit8
    {
        SuppressGameCardAccess = 0x1, //!< ゲームカードへのアクセスを抑制します。
    };

    /**
    * @brief    アプリケーション起動時に必要とするネットワークサービスのライセンスを指定するフラグです。
    */
    enum class RequiredNetworkServiceLicenseOnLaunchFlag : Bit8
    {
        None        = 0,        //!<    必要なライセンスはありません
        Common      = 1 << 0,   //!<    一般的なライセンスを必要とします。
    };

    /**
    * @brief    アプリケーション管理データです。
    */
    struct ApplicationControlProperty
    {
        static const int MaxLanguageCount = 16;

        ApplicationTitle                title[MaxLanguageCount];        //!< 言語ごとのタイトルです。
        char                            isbn[37];                       //!< ISBN です。
        StartupUserAccount              startupUserAccount;             //!< アプリケーションが起動時に要求するアカウントです。
        Bit8                            reserved_0x3026;
        AddOnContentRegistrationType    addOnContentRegistrationType;   //!< 追加コンテンツの登録方法です。
        Bit32                           attributeFlag;
        Bit32                           supportedLanguageFlag;          //!< アプリケーションが対応している言語のフラグです。
        Bit32                           parentalControlFlag;            //!< アプリケーションが使用しているペアレンタルコントロール設定対象項目のフラグです。
        Screenshot                      screenShot;                     //!< アプリケーション起動中のスクリーンショット撮影の可否フラグです。
        VideoCapture                    videoCapture;                   //!< アプリケーション起動中の動画撮影の可否フラグです。
        DataLossConfirmation            dataLossConfirmation;           //!< オートセーブ対応フラグです。
        PlayLogPolicy                   playLogPolicy;                  //!< アプリケーションの起動に関するログの記録方法です。
        PresenceGroupId                 presenceGroupId;                //!< 複数アプリケーション間でフレンドプレゼンスを共有するために使用されるID です。
        int8_t                          ratingAge[32];                  //!< レーティング団体に設定されている年齢です。
        char                            displayVersion[16];             //!< ユーザへの表示用に使用するアプリケーションのバージョン表記です。
        Bit64                           addOnContentBaseId;             //!< アプリケーションの追加コンテンツベースIDです。
        Bit64                           saveDataOwnerId;                //!< アプリケーションのセーブデータのアクセス権所有者を示す ID です。
        int64_t                         userAccountSaveDataSize;        //!< アプリケーションのユーザーアカウントセーブデータのサイズです。
        int64_t                         userAccountSaveDataJournalSize; //!< アプリケーションのユーザーアカウントセーブデータのジャーナル領域サイズです。
        int64_t                         deviceSaveDataSize;             //!< アプリケーションの本体セーブデータのサイズです。
        int64_t                         deviceSaveDataJournalSize;      //!< アプリケーションの本体セーブデータのジャーナル領域サイズです。
        int64_t                         bcatDeliveryCacheStorageSize;   //!< アプリケーションのデータ配信キャッシュストレージのサイズです。
        ApplicationErrorCodeCategory    applicationErrorCodeCategory;   //!< アプリケーション独自のエラーコードのカテゴリです。
        Bit64                           localCommunicationId[8];        //!< ローカル通信識別子です。
        LogoType                        logoType;                       //!< アプリケーションのロゴデータに使われるライセンス種別です。
        LogoHandling                    logoHandling;                   //!< アプリケーションのロゴ表示管理方法の種別です。
        RuntimeAddOnContentInstall      runtimeAddOnContentInstall;     //!< アプリ実行中の、追加コンテンツコミット制御方法です。
        Bit8                            reserved_0x30F3[3];
        CrashReport                     crashReport;                    //!< クラッシュレポートの保存可否フラグです。
        Hdcp                            hdcp;                           //!< HDCP の利用状態の識別子です。
        Bit64                           seedForPseudoDeviceId;          //!< 疑似本体 ID 生成用のシードです。
        char                            bcatPassphrase[65];             //!< データ配信用のパスフレーズです。
        Bit8                            reserved_0x3141;
        Bit8                            reservedForUserAccountSaveDataOperation[6];
        int64_t                         userAccountSaveDataSizeMax;                         //!< アプリケーションのユーザーアカウントセーブデータのサイズの拡張上限値です。
        int64_t                         userAccountSaveDataJournalSizeMax;                  //!< アプリケーションのユーザーアカウントセーブデータのジャーナル領域サイズの拡張上限値です。
        int64_t                         deviceSaveDataSizeMax;          //!< アプリケーションの本体セーブデータのサイズの拡張上限値です。
        int64_t                         deviceSaveDataJournalSizeMax;   //!< アプリケーションの本体セーブデータのジャーナル領域サイズの拡張上限値です。
        int64_t                         temporaryStorageSize;           //!< 一時ストレージのサイズです。
        int64_t                         cacheStorageSize;               //!< キャッシュストレージのサイズです。
        int64_t                         cacheStorageJournalSize;        //!< キャッシュストレージのジャーナルサイズです。
        int64_t                         cacheStorageDataAndJournalSizeMax;                  //!< アプリがキャッシュストレージを作成する場合の最大サイズです。
        int16_t                         cacheStorageIndexMax;           //!< アプリがキャッシュストレージを複数作成する場合のインデックスの最大値です。
        Bit8                            reserved_0x318a[6];
        ncm::ApplicationId              playLogQueryableApplicationId[PlayLogQueryableApplicationCountMax]; //!< プレイ履歴を取得できるアプリケーションのリストです。
        PlayLogQueryCapability          playLogQueryCapability;         //!< プレイ履歴の取得権限です。
        Bit8                            repairFlag;                     //!< 修理ツールの動作を指定するフラグです。
        uint8_t                         programIndex;                   //!< プログラムのインデックス
        Bit8                            requiredNetworkServiceLicenseOnLaunchFlag;          //!< アプリケーション起動時に必要とするネットワークサービスのライセンスフラグです。
        Bit8                            reserved2[3564];

        /**
        * @brief    各言語に設定されているタイトル情報を取得します。
        *
        * @return   タイトル情報が返ります。未設定の場合は空文字列が返ります。
        */
        const ApplicationTitle& GetTitle(const settings::LanguageCode& language) const NN_NOEXCEPT;

        /**
        * @brief    本体言語設定を参照してタイトル情報を取得します。
        *
        *           本体言語設定にマッチするタイトル情報を取得します。
        *           見つからなかった場合、言語優先順位に従って探索し、見つかったものを取得します。
        *
        * @return   タイトル情報が返ります。タイトル情報が１つも見つからない場合は空文字列が返ります。
        */
        const ApplicationTitle& GetDefaultTitle() const NN_NOEXCEPT;

        /**
        * @brief    指定された言語の対応可否を判定します。
        */
        bool SupportsLanguage(const settings::LanguageCode& language) const NN_NOEXCEPT;

        /**
        * @brief    各レーティング団体に設定されている年齢を取得します。
        *
        * @return   年齢が返ります。未設定の場合は負値が返ります。
        */
        int8_t GetAge(RatingOrganization organization) const NN_NOEXCEPT
        {
            return ratingAge[static_cast<int>(organization)];
        }
    };

    NN_STATIC_ASSERT(sizeof(ApplicationControlProperty) == 16384);

    /**
    * @brief    アプリケーション管理データの探索先です。
    */
    enum class ApplicationControlSource : Bit8
    {
        CacheOnly,                //!< キャッシュのみ探索する
        Storage,                  //!< キャッシュにない場合、ストレージにあるコンテンツを探索する
        StorageOnly,              //!< ストレージのみ探索する
    };

    /**
    * @brief    アプリケーション管理データを扱うためのクラスです。
    */
    class ApplicationControlDataAccessor
    {
    public:
        explicit ApplicationControlDataAccessor(void* data, size_t size) NN_NOEXCEPT : m_Data(data), m_Size(size) {}

        /**
        * @brief    アプリケーション管理データを取得します。
        */
        const ApplicationControlProperty& GetProperty() const NN_NOEXCEPT;

        /**
        * @brief    アプリケーションのアイコンデータを取得します。
        */
        const void* GetIconData() const NN_NOEXCEPT;

        /**
        * @brief    アプリケーションのアイコンサイズを取得します。
        */
        size_t GetIconSize() const NN_NOEXCEPT;

    private:
        void* m_Data;
        size_t m_Size;
    };

    /**
    * @brief    アプリケーション管理データをアプリケーション管理データベースから取得します。
    *
    * @details  指定したアプリケーション ID のアプリケーション記録にアプリケーション本体が存在する場合は
    *           記録にある最新のバージョンの取得を試みます。アプリケーション記録もしくは本体が存在しない場合は
    *           キャッシュの中で最も新しいものの取得を試みます。
    *
    *           CacheOnly 以外のソースが指定された場合は、キャッシュに見つからない場合に当該ソースから
    *           アプリケーション管理データの取得を試みます。取得できた場合、キャッシュに書き込みつつ
    *           取得したアプリケーション管理データを返します。キャッシュには最大保持数が定義されており LRU で吐き出されます。
    *           キャッシュの最大保持数は GetMaxApplicationControlDataCacheCount() で取得できます。
    *
    *           取得したアプリケーション管理データから属性およびアイコンを読み出すには ApplicationControlDataAccessor を利用してください。
    *
    * @return   処理の結果が返ります。メディアアクセスエラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultApplicationControlDataNotFound    指定されたソースにデータアイコンが見つかりません。
    * @retval   ResultBufferNotEnough                   与えられたバッファがアプリケーション管理データより小さいです。
    *
    * @pre
    *           - InitializeForApplicationManager が呼び出されている
    *
    * @post
    *           - outValue にアプリケーション管理属性が書き込まれている
    */
    Result GetApplicationControlData(size_t* outValue, void* buffer, size_t bufferSize, ApplicationControlSource source, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーション管理データの最大キャッシュ保持数を返します。
    *
    * @pre
    *           - InitializeForApplicationManager が呼び出されている
    */
    int GetMaxApplicationControlCacheCount() NN_NOEXCEPT;

    /**
    * @brief    最新のアプリケーション管理データをダウンロードして、アプリケーション管理データベースに登録します。
    *
    * @details  非同期処理をリクエストして、AsyncResult を返します。
    *           処理の結果は AsyncResult から取得してください。
    *           処理が成功するためには、インフラ通信が確立している必要があります。
    *           この関数で同時にリクエストできる非同期処理は最大１つです。
    *           また、下記の関数と最大非同期処理リクエスト可能数を共有しています。
    *           - RequestApplicationUpdateInfo
    *           - RequestUpdateApplication
    *           - RequestUpdateApplicationPatchOnly
    *           - RequestVerifyApplication
    *           - RequestVerifyAddOnContentsRights
    *
    * @return   処理の結果が返ります。通信エラーなどは下位レイヤの Result が返されます。必要があればエラーコードを表示してください。
    * @retval   ResultOutOfMaxRunningTask               同時に発行できるリクエスト数が最大に達しています。
    * @retval   ResultCanceled                          （AsyncResult から返される Result）処理がキャンセルされました。
    * @retval   ResultApplicationControlNotDelivered    （AsyncResult から返される Result）管理データが配信されていません。
    */
    Result RequestDownloadApplicationControlData(AsyncResult* outValue, ncm::ApplicationId id) NN_NOEXCEPT;

    /**
    * @brief    アプリケーション管理データのキャッシュを全て無効化します。
    *
    * @pre
    *           - InitializeForApplicationManager が呼び出されている
    */
    void InvalidateAllApplicationControlCache() NN_NOEXCEPT;

    /**
    * @brief    指定したアプリケーションのアプリケーション管理データのキャッシュを無効化します。
    *
    * @pre
    *           - InitializeForApplicationManager が呼び出されている
    */
    void InvalidateApplicationControlCache(ncm::ApplicationId id) NN_NOEXCEPT;

}}  // namespace ns
