﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMetaType.h>
#include <nn/ncm/ncm_StorageId.h>

namespace nn { namespace ns {

    enum class ContentMetaRightsCheck : Bit8
    {
        NotChecked,         //!< 権利がチェックされていません。
        NotNeeded,          //!< 権利が不要なコンテンツです。
        CommonRights,       //!< 共通権利を所有しています。
        PersonalizedRights, //!< 固有権利を所有しています。
        NoRights,           //!< 権利を所有していません。
    };

    /**
    * @brief    アプリケーションのコンテンツメタ状態です。
    */
    struct ApplicationContentMetaStatus
    {
        ncm::ContentMetaType type;          //!< コンテンツメタのタイプです。
        ncm::StorageId installedStorage;    //!< インストールされているストレージです。StorageId::None ならインストールされていません。
        ContentMetaRightsCheck rightsCheck; //!< コンテンツメタの権利状態です。
        Bit8 reserved;
        uint32_t version;                   //!< コンテンツメタのバージョンです。
        Bit64 id;                           //!< コンテンツメタの ID です。
    };

    /**
    * @brief    指定したアプリケーション記録のコンテンツメタ数を取得します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound   指定されたアプリケーション記録が見つかりません。
    */
    Result CountApplicationContentMeta(int* outValue, ncm::ApplicationId appId) NN_NOEXCEPT;

    /**
    * @brief    指定したアプリケーション記録のコンテンツメタ状態を取得します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound   指定されたアプリケーション記録が見つかりません。
    */
    Result ListApplicationContentMetaStatus(int* outCount, ApplicationContentMetaStatus outList[], int count, ncm::ApplicationId appId, int offset) NN_NOEXCEPT;

    /**
    * @brief    指定したアプリケーション記録のコンテンツメタ状態を権利チェック付きで取得します。
    *
    * @return   処理の結果が返ります。
    * @retval   ResultApplicationRecordNotFound   指定されたアプリケーション記録が見つかりません。
    */
    Result ListApplicationContentMetaStatusWithRightsCheck(int* outCount, ApplicationContentMetaStatus outList[], int count, ncm::ApplicationId appId, int offset) NN_NOEXCEPT;

}}  // namespace nn::ns
