﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_ApplyDeltaTaskId.h>
#include <nn/nim/nim_LocalCommunicationReceiveApplicationTaskInfo.h>
#include <nn/nim/nim_LocalCommunicationSendApplicationTaskId.h>
#include <nn/nim/nim_LocalCommunicationReceiveSystemUpdateTaskId.h>
#include <nn/nim/nim_LocalCommunicationReceiveSystemUpdateTaskInfo.h>
#include <nn/nim/nim_LocalCommunicationSendSystemUpdateTaskId.h>
#include <nn/nim/nim_LocalCommunicationSendSystemUpdateTaskInfo.h>
#include <nn/nim/nim_NetworkInstallTaskInfo.h>
#include <nn/nim/nim_SystemUpdateTaskInfo.h>
#include <nn/nim/srv/nim_ApplyDownloadedDeltaTask.h>
#include <nn/nim/srv/nim_DeviceAccountStore.h>
#include <nn/nim/srv/nim_LocalCommunicationReceiveApplicationTask.h>
#include <nn/nim/srv/nim_LocalCommunicationSendApplicationTask.h>
#include <nn/nim/srv/nim_LocalCommunicationReceiveSystemUpdateTask.h>
#include <nn/nim/srv/nim_LocalCommunicationSendSystemUpdateTask.h>
#include <nn/nim/srv/nim_NetworkSystemUpdateTask.h>
#include <nn/nim/srv/nim_NetworkInstallTask.h>
#include <nn/nim/srv/nim_OsUtil.h>

namespace nn { namespace nim { namespace srv {

    template<class TaskT, class IdT>
    class TaskHolderBase
    {
    public:
        typedef IdT IdType;

        TaskHolderBase() NN_NOEXCEPT : m_IsRunning() {}

        NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
        {
            return m_Task;
        }

        TaskT& GetTask() NN_NOEXCEPT
        {
            return *m_Task;
        }

        IdT GetId() const NN_NOEXCEPT
        {
            return m_Id;
        }

        void Finalize() NN_NOEXCEPT
        {
            m_Task->Cleanup();
            m_Task = util::nullopt;
        }

        void PrepareShutdown() NN_NOEXCEPT
        {
            m_Task = util::nullopt;
        }

        bool IsRunning() NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);
            return m_IsRunning;
        }

        void SetRunning(bool isRunning) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);
            m_IsRunning = isRunning;
        }

        void ResetLastResult() NN_NOEXCEPT
        {
            m_Task->ResetLastResult();
        }

    protected:
        util::optional<TaskT> m_Task;
        IdT m_Id;
        bool m_IsRunning;
        NonRecursiveMutex m_Mutex;
    };

    class SystemUpdateTaskHolder : public TaskHolderBase<NetworkSystemUpdateTask, SystemUpdateTaskId>
    {
    public:
        typedef TaskHolderBase<NetworkSystemUpdateTask, SystemUpdateTaskId>::IdType IdType;
        typedef SystemUpdateTaskInfo InfoType;

        Result Initialize(DeviceContext* deviceContext, const util::Uuid& uuid, const ncm::ContentMetaKey& key, Bit32 config, const char* dataFilePath) NN_NOEXCEPT;

        SystemUpdateTaskInfo GetInfo() NN_NOEXCEPT;
    };

    class NetworkInstallTaskHolder : public TaskHolderBase<NetworkInstallTask, NetworkInstallTaskId>
    {
    public:
        typedef TaskHolderBase<NetworkInstallTask, NetworkInstallTaskId>::IdType IdType;
        typedef NetworkInstallTaskInfo InfoType;

        Result Initialize(DeviceContext* deviceContext, DeviceAccountStore* deviceAccountStore, const util::Uuid& uuid, const char* metaFilePath, const char* dataFilePath, ovln::SenderForOverlayType* sender) NN_NOEXCEPT;

        NetworkInstallTaskInfo GetInfo() NN_NOEXCEPT;

        Result SetAttribute(Bit64 attribute) NN_NOEXCEPT;
    };

    class ApplyDeltaTaskHolder : public TaskHolderBase<ApplyDownloadedDeltaTask, ApplyDeltaTaskId>
    {
    public:
        typedef TaskHolderBase<ApplyDownloadedDeltaTask, ApplyDeltaTaskId>::IdType IdType;
        typedef ApplyDeltaTaskInfo InfoType;

        Result Initialize(const util::Uuid& uuid, const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT;

        ApplyDeltaTaskInfo GetInfo() NN_NOEXCEPT;
    };

    class LocalCommunicationReceiveApplicationTaskHolder : public TaskHolderBase<LocalCommunicationReceiveApplicationTask, LocalCommunicationReceiveApplicationTaskId>
    {
    public:
        typedef TaskHolderBase<LocalCommunicationReceiveApplicationTask, LocalCommunicationReceiveApplicationTaskId>::IdType IdType;
        typedef LocalCommunicationReceiveApplicationTaskInfo InfoType;

        Result Initialize(const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT;
        Result InitializeForCleanup(const util::Uuid& uuid, const char* metaFilePath, const char* dataFilePath) NN_NOEXCEPT;

        LocalCommunicationReceiveApplicationTaskInfo GetInfo() NN_NOEXCEPT;
    };

    class LocalCommunicationSendApplicationTaskHolder : public TaskHolderBase<LocalCommunicationSendApplicationTask, LocalCommunicationSendApplicationTaskId>
    {
    public:
        typedef TaskHolderBase<LocalCommunicationSendApplicationTask, LocalCommunicationSendApplicationTaskId>::IdType IdType;
        typedef LocalCommunicationSendApplicationTaskInfo InfoType;

        Result Initialize(const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const ncm::StorageContentMetaKey keyList[], int listCount, ncm::ApplicationId id) NN_NOEXCEPT;

        LocalCommunicationSendApplicationTaskInfo GetInfo() NN_NOEXCEPT;
    };

    class LocalCommunicationReceiveSystemUpdateTaskHolder : public TaskHolderBase<LocalCommunicationReceiveSystemUpdateTask, LocalCommunicationReceiveSystemUpdateTaskId>
    {
    public:
        typedef TaskHolderBase<LocalCommunicationReceiveSystemUpdateTask, LocalCommunicationReceiveSystemUpdateTaskId>::IdType IdType;
        typedef LocalCommunicationReceiveSystemUpdateTaskInfo InfoType;

        Result Initialize(const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key, Bit32 config, const char* dataFilePath) NN_NOEXCEPT;
        Result InitializeForCleanup(const util::Uuid& uuid, const ncm::ContentMetaKey& key, Bit32 config, const char* dataFilePath) NN_NOEXCEPT;

        LocalCommunicationReceiveSystemUpdateTaskInfo GetInfo() NN_NOEXCEPT;
    };

    class LocalCommunicationSendSystemUpdateTaskHolder : public TaskHolderBase<LocalCommunicationSendSystemUpdateTask, LocalCommunicationSendSystemUpdateTaskId>
    {
    public:
        typedef TaskHolderBase<LocalCommunicationSendSystemUpdateTask, LocalCommunicationSendSystemUpdateTaskId>::IdType IdType;
        typedef LocalCommunicationSendSystemUpdateTaskInfo InfoType;

        Result Initialize(const util::Uuid& uuid, uint32_t ipv4, uint16_t port, const ncm::ContentMetaKey& key) NN_NOEXCEPT;

        LocalCommunicationSendSystemUpdateTaskInfo GetInfo() NN_NOEXCEPT;
    };

    template<class TaskHolderT, int HolderCount>
    class TaskHolderManager
    {
    public:
        typedef TaskHolderT TaskHolderType;
        typedef typename TaskHolderT::IdType IdType;
        typedef typename TaskHolderT::InfoType InfoType;

        Result Destroy(IdType id) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            TaskHolderT* holder;
            NN_RESULT_DO(FindHolder(&holder, id));
            NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

            holder->Finalize();

            NN_RESULT_SUCCESS;
        }

        int Count() NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            int count = 0;
            for (auto& holder : m_HolderList)
            {
                if (holder)
                {
                    count++;
                }
            }

            return count;
        }

        int List(IdType outList[], int count) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            int readCount = 0;
            for (auto& holder : m_HolderList)
            {
                if (readCount >= count)
                {
                    break;
                }

                if (holder)
                {
                    outList[readCount] = holder.GetId();
                    readCount++;
                }
            }

            return readCount;
        }

        IdType GetId(int index) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            int count = 0;
            for (auto& holder : m_HolderList)
            {
                if (holder)
                {
                    if (count == index)
                    {
                        return holder.GetId();
                    }
                    count++;
                }
            }

            NN_ABORT("must not come here");
        }

        Result GetInfo(InfoType* outValue, IdType id) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            TaskHolderT* holder;
            NN_RESULT_DO(FindHolder(&holder, id));

            *outValue = holder->GetInfo();

            NN_RESULT_SUCCESS;
        }

        Result Commit(IdType id) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            TaskHolderT* holder;
            NN_RESULT_DO(FindHolder(&holder, id));
            NN_RESULT_THROW_UNLESS(!holder->IsRunning(), ResultTaskStillRunning());

            NN_RESULT_DO(holder->GetTask().Commit());

            NN_RESULT_SUCCESS;
        }

        Result FindVacantHolder(TaskHolderT** outValue) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            for (auto& holder : m_HolderList)
            {
                if (!holder)
                {
                    *outValue = &holder;
                    NN_RESULT_SUCCESS;
                }
            }

            NN_RESULT_THROW(ResultOutOfMaxTask());
        }

        Result FindHolder(TaskHolderT** outValue, IdType id) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            for (auto& holder : m_HolderList)
            {
                if (holder && holder.GetId() == id)
                {
                    *outValue = &holder;
                    NN_RESULT_SUCCESS;
                }
            }

            NN_RESULT_THROW(ResultTaskNotFound());
        }

        void PrepareShutdown() NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> guard(m_Mutex);

            for (auto& holder : m_HolderList)
            {
                if (holder)
                {
                    holder.PrepareShutdown();
                }
            }
        }

    private:
        RecursiveMutex m_Mutex;
        TaskHolderT m_HolderList[HolderCount];
    };
}}}
