﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <cstring>

namespace nn
{
namespace nifm
{

/**
 * @brief 無線 LAN 認証方式を表します。
 */
enum Authentication
{
    Authentication_Invalid, //!<  無効
    Authentication_Open,    //!<  Open 802.11 認証
    Authentication_Shared,  //!<  Shared 802.11 認証
    Authentication_Wpa,     //!<  WPA-Enterprise 802.11 認証（非サポート）
    Authentication_WpaPsk,  //!<  WPA-Personal 802.11 認証
    Authentication_Wpa2,    //!<  WPA2-Enterprise 802.11 認証（非サポート）
    Authentication_Wpa2Psk, //!<  WPA2-Personal 802.11 認証
    Authentication_Unknown, //!<  不明
};

/**
 * @brief 無線 LAN 暗号方式を表します。
 */
enum Encryption
{
    Encryption_Invalid, //!< 無効
    Encryption_None,    //!< 暗号化なし
    Encryption_Wep,     //!< WEP
    Encryption_Tkip,    //!< TKIP（非サポート）
    Encryption_Aes,     //!< AES-CCMP
};

/**
 * @brief SSID 情報を表す構造体です。
 */
struct Ssid
{
    static const int HexSize = 32;  //!< SSID バイト列の最大サイズ
    uint8_t length;                 //!< SSID バイト列のサイズ
    unsigned char hex[HexSize];     //!< SSID バイト列
};

/**
 * @brief 認証方式および暗号化方式を表す構造体です。
 */
struct AuthEncryption
{
    Authentication authentication;  //!< 無線 LAN 認証方式
    Encryption encryption;          //!< 無線 LAN 暗号方式
};

/**
 * @brief 暗号化鍵情報を表す構造体です。
 */
struct SharedKey
{
    static const int KeyMaterialSize = 64;  //!< 暗号化鍵の元となるバイト列の最大サイズ
    uint8_t length;                         //!< 暗号化鍵の元となるバイト列のサイズ
    char keyMaterial[KeyMaterialSize];      //!< 暗号化鍵の元となるバイト列（パスワード、パスフレーズ）
};

/**
 * @brief SSID 設定を表す構造体です。
 */
struct SsidConfig
{
    Ssid ssid;          //!< SSID 情報を表す構造体
    bool nonBroadcast;  //!< ステルス設定の AP に対して接続試行を行うか否かを表すフラグ
};

/**
 * @brief セキュリティ情報を表す構造体です。
 */
struct Security
{
    AuthEncryption authEncryption;  //!< 認証方式および暗号化方式を表す構造体
    SharedKey sharedKey;            //!< 鍵情報を表す構造体
};

/**
 * @brief 無線 LAN 設定情報を表す構造体です。
 */
struct WirelessSettingData
{
    SsidConfig ssidConfig;  //!< SSID 設定を表す構造体
    Security security;      //!< セキュリティ情報を表す構造体
};

/**
 * @brief 無効な無線 LAN 設定情報を表します。
 */
static const WirelessSettingData InvalidWirelessSettingData = {};


inline bool operator==(const Ssid& lhs, const Ssid& rhs)
{
    return
        lhs.length == rhs.length &&
        std::memcmp(lhs.hex, rhs.hex, lhs.length) == 0;
}

inline bool operator!=(const Ssid& lhs, const Ssid& rhs)
{
    return !(lhs == rhs);
}

inline bool operator==(const AuthEncryption& lhs, const AuthEncryption& rhs)
{
    return
        lhs.authentication == rhs.authentication &&
        lhs.encryption == rhs.encryption;
}

inline bool operator!=(const AuthEncryption& lhs, const AuthEncryption& rhs)
{
    return !(lhs == rhs);
}

inline bool operator==(const SharedKey& lhs, const SharedKey& rhs)
{
    return
        lhs.length == rhs.length &&
        std::memcmp(lhs.keyMaterial, rhs.keyMaterial, lhs.length) == 0;
}

inline bool operator!=(const SharedKey& lhs, const SharedKey& rhs)
{
    return !(lhs == rhs);
}

inline bool operator==(const SsidConfig& lhs, const SsidConfig& rhs)
{
    return
        lhs.ssid == rhs.ssid &&
        lhs.nonBroadcast == rhs.nonBroadcast;
}

inline bool operator!=(const SsidConfig& lhs, const SsidConfig& rhs)
{
    return !(lhs == rhs);
}

inline bool operator==(const Security& lhs, const Security& rhs)
{
    return
        lhs.authEncryption == rhs.authEncryption &&
        lhs.sharedKey == rhs.sharedKey;
}

inline bool operator!=(const Security& lhs, const Security& rhs)
{
    return !(lhs == rhs);
}

inline bool operator==(const WirelessSettingData& lhs, const WirelessSettingData& rhs)
{
    return
        lhs.ssidConfig == rhs.ssidConfig &&
        lhs.security == rhs.security;
}

inline bool operator!=(const WirelessSettingData& lhs, const WirelessSettingData& rhs)
{
    return !(lhs == rhs);
}

}
}
