﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/nifm_Common.h>

#include <nn/nifm/nifm_TypesRequest.h>

#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_TypedStorage.h>


namespace nn
{
namespace nifm
{

struct RequestParameters;

/**
 * @brief 利用要求を表します。
 *
 * @details
 *  このクラスはシステム内部の特殊用途のために用意されています。
 *  一般的なネットワーク接続を求める場合は @ref nn::nifm::NetworkConnection クラスを利用してください。
 */
class Request
{
    NN_DISALLOW_COPY(Request);
    NN_DISALLOW_MOVE(Request);

private:
    // SystemEvent 内の CriticalSection のサイズが大きい Windows に合わせて 272U
    using Storage = nn::util::TypedStorage<char, 272U, sizeof(void*)>;

private:
    Storage m_Storage;

public:
    /**
     * @brief   コンストラクタです。
     *
     * @param[in]  parameters   利用要求の内容を指定します
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
     *  Request クラスのオブジェクトを大域的に配置することはできません。@n
     *
     *  GetSystemEvent 関数で取得されるイベントのクリアモードは nn::os::EventClearMode_AutoClear に設定されます。
     */
    explicit Request(const RequestParameters& parameters) NN_NOEXCEPT;

    /**
     * @brief   コンストラクタです。
     *
     * @param[in]  parameters       利用要求の内容を指定します
     * @param[in]  eventClearMode   GetSystemEvent 関数で取得されるシステムイベントのクリアモードを指定します
     *
     * @pre
     *   - ライブラリが初期化済み
     *
     * @details
     *  このコンストラクタの実行前にネットワーク接続ライブラリを初期化する必要があるため、
     *  Request クラスのオブジェクトを大域的に配置することはできません。
     */
    Request(const RequestParameters& parameters, nn::os::EventClearMode eventClearMode) NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     *
     * @details
     *  オブジェクトを破棄した場合、そのオブジェクトを通じておこなっていた提出も取り下げられます。
     *  これにより、システムのネットワーク接続が切断されたり、通信モジュールの占有が解除される可能性があることに注意してください。
     *  ネットワーク接続や通信モジュールの占有を維持したい間は、提出した要求オブジェクトを維持する必要があります。
     */
    ~Request() NN_NOEXCEPT;

    /**
     * @brief   利用要求を提出します。
     *
     * @details
     *  本関数を呼ぶと非同期処理が発生し、利用要求を受理できるかの検証がおこなわれます。
     *  検証中は GetRequestState 関数が RequestState_OnHold を返します。
     *  アプリケーションは本 API のコール後、定期的に GetRequestState の状態を確認もしくは
     *  GetSystemEvent 関数で得られるイベントのシグナルを監視し、
     *  GetRequestState 関数の応答が RequestState_OnHold から変化するのを待機してください。@n
     *
     *  利用要求の検証は本 API のコール1回につき1回のみおこなわれます。
     *  検証完了後あらためて利用要求を検証するには、もう一度本 API を呼び直す必要があります。@n
     *
     *  この関数はブロックすることなくすぐに返ります。@n
     *
     *  この関数は SubmitAndWait 関数とスレッドアンセーフです。
     */
    void Submit() NN_NOEXCEPT;

    /**
     * @brief   利用要求を提出し、結果が確定するまでブロックします。
     *
     * @details
     *  本関数を呼ぶと利用要求を受理できるかの検証がおこなわれ、検証の結果が出るまでブロックします。
     *  本関数を抜けたら GetRequestState 関数で要求の検証結果を確認してください。@n
     *  GetRequestState 関数が RequestState_OnHold を返すあいだに利用要求を取り下げると、実際には受理可能な利用要求を受理できないことがあるため、この間はユーザー操作以外でのキャンセルはおこなわないことを推奨します。@n
     *
     *  利用要求の検証は本 API のコール1回につき1回のみおこなわれます。
     *  検証完了後あらためて利用要求を検証するには、もう一度本 API を呼び直す必要があります。@n
     *
     *  この関数を抜けたあと、 GetSystemEvent 関数から取得されるイベントはシグナル状態になっていることに注意してください。@n
     *
     *  システムの内部状態によって、検証処理はごく短時間で完了することがあります。@n
     *
     *  この関数は SubmitRequest 関数とスレッドアンセーフです。
     */
    void SubmitAndWait() NN_NOEXCEPT;

    /**
     * @brief   提出した利用要求を取り消します。
     *
     * @details
     *  システムに提出していた利用要求を取り下げます。@n
     *
     *  この関数は利用要求が検証中か否かに関わらず呼ぶことができます。
     *  SubmitAndWait 関数のブロックを解除したい場合、この関数を別スレッドから呼んでください。@n
     *
     *  この関数を抜けたあと、 GetSystemEvent 関数から取得されるイベントはシグナル状態になっていることに注意してください。@n
     *
     *  この関数は要求の内容に興味をなくしたことを宣言するだけであり、ネットワーク接続の切断や通信モジュールの解放を要求するものではありません。
     */
    void Cancel() NN_NOEXCEPT;

    /**
     * @brief   利用要求の状態を取得します。
     *
     * @return  利用要求の状態が返ります。
     * @retval  RequestState_Free       要求は提出されていません
     * @retval  RequestState_OnHold     要求は検証中です
     * @retval  RequestState_Accepted   要求は受理されています
     * @retval  RequestState_Blocking   要求は取り下げられるべきです
     *
     * @details
     *  利用要求の状態を取得します。
     */
    RequestState GetRequestState() NN_NOEXCEPT;

    /**
     * @brief   利用要求の検証結果を取得します。
     *
     * @retresult
     *   @handleresult{
     *     nn::ResultSuccess, 利用要求は受理されました。
     *   }
     * @endretresult
     *
     * @details
     *  利用要求の検証結果を取得します。@n
     *
     *  この関数は正式版ではありません。
     */
    nn::Result GetResult() NN_NOEXCEPT;

    /**
     * @brief   利用要求のハンドルを取得します。
     *
     * @return  利用要求のハンドルが返ります。
     *
     * @details
     *  利用要求のハンドルを取得します。
     */
    RequestHandle GetHandle() NN_NOEXCEPT;

    /**
     * @brief   利用要求の状態変化を待ち受けるためのシステムイベントを取得します。@n
     *
     * @return  nn::os::SystemEvent オブジェクトへの参照を返します。
     *
     * @details
     *  利用要求の状態が変化したときにシグナルされるシステムイベントへの参照を取得します。@n
     *
     *  システムイベントのクリアモードは、本クラスのコンストラクタに指定する引数によって決定されます。
     */
    nn::os::SystemEvent& GetSystemEvent() NN_NOEXCEPT;
};

}
}

