﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>

#include <nn/lmem/lmem_UnitHeap.h>
#include <nn/lmem/lmem_FrameHeap.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ExpHeapAllocator.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class HeapBase
{
public:
    virtual ~HeapBase() NN_NOEXCEPT
    {
    }

    virtual void* Allocate(size_t size) NN_NOEXCEPT = 0;
    virtual void* Allocate(size_t size, int alignment) NN_NOEXCEPT = 0;
    virtual void Free(void* pBlock) NN_NOEXCEPT = 0;
    virtual void Destroy() NN_NOEXCEPT = 0;
    virtual void Clear() NN_NOEXCEPT = 0;

    template<typename T, typename... Ts>
    T* Create(Ts... args)
    {
        void* p = Allocate(sizeof(T));

        if (p != nullptr)
        {
            return new(p) T(args...);
        }
        else
        {
            return nullptr;
        }
    }

    template<typename T>
    void Delete(T* p)
    {
        p->~T();
        Free(p);
    }
};

class UnitHeapBase : public HeapBase
{
private:
    nn::lmem::HeapHandle m_HeapHandle;
    const size_t m_UnitSize;
    const int m_UnitAlignment;

public:
    UnitHeapBase(nn::lmem::HeapHandle heapHandle, size_t unitSize, int unitAlignment)
        : m_HeapHandle(heapHandle),
          m_UnitSize(unitSize),
          m_UnitAlignment(unitAlignment)
    {
        NN_UNUSED(m_UnitSize);
        NN_UNUSED(m_UnitAlignment);
    }

    virtual ~UnitHeapBase() NN_NOEXCEPT
    {
        Destroy();
    }

    virtual void* Allocate(size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_EQUAL(size, m_UnitSize);
        NN_UNUSED(size);

        return nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
    }

    virtual void* Allocate(size_t size, int alignment) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_EQUAL(size, m_UnitSize);
        NN_UNUSED(size);

        NN_SDK_ASSERT_EQUAL(alignment, m_UnitAlignment);
        NN_UNUSED(alignment);

        return nn::lmem::AllocateFromUnitHeap(m_HeapHandle);
    }

    virtual void Free(void* pBlock) NN_NOEXCEPT NN_OVERRIDE
    {
        nn::lmem::FreeToUnitHeap(m_HeapHandle, pBlock);
    }

    virtual void Destroy() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::lmem::DestroyUnitHeap(m_HeapHandle);
    }

protected:
    void SetHeapHandle(nn::lmem::HeapHandle heapHandle) NN_NOEXCEPT
    {
        m_HeapHandle = heapHandle;
    }
};

template<typename T, int count>
class UnitHeap : public UnitHeapBase
{
private:
    using StorageUnit = nn::util::TypedStorage<T, sizeof(T), NN_ALIGNOF(T)>;

private:
    StorageUnit m_Buffer[count];
    char m_BufferPlus[4];   // X86 でスタックに確保された場合、 TypedStorage のアラインメント指定にかかわらず、8バイトアラインメントが取られない場合があることへの対策
    nn::lmem::HeapCommonHead m_HeapCommonHead;

public:
    UnitHeap() NN_NOEXCEPT
        : UnitHeapBase(
              nn::lmem::CreateUnitHeap(
                  m_Buffer,
                  sizeof(m_Buffer) + sizeof(m_BufferPlus),
                  sizeof(T),
                  nn::lmem::CreationOption_NoOption,
                  NN_ALIGNOF(T),
                  &m_HeapCommonHead
              ),
              sizeof(T),
              NN_ALIGNOF(T)
          )
    {
    }

    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        Destroy();
        SetHeapHandle(
            nn::lmem::CreateUnitHeap(
                m_Buffer,
                sizeof(m_Buffer) + sizeof(m_BufferPlus),
                sizeof(T),
                nn::lmem::CreationOption_NoOption,
                NN_ALIGNOF(T),
                &m_HeapCommonHead
            )
        );
    }

    template<typename... Ts>
    T* Create(Ts... args) NN_NOEXCEPT
    {
        return HeapBase::Create<T, Ts...>(args...);
    }
};

class FrameHeapBase : public HeapBase
{
private:
    nn::lmem::HeapHandle m_HeapHandle;

public:
    explicit FrameHeapBase(nn::lmem::HeapHandle heapHandle)
        : m_HeapHandle(heapHandle)
    {
    }

    virtual ~FrameHeapBase() NN_NOEXCEPT
    {
        Destroy();
    }

    virtual void* Allocate(size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::lmem::AllocateFromFrameHeap(m_HeapHandle, size);
    }

    virtual void* Allocate(size_t size, int alignment) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::lmem::AllocateFromFrameHeap(m_HeapHandle, size, alignment);
    }

    virtual void Free(void* pBlock) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pBlock);
        NN_SDK_ASSERT(false, "FrameHeapBase::Free() is not implemented.\n");
    }

    virtual void Destroy() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::lmem::DestroyFrameHeap(m_HeapHandle);
    }

    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::lmem::FreeToFrameHeap(m_HeapHandle, nn::lmem::FreeMode_All);
    }
};

template<size_t BufferSize>
class FrameHeap : public FrameHeapBase
{
private:
    nn::util::TypedStorage<char, BufferSize, 8> m_Buffer;
    nn::lmem::HeapCommonHead m_HeapCommonHead;

public:
    FrameHeap() NN_NOEXCEPT
        : FrameHeapBase(nn::lmem::CreateFrameHeap(&m_Buffer, BufferSize, nn::lmem::CreationOption_NoOption, &m_HeapCommonHead))
    {
    }
};

class ExpHeapBase : public HeapBase
{
private:
    nn::lmem::HeapHandle m_HeapHandle;

public:
    explicit ExpHeapBase(nn::lmem::HeapHandle heapHandle)
        : m_HeapHandle(heapHandle)
    {
    }

    virtual ~ExpHeapBase() NN_NOEXCEPT
    {
        Destroy();
    }

    virtual void* Allocate(size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::lmem::AllocateFromExpHeap(m_HeapHandle, size);
    }

    virtual void* Allocate(size_t size, int alignment) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::lmem::AllocateFromExpHeap(m_HeapHandle, size, alignment);
    }

    virtual void Free(void* pBlock) NN_NOEXCEPT NN_OVERRIDE
    {
        nn::lmem::FreeToExpHeap(m_HeapHandle, pBlock);
    }

    virtual void Destroy() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::lmem::DestroyExpHeap(m_HeapHandle);
    }

protected:
    void SetHeapHandle(nn::lmem::HeapHandle heapHandle) NN_NOEXCEPT
    {
        m_HeapHandle = heapHandle;
    }
};

template<size_t BufferSize>
class ExpHeap : public ExpHeapBase
{
private:
    nn::util::TypedStorage<char, BufferSize, 8> m_Buffer;

public:
    ExpHeap() NN_NOEXCEPT
        : ExpHeapBase(nn::lmem::CreateExpHeap(&m_Buffer, BufferSize, nn::lmem::CreationOption_NoOption))
    {
    }

    virtual void Clear() NN_NOEXCEPT NN_OVERRIDE
    {
        Destroy();
        SetHeapHandle(nn::lmem::CreateExpHeap(&m_Buffer, BufferSize, nn::lmem::CreationOption_NoOption));
    }
};

template<size_t Size>
class SfExpHeapAllocator
{
private:
    char m_Buffer[Size];
    nn::sf::ExpHeapAllocator m_ExpHeapAllocator;

public:
    explicit SfExpHeapAllocator(nn::lmem::CreationOption option) NN_NOEXCEPT
    {
        m_ExpHeapAllocator.Attach(nn::lmem::CreateExpHeap(m_Buffer, Size, option));
    }

    nn::sf::ExpHeapAllocator* GetBase() NN_NOEXCEPT
    {
        return &m_ExpHeapAllocator;
    }
};

}
}
}
