﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nifm/detail/nifm_CommonDetail.h>
#include <nn/nifm/detail/core/accessPoint/nifm_AccessPointBase.h>

#include <nn/nifm/nifm_TypesAccessPointExt.h>


namespace nn
{
namespace nifm
{
namespace detail
{

class WirelessInterface;

enum class AccessPointSortKind
{
    Rssi_Ascent,
    Rssi_Descent,
    Ssid_Ascent,
    Ssid_Descent,
};

class WirelessAccessPoint : public AccessPointBase
{
    friend WirelessInterface;

private:
    AccessPointExtData m_AccessPointExt;

    // TODO: その他のパラメータ
    // TODO: ID情報は不要？

public:
    explicit WirelessAccessPoint(WirelessInterface* pWirelessInterface) NN_NOEXCEPT;
    virtual ~WirelessAccessPoint() NN_NOEXCEPT;

    virtual NetworkType GetNetworkType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return NetworkType_Internet;
    }

    // ソート時の優先順位を返します
    // 値が小さいほど高優先度です
    // オーバーライドしない場合 0 を返します
    virtual int GetPriority() const NN_NOEXCEPT NN_OVERRIDE;

    // オブジェクトの内容を AccessPointData に変換します
    // 仮想的なアクセスポイントの場合、失敗を返し、エクスポートをしません
    virtual nn::Result TryExport(AccessPointData* pOutAccessPointData) const NN_NOEXCEPT NN_OVERRIDE;

    virtual AccessPointBase* CopyTo(void* p, size_t size) const NN_NOEXCEPT NN_OVERRIDE;
    virtual size_t GetSize() const NN_NOEXCEPT NN_OVERRIDE;

private:
    virtual nn::Result ConnectImpl(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DisconnectImpl() const NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result ReleaseImpl() const NN_NOEXCEPT NN_OVERRIDE;
    virtual void UpdateAccessPoint(const NetworkProfileBase& networkProfile) NN_NOEXCEPT NN_OVERRIDE;

    // 派生クラスが持つアトリビュートが指定された条件に合致するか調べます
    // 基底クラスが持つアトリビュートが指定された条件に合致しない場合、この関数は呼ばれません
    // 派生クラスが独自のアトリビュートを持たない場合、オーバーライドは不要です
    virtual bool IsPossiblyAvailableImpl(const NetworkProfileBase& networkProfile, const AggregatedRequestType& aggregatedRequest) const NN_NOEXCEPT NN_OVERRIDE;

    // サポート/非サポートにかかわらず最も強いセキュリティタイプを返す
    bool GetTopSecurity(Authentication *pOutAuthentiCation, Encryption* pOutEncryption, Encryption* pOutGroupEncryption) const NN_NOEXCEPT;

public:
    Ssid* GetSsid(Ssid *pOutSsid) const NN_NOEXCEPT;
    MacAddress* GetBssid(MacAddress *pOutBssid) const NN_NOEXCEPT;
    int32_t GetRssi() const NN_NOEXCEPT;
    LinkLevel GetLinkLevel() const NN_NOEXCEPT;
    int16_t GetChannel() const NN_NOEXCEPT;
    Authentication GetAuthentication() const NN_NOEXCEPT;
    Encryption GetEncryption() const NN_NOEXCEPT;
    Encryption GetGroupEncryption() const NN_NOEXCEPT;

    void SetSsid(const Ssid &ssid) NN_NOEXCEPT;
    void SetSsid(const unsigned char* ssid, size_t length) NN_NOEXCEPT;
    void SetBssid(const MacAddress &bssid) NN_NOEXCEPT;
    void SetRssi(int32_t indication) NN_NOEXCEPT;
    void SetLinkLevel(LinkLevel linkLevel) NN_NOEXCEPT;
    void SetChannel(int16_t channel) NN_NOEXCEPT;

    void SetWpaAuthenticationAndEncryption(const Authentication authentication, const Encryption encryption, const Encryption groupEncryption) NN_NOEXCEPT;
    void SetWepAuthenticationAndEncryption(const Authentication authentication, const Encryption encryption) NN_NOEXCEPT;

    virtual bool operator==(const AccessPointBase& rh) const NN_NOEXCEPT NN_OVERRIDE;
    virtual bool operator==(const WirelessAccessPoint& rh) const NN_NOEXCEPT NN_OVERRIDE;

    // 指定した認証・暗号化方式をサポートする AP か否か
    bool IsSupported(Authentication auth, Encryption enc) const NN_NOEXCEPT;
    // サポート対象の認証・暗号化方式のいずれかで接続可能な AP か否か
    bool IsSupported() const NN_NOEXCEPT;

    bool IsStealth() const NN_NOEXCEPT;
    static bool IsStealth(const Ssid& ssid) NN_NOEXCEPT;

    bool IsEqual(const Ssid& ssid) const NN_NOEXCEPT;
    bool IsPossiblyEqual(const Ssid& ssid, bool nonBroadcast) const NN_NOEXCEPT;

    bool IsEqual(const MacAddress& bssid) const NN_NOEXCEPT;
};

}
}
}
