﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/news_TypesAdmin.h>

namespace nn { namespace news {

namespace detail { namespace ipc {

class INewsDatabaseService;

}}

/*!
    @brief      ニュースの管理データベースを扱うクラスです。
*/
class Database
{
private:
    NN_DISALLOW_COPY(Database);
    NN_DISALLOW_MOVE(Database);

public:
    /*!
        @brief      コンストラクタです。
    */
    Database() NN_NOEXCEPT;

    /*!
        @brief      デストラクタです。
    */
    ~Database() NN_NOEXCEPT;

    /*!
        @brief      データベースを開きます。

        @return     処理結果。

        @details
                    データベースにアクセスするための準備を行います。

                    データベースを開くことで、バックグラウンドで行われるニュースデータの保存とデータベースの更新に対するロックが行われます。@n
                    データベースを開いている間、ストレージの空き容量不足によるニュースデータの自動削除を抑制することができます。
    */
    nn::Result Open() NN_NOEXCEPT;

    /*!
        @brief      データベースを閉じます。
    */
    void Close() NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでレコードを取得します。

        @param[out] outCount        取得した数。
        @param[out] outRecords      レコードリスト。
        @param[in]  wherePhrase     WHERE 句。
        @param[in]  orderByPhrase   ORDER BY 句。
        @param[in]  offset          オフセット。
        @param[in]  count           レコードリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outRecords != nullptr
            - wherePhrase != nullptr
            - strlen(wherePhrase) <= @ref WherePhraseLengthMax
            - orderByPhrase != nullptr
            - strlen(orderByPhrase) <= @ref OrderByPhraseLengthMax
            - offset >= 0
            - count > 0
            - データベースを開いている状態である。

        @details
                    本関数は、以下の SQL 文を発行してフィルタリング・ソートを行います。@n
                    WHERE 句を空文字列にした場合、全てのレコードが取得できます。

                    - SELECT * FROM news WHERE $wherePhrase ORDER BY $orderByPhrase LIMIT $offset, $count
    */
    nn::Result GetList(int* outCount, NewsRecord* outRecords,
        const char* wherePhrase, const char* orderByPhrase, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでデータ件数を取得します。

        @param[out] outCount    取得した数。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - wherePhrase != nullptr
            - strlen(wherePhrase) <= @ref WherePhraseLengthMax
            - データベースを開いている状態である。

        @details
                    本関数は、以下の SQL 文を発行して件数を取得します。@n
                    WHERE 句を空文字列にした場合、保存されているデータ件数を取得できます。

                    - SELECT COUNT(*) FROM news WHERE $wherePhrase
    */
    nn::Result Count(int* outCount, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定したクエリでデータ件数を取得します。

        @param[out] outCount    取得した数。
        @param[in]  distinct    キーの重複を排除するかどうか。@n
                                キーの重複を排除した場合、指定したキーのユニーク数を取得することができます。
        @param[in]  key         キー。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - key != nullptr
            - 0 <= strlen(key) <= @ref KeyLengthMax
            - wherePhrase != nullptr
            - strlen(wherePhrase) <= @ref WherePhraseLengthMax
            - データベースを開いている状態である。

        @details
                    本関数は、以下の SQL 文を発行して件数を取得します。

                    - SELECT COUNT([DISTINCT] $key) FROM news WHERE $wherePhrase
    */
    nn::Result Count(int* outCount, bool distinct, const char* key, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定した条件に当てはまる整数値を持つキーの値を更新します。

        @param[in]  key         キー。
        @param[in]  value       代入値。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - key != nullptr
            - 0 <= strlen(key) <= @ref KeyLengthMax
            - wherePhrase != nullptr
            - strlen(wherePhrase) <= @ref WherePhraseLengthMax
            - データベースを開いている状態である。

        @details
                    指定したキーが整数値を持たない場合や更新不可の場合、本関数は失敗します。

                    本関数は以下の SQL 文を発行します。

                    - UPDATE news SET $key = $value WHERE $wherePhrase
    */
    nn::Result Update(const char* key, int32_t value, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定した条件に当てはまるキーの文字列値を更新します。

        @param[in]  key         キー。
        @param[in]  value       代入値。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - key != nullptr
            - 0 <= strlen(key) <= @ref KeyLengthMax
            - value != nullptr
            - strlen(value) <= @ref StringValueLengthMax
            - wherePhrase != nullptr
            - strlen(wherePhrase) <= @ref WherePhraseLengthMax
            - データベースを開いている状態である。

        @details
                    指定したキーが文字列値を持たない場合や更新不可の場合、本関数は失敗します。

                    本関数は以下の SQL 文を発行します。

                    - UPDATE news SET $key = '$value' WHERE $wherePhrase
    */
    nn::Result Update(const char* key, const char* value, const char* wherePhrase) NN_NOEXCEPT;

    /*!
        @brief      指定した条件に当てはまる整数値を持つキーの値を加算更新します。

        @param[in]  key         キー。
        @param[in]  value       加算値。
        @param[in]  wherePhrase WHERE 句。

        @return     処理結果。

        @pre
            - key != nullptr
            - 0 <= strlen(key) <= @ref KeyLengthMax
            - wherePhrase != nullptr
            - strlen(wherePhrase) <= @ref WherePhraseLengthMax
            - データベースを開いている状態である。

        @details
                    指定したキーが整数値を持たない場合や更新不可の場合、本関数は失敗します。

                    本関数は以下の SQL 文を発行します。

                    - UPDATE news SET $key = $key + $value WHERE $wherePhrase
    */
    nn::Result UpdateWithAddition(const char* key, int32_t value, const char* wherePhrase) NN_NOEXCEPT;

private:
    //
    detail::ipc::INewsDatabaseService* m_Handle;
};

}}
