﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/news/detail/service/news_Common.h>

namespace nn { namespace news { namespace detail { namespace service { namespace core {

/*!
    @brief      タスクの管理モジュールです。
*/
class NewsTaskManager
{
public:
    /*!
        @brief      タスクの最大数です。
    */
    static const int TaskCountMax = 1024;

    /*!
        @brief      同時に購読できるトピックの最大数です。
    */
    static const int SubscriptionCountMax = 256;

    /*!
        @brief      タスクの状態です。
    */
    enum TaskStatus : char
    {
        TaskStatus_Empty                   = 0,   //!< 状態なし。
        TaskStatus_Runnable                = 'Q', //!< 実行可能状態。
        TaskStatus_Running                 = 'R', //!< 実行状態。
        TaskStatus_SystemUpdateRequired    = 'S', //!< システム更新要求状態。
        TaskStatus_NintendoAccountRequired = 'N', //!< ニンテンドーアカウント登録要求状態。
        TaskStatus_Wait                    = 'W', //!< 待機状態。
        TaskStatus_Done                    = 'D', //!< 完了状態。
        TaskStatus_Error                   = 'E', //!< エラーによる一時停止状態。
        TaskStatus_Expired                 = 'X'  //!< 有効期限切れによる停止状態。
    };

    /*!
        @brief      購読状態です。
    */
    enum InternalSubscriptionStatus : char
    {
        InternalSubscriptionStatus_Empty        = 0,   //!< 状態なし。
        InternalSubscriptionStatus_Deleted      = 'D', //!< 削除状態。
        InternalSubscriptionStatus_Unsubscribed = 'U', //!< 購読解除状態。
        InternalSubscriptionStatus_Subscribed   = 'S'  //!< 購読状態。
    };

    /*!
        @brief      タスクレコードです。
    */
    struct Record
    {
        // 32
        TopicId topicId;
        // 40
        ETag eTag;
        // 8
        RevisionHash revisionHash;
        // 8
        TaskStatus status;
        InternalSubscriptionStatus subscription;
        bool isAuto;
        bool isNpnsRegistered;
        bool isNpnsPerformed;
        char reserved;
        char unused[2];
        // 8
        uint64_t unsubscribedIndex;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 96);

    /*!
        @brief      システム起動中のみ使用される揮発性のレコードです。
    */
    struct VolatileRecord
    {
        // 8
        nn::os::Tick runnableTick;
    };

private:
    /*!
        @brief      コンストラクタです。
    */
    NewsTaskManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static NewsTaskManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(NewsTaskManager, instance);
        return instance;
    }

public:
    /*!
        @brief      タスクを読み込みます。

        @return     処理結果。

        @details
                    待機状態、または、エラー状態のタスクはすべて実行可能状態に遷移します。@n
                    これは、以下の理由のためです。

                    - 起動していない間（スリープ中ではない）の時間経過の計測する処理を省くため。
                    - システム再起動後にエラー状態の回復が見込まれるため。

                    本関数はシステム起動時に 1 回だけ呼び出してください。
    */
    nn::Result Load() NN_NOEXCEPT;

    /*!
        @brief      タスクをクリアします。
    */
    nn::Result Clear() NN_NOEXCEPT;

    /*!
        @brief      スケジュール処理を行います。

        @details
                    現在時刻が、待機状態タスクの実行可能時刻になった場合、実行可能状態に遷移します。
    */
    void Schedule() NN_NOEXCEPT;

    /*!
        @brief      購読状態を取得します。

        @param[in]  topicId トピック ID。

        @return     購読状態。
    */
    SubscriptionStatus GetSubscriptionStatus(const TopicId& topicId) NN_NOEXCEPT;

    /*!
        @brief      購読状態を設定します。

        @param[in]  topicId トピック ID。
        @param[in]  status  購読状態。

        @return     処理結果。

        @details
                    購読状態によって、レコードは以下のようになります。

                    - SubscriptionStatus_Unconfigured@n
                      レコードは検索対象から外れ、新規登録時に空きがない場合、上書き対象になります。
                    - SubscriptionStatus_Unsubscribed@n
                      レコードは検索対象には残りますが、実行タスクの選択から外れます。
                    - SubscriptionStatus_Subscribed、SubscriptionStatus_AutoSubscribed@n
                      タスクが新規登録される場合、実行可能状態になります。
    */
    nn::Result SetSubscriptionStatus(const TopicId& topicId, SubscriptionStatus status) NN_NOEXCEPT;

    /*!
        @brief      現在実行すべきタスクを取得します。

        @param[in]  outRecord   レコード。

        @return     処理結果。

        @pre
            - outRecord != nullptr

        @details
                    本関数は、実行状態のタスクを優先して取得します。@n
                    実行状態のタスクが存在しない場合、実行可能状態のタスクを 1 つ選択します。

                    現在実行すべきタスクがない場合、 ResultNotFound を返します。
    */
    nn::Result GetCurrentTask(Record* outRecord) const NN_NOEXCEPT;

    /*!
        @brief      トピックリストを取得します。

        @param[out] outCount    取得したトピック ID 数。
        @param[out] outTopicIds トピック ID リスト。
        @param[in]  count       トピック ID リストの要素数。
        @param[in]  filter      フィルター。

        @pre
            - outCount != nullptr
            - outTopicIds != nullptr
            - count > 0
    */
    void GetTopicList(int* outCount, TopicId* outTopicIds, int count, SubscriptionStatusFilter filter) const NN_NOEXCEPT;

    /*!
        @brief      次回のスケジュールまでの時間間隔を取得します。

        @param[out] outInterval 次回のスケジュールまでの時間間隔。

        @return     スケジュールすべきタスクが存在するかどうか。

        @pre
            - outInterval != nullptr
    */
    bool GetNextScheduleInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT;

    /*!
        @brief      NPNS の購読処理を実行すべきかどうかを取得します。

        @return     NPNS の購読処理を実行すべきかどうか。
    */
    bool IsNpnsSubscriptionProcessingRequired() const NN_NOEXCEPT;

    /*!
        @brief      購読処理を行うべき NPNS トピックを取得します。

        @param[in]  outTopicId  トピック ID。

        @return     購読処理を行うべきトピックが存在するかどうか。

        @pre
            - outTopicId != nullptr
    */
    bool GetNpnsTopicToBeSubscribed(TopicId* outTopicId) const NN_NOEXCEPT;

    /*!
        @brief      購読解除処理を行うべき NPNS トピックを取得します。

        @param[in]  outTopicId  トピック ID。

        @return     購読解除処理を行うべきトピックが存在するかどうか。

        @pre
            - outTopicId != nullptr
    */
    bool GetNpnsTopicToBeUnsubscribed(TopicId* outTopicId) const NN_NOEXCEPT;

    /*!
        @brief      スケジュール処理を行うべき時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::TimerEvent& GetScheduleEvent() NN_NOEXCEPT;

    /*!
        @brief      NPNS の購読・購読解除処理が必要となった時に Signal されるイベントを取得します。

        @return     イベント。
    */
    nn::os::Event& GetSubscriptionEvent() NN_NOEXCEPT;

    /*!
        @brief      指定したタスクを実行可能状態にします。

        @param[in]  topicId トピック ID。

        @return     処理結果。
    */
    nn::Result Run(const TopicId& topicId) NN_NOEXCEPT;

    /*!
        @brief      指定したタスクを実行状態にします。

        @param[in]  topicId トピック ID。

        @return     処理結果。

        @details
                    本関数はタスクを実行可能状態ではなく、いきなり実行状態にします。@n
                    現在実行状態のタスクは、実行可能状態に変更されます。

                    TODO: Running 状態は廃止し、その代わりタスクリストの先頭に移動させる対応を行う。
    */
    nn::Result RunImmediately(const TopicId& topicId) NN_NOEXCEPT;

    /*!
        @brief      指定したタスクを待機状態にします。

        @param[in]  topicId     トピック ID。
        @param[in]  waitTime    待機時間。

        @return     処理結果。

        @details
                    実行可能状態、実行状態のタスクを待機状態にします。@n
                    待機時間に負数を指定した場合、実行可能状態への遷移は電源断からの復帰時のみとなります。
    */
    nn::Result Wait(const TopicId& topicId, int32_t waitTime) NN_NOEXCEPT;

    /*!
        @brief      指定したタスクの配信リストの ETag を更新します。

        @param[in]  topicId トピック ID。
        @param[in]  eTag    配信リストの ETag。

        @return     処理結果。

        @details
                    配信リストの ETag を更新することで、配信リスト取得時に If-None-Match リクエストが行われるようになります。
    */
    nn::Result UpdateETag(const TopicId& topicId, const ETag& eTag) NN_NOEXCEPT;

    /*!
        @brief      プッシュ通知の受信を通知します。

        @param[in]  topicId         トピック ID。
        @param[in]  isPseudo        擬似トピックかどうか。
        @param[in]  revisionHash    リビジョンのハッシュ値。
        @param[in]  waitTime        待機時間。

        @return     処理結果。

        @details
                    リビジョンのハッシュ値がタスクに記録されているものと一致した場合、タスクの状態は変化しません。

                    プッシュ通知トリガーによる実行では、多数台同時リクエストが発生する可能性があります。@n
                    負荷分散のため、タスクの実行はデバイスごとにランダムな待機時間を設定してください。

                    タスク管理外のトピック ID が指定された場合、NPNS トピックの購読解除を要求する特別なタスクが作成されます。@n
                    ただし、空きがない場合は何もしません。
    */
    nn::Result NotifyNotificationReceived(const TopicId& topicId, bool isPseudo, const RevisionHash& revisionHash, int32_t waitTime = 0) NN_NOEXCEPT;

    /*!
        @brief      タスクの実行完了を通知します。

        @param[in]  record  レコード。
        @param[in]  result  処理結果。

        @return     処理結果。
    */
    nn::Result NotifyDone(const Record& record, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      NPNS トピックの購読処理が行われたことを通知します。

        @param[in]  topicId トピック ID。
        @param[in]  result  処理結果。

        @return     処理結果。
    */
    nn::Result NotifyNpnsTopicSubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      NPNS トピックの購読解除処理が行われたことを通知します。

        @param[in]  topicId トピック ID。
        @param[in]  result  処理結果。

        @return     処理結果。
    */
    nn::Result NotifyNpnsTopicUnsubscriptionProcessed(const TopicId& topicId, nn::Result result) NN_NOEXCEPT;

    /*!
        @brief      ニンテンドーアカウントの紐付けを通知します。

        @return     処理結果。
    */
    nn::Result NotifyNintendoAccountLinked() NN_NOEXCEPT;

    /*!
        @brief      配信リスト URL に指定する居住国リストが更新されたことを通知します。

        @return     処理結果。
    */
    nn::Result NotifyAccountCountryListUpdated() NN_NOEXCEPT;

    /*!
        @brief      システムが更新済みであることを通知します。

        @return     処理結果。
    */
    nn::Result NotifySystemUpdated() NN_NOEXCEPT;

    /*!
        @brief      スリープ復帰を通知します。
    */
    void NotifySleepAwaked() NN_NOEXCEPT;

    /*!
        @brief      ネットワークの接続を通知します。
    */
    void NotifyNetworkConnected() NN_NOEXCEPT;

private:
    //
    typedef bool (*FilterFunction)(InternalSubscriptionStatus status);

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    nn::os::TimerEvent m_ScheduleEvent;
    //
    nn::os::Event m_SubscriptionEvent;
    //
    Record m_Records[TaskCountMax];
    VolatileRecord m_VolatileRecords[TaskCountMax];
    //
    int m_SubscriptionCount;
    //
    bool m_IsNetworkConnectionAvailable;

private:
    //
    nn::Result LoadImpl() NN_NOEXCEPT;
    //
    nn::Result Save() NN_NOEXCEPT;
    //
    bool Verify() const NN_NOEXCEPT;
    //
    int SearchRecord(const TopicId& topicId) const NN_NOEXCEPT;
    //
    nn::Result AddRecord(const TopicId& topicId, SubscriptionStatus status) NN_NOEXCEPT;
    //
    void SetRunnable(int index) NN_NOEXCEPT;
    void SetRunnableWithoutSignal(int index) NN_NOEXCEPT;
    //
    void SetWait(int index, int32_t waitTime) NN_NOEXCEPT;
    //
    void SetScheduleTimer() NN_NOEXCEPT;
    bool GetScheduleMinInterval(nn::TimeSpan* outInterval) const NN_NOEXCEPT;

private:
    //
    static bool FilterWithNone(InternalSubscriptionStatus status) NN_NOEXCEPT;
    static bool FilterWithUnsubscribed(InternalSubscriptionStatus status) NN_NOEXCEPT;
    static bool FilterWithSubscribed(InternalSubscriptionStatus status) NN_NOEXCEPT;
    //
    static bool IsServerTemporaryError(nn::Result result) NN_NOEXCEPT;
    static bool IsServerFailureError(nn::Result result) NN_NOEXCEPT;
};

}}}}}
