﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   LBL ライブラリのBL点灯/消灯機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/lbl/lbl_Type.h>

namespace nn {
namespace lbl {

/**
 * @brief LCDバックライトを点灯します。
 *
 * @param[in]   fadeInTime  点灯させるまでの遷移時間(即座に点灯するときは0を指定)
 * @pre         初期化されている必要があります。
 *              @a fadeInTime.GetNanoSeconds() >= 0 である必要があります。
 * @post        LCDバックライトの点灯の状態への遷移が開始されます。
 *              このとき、既に点灯の状態であるときは何も処理されません。
 *              消灯の状態であるときは、GetBacklightSwitchStatus() で得られる値が @a BacklightSwitchStatus_OffInTransitionToOn となり、
 *              @a fadeInTime で指定した遷移時間後には点灯の状態への遷移が完了し、
 *              GetBacklightSwitchStatus() で得られる値が @a BacklightSwitchStatus_OnWithStability となります。
 * @details     LCDバックライトを点灯します。
 */
void SwitchBacklightOn(nn::TimeSpan fadeInTime) NN_NOEXCEPT;

/**
 * @brief LCDバックライトを消灯します。
 *
 * @param[in]   fadeOutTime 消灯させるまでの遷移時間(即座に消灯するときは0を指定)
 * @pre         初期化されている必要があります。
 *              @a fadeOutTime.GetNanoSeconds() >= 0 である必要があります。
 * @post        LCDバックライトの消灯の状態への遷移が開始されます。
 *              このとき、既に消灯の状態であるときは何も処理されません。
 *              点灯の状態であるときは、GetBacklightSwitchStatus() で得られる値が @a BacklightSwitchStatus_OnInTransitionToOff となり、
 *              @a fadeOutTime で指定した遷移時間後には消灯の状態への遷移が完了し、
 *              GetBacklightSwitchStatus() で得られる値が @a BacklightSwitchStatus_OffWithStability となります。
 * @details     LCDバックライトを消灯します。
 */
void SwitchBacklightOff(nn::TimeSpan fadeOutTime) NN_NOEXCEPT;

/**
 * @brief LCDバックライトの点灯/消灯の状態を取得します。
 *
 * @return      点灯/消灯の状態を返します。
 * @pre         初期化されている必要があります。
 * @details     LCDバックライトの点灯/消灯の状態を取得します。
 */
BacklightSwitchStatus GetBacklightSwitchStatus() NN_NOEXCEPT;

/**
 * @brief 低輝度化を有効にします。
 *
 * @pre         初期化されている必要があります。
 * @post        低輝度化が有効になります。
 * @details     低輝度化を有効にします。
 */
void EnableDimming() NN_NOEXCEPT;

/**
 * @brief 低輝度化を無効にします。
 *
 * @pre         初期化されている必要があります。
 * @post        低輝度化が無効になります。
 * @details     低輝度化を無効にします。
 */
void DisableDimming() NN_NOEXCEPT;

/**
 * @brief 低輝度化の有効/無効の状態を取得します。
 *
 * @return      低輝度化の有効/無効の状態を返します(true: 有効, false: 無効)。
 * @pre         初期化されている必要があります。
 * @details     低輝度化の有効/無効の状態を取得します。
 */
bool IsDimmingEnabled() NN_NOEXCEPT;

} // lbl
} // nn


