﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレットの共通引数の受け渡しをおこなうクラスの非公開ヘッダ
 */

#pragma once

#include <nn/la/la_CommonArguments.h>

#include <nn/applet/applet.h>


namespace nn { namespace la {

class CommonArgumentsReader;

/**
 * @brief   ライブラリアプレットの共通引数の受け渡しをおこなうクラスです。
 *          これを利用するアプレットは、 libnn_applet.a, libnn_la.a をリンクする必要があります。
 */
class CommonArgumentsWriter
{
private:
    detail::CommonArgumentsDataLatest  m_CommonArgumentsData;
public:

    static const size_t cExportSize = sizeof(detail::CommonArgumentsHeader) + sizeof(detail::CommonArgumentsDataLatest); //!< Export に必要なバッファサイズです。

    /**
     * @brief   ライブラリアプレットの共通変数を構築するコンストラクタです。
     *
     * @param[in]   laVersionMajor   実行しようとしているライブラリアプレットのメジャーバージョンを指定します。
     * @param[in]   laVersionMinor   実行しようとしているライブラリアプレットのマイナーバージョンを指定します。
     *
     * @details
     *  メジャーバージョンは、市場にリリースされた時点で更新します。
     *  メジャーバージョン違いのプロトコルは必ず後方互換を保障する必要があります。
     *  マイナーバージョンは、開発段階でプロトコルを更新する際に使用しますが、
     *  マイナーバージョン違いに関しては互換性は必ずしも必要はありません。
     *
     */
    explicit CommonArgumentsWriter(uint16_t laVersionMajor, uint16_t laVersionMinor) NN_NOEXCEPT : CommonArgumentsWriter( (laVersionMajor << 16) | laVersionMinor )
    {
    }

    /**
     * @brief   ライブラリアプレットの共通変数を構築するコンストラクタです。
     */
    CommonArgumentsWriter() NN_NOEXCEPT : CommonArgumentsWriter( 0 )
    {
    }

    /**
     * @brief   ライブラリアプレットの共通変数を構築するコンストラクタです。
     *
     * @param[in]   laVersion  majorVersion, minorVersion をまとめたバージョンを指定します。
     *
     * @details
     *   上位16bit をメジャーバージョン、下位16bit をマイナーバージョンとして扱います。
     *
     */
    explicit CommonArgumentsWriter(uint32_t laVersion) NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     *
     * @details
     */
    ~CommonArgumentsWriter() NN_NOEXCEPT;

    /**
     * @brief   実行しようとしているライブラリアプレットのバージョンを設定します。
     *
     * @param[in]   laVersionMajor   実行しようとしているライブラリアプレットのメジャーバージョンを指定します。
     * @param[in]   laVersionMinor   実行しようとしているライブラリアプレットのマイナーバージョンを指定します。
     *
     */
    void SetLaVersion( uint16_t laVersionMajor, uint16_t laVersionMinor) NN_NOEXCEPT;

    /**
    * @brief   実行しようとしているライブラリアプレットのバージョンを設定します。
    *
    * @param[in]   laVersion   majorVersion, minorVersion をまとめたバージョンを指定します。
    *
    */
    void SetLaVersion(uint32_t laVersion) NN_NOEXCEPT;

    /**
     * @brief   ライブラリアプレット起動直後にサウンドを鳴らすかどうかを指定します。
     *
     * @details
     *  この関数を呼んで明示的に指定しない場合、 false が指定されたものとして扱われます。
     */
    void SetPlayStartupSound(bool isPlayStartupSound) NN_NOEXCEPT;


    /**
     * @brief   共通変数を一つのストレージにまとめ、ライブラリアプレットの入力チャンネルに push します。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     *
     * @details
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle libraryAppletHandle) const NN_NOEXCEPT;

    /**
     * @brief   共通変数をストレージの先頭に書き込みます。
     *
     * @param[in]   storageHandle     書き込み先のストレージハンドルです。
     *
     * @return      書き込んだデータのサイズを返します。
     *
     * @details
     */
    size_t WriteToStorage(nn::applet::StorageHandle storageHandle) const NN_NOEXCEPT;


    /**
     * @internal
     *
     * @brief   Export() で共通引数を出力するのに必要なサイズを取得します。
     *
     * @return  Export() で共通引数を出力するのに必要なサイズが返ります。
     *
     * @details
     *  この関数が返す値は、ライブラリの更新で増加することがあることに注意してください。
     */
    size_t GetExportSize() const NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   共通引数をバッファにエクスポートします。
     *
     * @details
     *  この関数は、システムプロセス内で構築した共通引数を、アプリケーションのシムレイヤに引き渡すために使用します。
     */
    void Export(void* buffer, size_t size) const NN_NOEXCEPT;
};

}} // namespace nn::la

