﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレット起動に関する非公開ヘッダ
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Tick.h>

//
// バージョンの運用
//
// ・市場にリリースした後にプロトコルを変更する場合には MajorVersion を１つインクリメントします。
//   この時、その Major バージョンでの最後の MinorVersion のプロトコルだけをその後互換性を保証します。
// ・MinorVersion は開発中に変更を入れる際いつでも更新可能です。

#define NN_DETAIL_LA_CURRENT_COMMON_VERSION_MAJOR               0   // NOLINT(preprocessor/const)
#define NN_DETAIL_LA_CURRENT_COMMON_VERSION_MINOR               1   // NOLINT(preprocessor/const)
#define NN_DETAIL_LA_COMMON_ARGUMENTS_DATA_TYPE(commonVersion)  NN_MACRO_CONCATENATE(detail::CommonArgumentsDataV, commonVersion)


namespace nn { namespace la {

namespace detail {

struct CommonArgumentsHeader
{
    uint32_t commonVersion;
    uint32_t commonSize;

    /**
     * @brief   バージョン情報を uint32_t の形式で取得します。
     *
     * @return  major と minor を含めた共通引数のバージョン情報が返ります。
     */
    uint32_t GetVersion() const NN_NOEXCEPT
    {
        return commonVersion;
    }

    /**
     * @brief   バージョン情報を uint16_t の形式で取得します。
     *
     * @return  major と minor を含めた共通引数のバージョン情報が返ります。
     */
    uint16_t GetMajorVersion() const NN_NOEXCEPT
    {
        return static_cast<uint16_t>( commonVersion >> 16 );
    }

    /**
     * @brief   バージョン情報を uint16_t の形式で取得します。
     *
     * @return  major と minor を含めた共通引数のバージョン情報が返ります。
     */
    uint16_t GetMinorVersion() const NN_NOEXCEPT
    {
        return static_cast<uint16_t>( commonVersion & 0xFFFF );
    }
};

struct CommonArgumentsDataV0
{
    static const uint16_t cMajorVersion = 0; //!< この構造体を使うメジャーバージョン。
    typedef CommonArgumentsDataV0 NextType;  //!< テンプレートで処理できるように、 バージョンアップ時に次のバージョンの型を設定する。

    uint32_t laVersion;            //!< LA 共通パラメータのバージョンです
    int32_t colorIndex;            //!< アプレットのテーマカラーの伝搬用。
    bool    isPlayStartupSound;    //!< アプレットのサウンド伝搬用。
    uint8_t padding_[7];           //!< パディング領域
    mutable uint64_t callerTimeStamp;   //!< アプレット起動の時間計測用。

    /**
     * @brief   呼び出し LA のバージョン情報を設定します。
     *
     * @param[in] major メジャーバージョンを指定します。
     * @param[in] minor マイナーバージョンを指定します。
     */
    void SetLaVersion( uint16_t major, uint16_t minor ) NN_NOEXCEPT
    {
        laVersion = (static_cast<uint32_t>(major) << 16) | minor;
    }

    /**
     * @brief   呼び出し LA のメジャーバージョン情報を取得します。
     *
     * @return メジャーバージョンを返します。
     */
    uint16_t GetLaVersionMajor() const NN_NOEXCEPT
    {
        return static_cast<uint16_t>( laVersion >> 16 );
    }

    /**
     * @brief   呼び出し LA のマイナーバージョン情報を取得します。
     *
     * @return メジャーバージョンを返します。
     */
    uint16_t GetLaVersionMinor() const NN_NOEXCEPT
    {
        return static_cast<uint16_t>( laVersion & 0xFFFF );
    }
};

typedef NN_DETAIL_LA_COMMON_ARGUMENTS_DATA_TYPE(NN_DETAIL_LA_CURRENT_COMMON_VERSION_MAJOR) CommonArgumentsDataLatest;


// 受け渡すパラメータを変更する場合、
//
// ・既存の構造体定義は残したまま、後ろに要素を増やした新しい構造体定義を追加する。
// ・NN_DETAIL_LA_CURRENT_COMMON_VERSION_MAJOR, NN_DETAIL_LA_CURRENT_COMMON_VERSION_MINOR マクロの値を新しいバージョンに合わせる。
// ・CommonArgumentsReader のコンストラクタに、追加した要素の初期化処理を追加する。
//
// ことで、一時的に呼び出し元が先行して新しくなったとしても古いクライアントでは値を使わないだけ、
// また、古い呼び出し元から新しいクライアントを呼び出した場合にも値が存在しないので初期値になるだけになります。
//
// それ以上の変更が必要な場合には、CommonArgumentsReader::UpdateCommonDataToLatest の特殊化を実装してください。
}

}} // namespace nn::la

