﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   I2C ドライバライブラリのサスペンド・レジューム機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>

namespace nn {
namespace i2c {
namespace driver {

//! @name サスペンド・レジューム
//! @{

/**
 * @brief   全バスをサスペンドします。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *
 * @post
 *  全バスをサスペンド状態になります。
 *  ただし、PowerBus として使用されている I2C はサスペンドされません。
 *
 * @details
 *  全バスをサスペンドします。@n
 *  ただし、PowerBus として I2C が使用されているハードウェアの場合、このバスはサスペンドされません。@n
 *  別途、SuspendPowerBuses() を呼び出して Suspend 状態にしてください。
 *  サスペンド後もセッション・バスのオープン状態は維持されますが、ハードウェアは省電力状態に移行し、機能が利用できない状態になります。@n
 *  また、サスペンド状態中は新たなセッションのオープンやオープン済セッションのクローズはできません。@n
 *  すでにサスペンド状態で本関数を呼び出した場合、何もしません。
 */
void SuspendBuses() NN_NOEXCEPT;


/**
 * @brief   PowerBus をサスペンドします。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *
 * @post
 *  PowerBus として使用されている I2C がサスペンド状態になります。
 *
 * @details
 *  PowerBus として I2C が使用されているハードウェアの場合、このバスをサスペンドします@n
 *  サスペンド後もセッション・バスのオープン状態は維持されますが、ハードウェアは省電力状態に移行し、機能が利用できない状態になります。@n
 *  また、サスペンド状態中は新たなセッションのオープンやオープン済セッションのクローズはできません。@n
 *  すでにサスペンド状態で本関数を呼び出した場合、何もしません。
 */
void SuspendPowerBuses() NN_NOEXCEPT;


/**
 * @brief   全バスをレジュームします。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *
 * @post
 *  全バスがサスペンド前の状態になります。
 *
 * @details
 *  全バスをレジュームし、サスペンド前の状態に戻します。@n
 *  ただし、PowerBus として I2C が使用されているハードウェアの場合、このバスはレジュームされません。@n
 *  別途、ResumePowerBuses() を呼び出してレジューム状態にしてください。
 *  サスペンド前の状態で本関数を呼び出した場合、何もしません。
 */
void ResumeBuses() NN_NOEXCEPT;

/**
 * @brief   PowerBus をレジュームします。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *
 * @post
 *  PowerBus として使用されている I2C がレジューム状態になります。
 *
 * @details
 *  PowerBus として I2C が使用されているハードウェアの場合、この Bus をレジュームします。@n
 *  サスペンド前の状態で本関数を呼び出した場合、何もしません。
 */
void ResumePowerBuses() NN_NOEXCEPT;

//! @}

} // dirver
} // i2c
} // nn


