﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_DeviceCode.h>
#include <nn/nn_SdkAssert.h>

#include <nn/dt.h>

#include <nn/i2c/detail/i2c_Log.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace i2c { namespace driver { namespace detail { namespace dt {

class IDeviceCodeNodeParser
{
    NN_DISALLOW_COPY(IDeviceCodeNodeParser);
    NN_DISALLOW_MOVE(IDeviceCodeNodeParser);

private:
    nn::util::IntrusiveListNode m_ListNode;
    friend class nn::util::IntrusiveListMemberNodeTraits<IDeviceCodeNodeParser, &IDeviceCodeNodeParser::m_ListNode>;

public:
    typedef nn::util::IntrusiveList<
        IDeviceCodeNodeParser, nn::util::IntrusiveListMemberNodeTraits<IDeviceCodeNodeParser, &IDeviceCodeNodeParser::m_ListNode>
    > List;

public:
    template<size_t N>
    IDeviceCodeNodeParser(const char* const(&compatibleNames)[N]) NN_NOEXCEPT
        : m_ppCompatibleNames(&compatibleNames[0])
        , m_CompatibleNameCount(N)
    {}

    virtual ~IDeviceCodeNodeParser() NN_NOEXCEPT {}

    bool IsCompatible(const char* pValueInProperty) const NN_NOEXCEPT;

    virtual nn::Result OnCompatibleNodeBegin(nn::dt::Node node) NN_NOEXCEPT
    {
        NN_UNUSED(node);
        NN_RESULT_SUCCESS;
    }

    virtual void OnCompatibleNodeEnd(nn::dt::Node node) NN_NOEXCEPT
    {
        NN_UNUSED(node);
    }

    virtual nn::Result OnDeviceCodeFound(nn::DeviceCode deviceCode, nn::dt::Node node) NN_NOEXCEPT = 0;

private:
    const char* const* m_ppCompatibleNames;
    const int m_CompatibleNameCount;
};

class DeviceCodeNodeRootParser
{
    NN_DISALLOW_COPY(DeviceCodeNodeRootParser);
    NN_DISALLOW_MOVE(DeviceCodeNodeRootParser);

public:
    //! サブシステムタグのプロパティ名 ("subsys-<subsystem>") を格納するバッファサイズ
    static constexpr int MaxSubsystemTagNameBytes = 16;

    //! compatible プロパティの最大長
    static constexpr int MaxCompatibleBytes = 128;

public:
    explicit DeviceCodeNodeRootParser(const char* pSubsystemTag) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS(strnlen(pSubsystemTag, NN_ARRAY_SIZE(m_SubsystemTag)), NN_ARRAY_SIZE(m_SubsystemTag));
        strncpy(m_SubsystemTag, pSubsystemTag, NN_ARRAY_SIZE(m_SubsystemTag));

        m_ParserList.clear();
    }

    void RegisterParser(IDeviceCodeNodeParser* pParser) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pParser);

        std::lock_guard<decltype(m_ParserListLock)> lock(m_ParserListLock);
        m_ParserList.push_back(*pParser);
    }

    void UnregisterParser(IDeviceCodeNodeParser* pParser) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pParser);

        std::lock_guard<decltype(m_ParserListLock)> lock(m_ParserListLock);
        m_ParserList.erase(m_ParserList.iterator_to(*pParser));
    }

    nn::Result ParseAll() NN_NOEXCEPT;

private:
    // ノードがパース対象であるかをチェック
    nn::Result CheckSubsystemTag(bool* pOutIsTagMatched, nn::dt::Node node) NN_NOEXCEPT;

    // ノードの compatible プロパティを読み、登録済パーサから対応するものを見つける
    nn::Result FindCompatibleParser(IDeviceCodeNodeParser** ppOutParser, nn::dt::Node node) NN_NOEXCEPT;

    // デバイスコードノードを一つパースする
    // 一つのノードに複数のデバイスコードが含まれる
    nn::Result ParseSingleNode(nn::dt::Node node) NN_NOEXCEPT;

private:
    char m_SubsystemTag[MaxSubsystemTagNameBytes];
    IDeviceCodeNodeParser::List m_ParserList;
    nn::os::SdkMutex m_ParserListLock;
};

}}}}} // nn::i2c::driver::detail::dt
