﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/util/util_BitFlagSet.h>

namespace nn { namespace hid { namespace detail {

/**
 * @brief       汎用的なコントローラーのデジタルボタンの定義です。
 * @details     内部実装やデバッグ限定
 *              外部公開禁止
 */
struct PadButton
{
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<0> A;             //!< コントローラーの a ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<1> B;             //!< コントローラーの b ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<2> X;             //!< コントローラーの x ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<3> Y;             //!< コントローラーの y ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<4> StickL;        //!< コントローラーの L スティックボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<5> StickR;        //!< コントローラーの R スティックボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<6> L;             //!< コントローラーの L ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<7> R;             //!< コントローラーの R ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<8> ZL;            //!< コントローラーの ZL ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<9> ZR;            //!< コントローラーの ZR ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<10> Start;        //!< コントローラーの Start ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<11> Select;       //!< コントローラーの Select ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<12> Left;         //!< コントローラーの 方向ボタン 左
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<13> Up;           //!< コントローラーの 方向ボタン 上
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<14> Right;        //!< コントローラーの 方向ボタン 右
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<15> Down;         //!< コントローラーの 方向ボタン 下
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<16> SL;           //!< コントローラーの SL ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<17> SR;           //!< コントローラーの SR ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<18> Home;         //!< コントローラーの Home ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<19> Shot;         //!< コントローラーの Shot ボタン
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<20> StickLLeft;   //!< 左スティックによる十字ボタンエミュレーション 左
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<21> StickLUp;     //!< 左スティックによる十字ボタンエミュレーション 上
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<22> StickLRight;  //!< 左スティックによる十字ボタンエミュレーション 右
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<23> StickLDown;   //!< 左スティックによる十字ボタンエミュレーション 下
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<24> StickRLeft;   //!< 右スティックによる十字ボタンエミュレーション 左
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<25> StickRUp;     //!< 右スティックによる十字ボタンエミュレーション 上
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<26> StickRRight;  //!< 右スティックによる十字ボタンエミュレーション 右
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<27> StickRDown;   //!< 右スティックによる十字ボタンエミュレーション 下
    typedef ::nn::util::BitFlagSet<32, PadButton>::Flag<28> Palma;        //!< Palma 専用ボタン
};


/**
 * @brief       汎用的なコントローラーのデジタルボタンの集合を扱う型です。
 * @details     内部実装やデバッグ限定
 *              外部公開禁止
 */
typedef ::nn::util::BitFlagSet<32, PadButton> PadButtonSet;

/**
 * @brief       ドライバがパースしたアナログスティックの入力状態を表す構造体です。ライブラリ内部での正規化などは行われていません。
 */
struct AnalogStickDriverState
{
     int32_t x;  //!< アナログスティックの x 軸座標
     int32_t y;  //!< アナログスティックの y 軸座標
};


/**
 * @brief       ドライバなどから取得された生のデジタルボタンやの左右のアナログスティックを備えたコントローラーの定義です。
 * @details     内部実装やデバッグ限定
 *              外部公開禁止
 */
struct PadDriverState
{
    int64_t sampleNumber;                //!< サンプル番号
    PadButtonSet buttons;                //!< デジタルボタンの状態です。
    NN_PADDING4;
    AnalogStickDriverState analogStickL; //!< 左アナログスティックの状態です。
    AnalogStickDriverState analogStickR; //!< 右アナログスティックの状態です。
};

}}} // namespace nn::hid::detail
