﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Gpio ドライバライブラリのサスペンド・レジューム機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>

namespace nn {
namespace gpio {
namespace driver {

//! @name サスペンド・レジューム
//! @{

/**
 * @brief   Gpio のサスペンド準備を行います。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *
 * @post
 *  Gpio がサスペンドできる状態になるもしくは SuspendLow() が呼び出せる状態になります。
 *
 * @details
 *  Gpio のサスペンド準備を行います。@n
 *  どのような処理を行うかは specs / hardware によって異なります。@n
 *  specs / hardwere によっては事後に SuspendLow() を呼ぶ必要があります。@n
 */
void Suspend() NN_NOEXCEPT;

/**
* @brief   低レイヤの Gpio のサスペンド準備を行います。
*
* @pre
*  ライブラリは初期化済の状態である必要があります。
*  Suspend() が呼ばれている必要があります。
*
* @post
*  Gpio のサスペンド準備が完了し、サスペンドできる状態になります。
*
* @details
*  Gpio のサスペンド準備を行います。@n
*  どのような処理を行うかは specs / hardware によって異なります。@n
*  specs / hardwere によっては本関数の呼び出しは必要ありません@n
*/
void SuspendLow() NN_NOEXCEPT;

/**
 * @brief   Gpio のレジューム準備を行います。
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。
 *  specs / hardware によっては事前に ResumeLow() が呼ばれている必要があります。
 *
 * @post
 *  Gpio のレジューム処理が完了し、使用可能になります。
 *
 * @details
 *  Gpio のレジューム処理が完了し、使用可能になります。@n
 *  specs / hardwere によっては事後に ResumeLow() を呼ぶ必要があります。@n
 */
void Resume() NN_NOEXCEPT;

/**
* @brief   例レイヤの Gpio のレジューム準備を行います。
*
* @pre
*  ライブラリは初期化済の状態である必要があります。
*
* @post
*  Resume() が呼び出せる状態になります。
*
* @details
*  低レイヤの Gpio のレジューム処理を行います。@n
*  どのような処理を行うかは specs / hardware によって異なります。@n
*  specs / hardwere によっては本関数の呼び出しは必要ありません@n
*/
void ResumeLow() NN_NOEXCEPT;

//! @}

} // dirver
} // gpio
} // nn
