﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバの開発用途限定のパッドアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/gpio/detail/gpio_Gen.h>
#include <nn/gpio/driver/gpio_PadAccessor.h>

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
#include <nn/nn_DeviceCode.h>
#endif

namespace nn {
namespace gpio {
namespace driver {

//! @name 初期化と終了処理
//! @{

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
/**
 * @brief 開発目的で使用するための特定パッドを直接指定して GPIO パッドとのセッションをオープンする API です。
 *
 * @param[out]  pOutSession            GPIO パッドのセッション用パラメータ
 * @param[in]   padNumber              アクセス対象の GPIO パッドの番号
 * @pre         指定するパッドが GPIO のパッドとして設定されている必要があります。
 * @post        pOutSession に必要な情報が入力され、指定された番号の GPIO パッドが使用可能な状態になります。
 * @details     指定した GPIO パッドとのセッションをオープンします。アクセスする GPIO パッドごとにセッションをオープンをしてください。@n
 *              本 API は開発用途限定の API となります。製品で使用する場合は OpenSession を使用してください。
 *              padNumber がボード上のどのピンと対応するかは別資料で提供します。
 */
void OpenSessionForDev(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT;
#endif

/**
* @brief 指定した WakeEventActiveFlag を強制的に立てます。
*
* @param[in]   name          GPIO パッド名
* @param[in]   isEnabled     WakeEventActiveFlag を強制的に立てるか否か
* @pre         ライブラリが初期化済みである。
* @post        次回の wake 時以降、 WakeEventActiveFlag が強制的に立った状態になります。
* @details     次回の wake 時以降、 指定した GPIO パッド の WakeBitFlag が強制的に立った状態になります@n
*              WakeBitFlag はスリープ時にリセットされますが、本関数を使用して設定したフラグは isEnabled に false を設定するまで立ったままの状態となります@n
*              本関数は OpenSession することなく使用可能です。@n
*              本関数は デバッグ・テスト用途専用です。製品では使用しないでください。@n
*/
#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
nn::Result SetWakeEventActiveFlagSetForDebug(nn::DeviceCode name, bool isEnabled) NN_NOEXCEPT;
#endif
#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
void SetWakeEventActiveFlagSetForDebug(GpioPadName name, bool isEnabled) NN_NOEXCEPT;
#endif

// TODO: nn::gpio::WakePinDebugMode に統合
/**
* @brief Wake pin のデバッグ設定です。
*
* @details SetWakePinDebugMode() に渡すことで各種デバッグ用の wake pin 状態に設定します
*/
enum WakePinDebugMode
{
    WakePinDebugMode_AutoImmediateWake = 1, //! 自動即起床モード
    WakePinDebugMode_NoWake            = 2, //! 無効モード（ただし電源ボタン押しのみは有効）
};

/**
* @brief wake pin のデバッグモードを設定します。
*
* @param[in]   mode     wake pin のデバッグモード
* @pre         ライブラリが初期化済みである。
* @post        wake pin 設定が指定されたデバッグモード用に上書きされます。
* @details     wake pin の状態を指定されたデバッグモードに設定します。@n
*              本関数は OpenSession することなく使用可能です。@n
*              本関数は デバッグ・テスト用途専用です。それ以外の目的に使用しないでください。
*/
void SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT;


//! @}
// ~初期化と終了処理

} // driver
} // gpio
} // nn


