﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバのパッドアクセス機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/gpio/detail/gpio_Gen.h>
#include <nn/gpio/gpio_Type.h>
#include <nn/gpio/gpio_Result.h>

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
#include <nn/nn_DeviceCode.h>
#endif

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
#include <nn/gpio/gpio_PadMap.h>
#endif

namespace nn {
namespace gpio {
namespace driver {

namespace detail {
    const size_t GpioPadSessionSize = 96;
    const size_t GpioPadSessionAlign = 8;
    struct NN_ALIGNAS(GpioPadSessionAlign) GpioPadSessionImplPadded;
}

/**
* @brief GPIO パッドのセッション用パラメータです。
*
* @details GPIO パッドのセッション用パラメータです。内部の変数に直接アクセスしないでください。
*/
struct GpioPadSession
{
    nn::util::TypedStorage<detail::GpioPadSessionImplPadded, detail::GpioPadSessionSize, detail::GpioPadSessionAlign> _impl;
};


//! @name 初期化と終了処理
//! @{

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
/**
 * @brief 指定した GPIO パッドとのセッションをオープンします。すなわちスレーブデバイスとの通信を可能な状態にします。
 *
 * @param[out]  pOutSession             GPIO パッドのセッション用パラメータ
 * @param[in]   pad                     アクセス対象の GPIO パッドのデバイスコード
 * @return      処理の結果を返します。
 * @retresult
 *      @handleresult{nn::ddsf::PermissionDenied, 指定した GPIO パッドへのアクセス権限がありません。}
 *      @handleresult{nn::gpio::ResultDeviceNotFound, 指定した GPIO パッドが見つかりませんでした。}
 *      @handleresult{nn::gpio::ResultAlreadyOpen, 指定した GPIO パッドはすでにオープンされています。}
 * @endretresult
 * @post        pOutSession に必要な情報が入力され、指定された GPIO パッドが使用可能な状態になります。
 * @details     指定した GPIO パッドとのセッションをオープンします。アクセスする GPIO パッドごとにセッションをオープンをしてください。
 *              GPIO ライブラリで対応している GPIO パッドかどうかは ASSERT でチェックされます。
 */
nn::Result OpenSession(GpioPadSession* pOutSession, nn::DeviceCode pad) NN_NOEXCEPT;
#endif
#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
/**
 * @brief 指定した GPIO パッドとのセッションをオープンします。すなわちスレーブデバイスとの通信を可能な状態にします。
 *
 * @param[out]  pOutSession            GPIO パッドのセッション用パラメータ
 * @param[in]   pad                   アクセス対象の GPIO パッドの番号
 * @post        pOutSession に必要な情報が入力され、指定された GPIO パッドが使用可能な状態になります。
 * @details     指定した GPIO パッドとのセッションをオープンします。アクセスする GPIO パッドごとにセッションをオープンをしてください。
 *              GPIO ライブラリで対応している GPIO パッドかどうかは ASSERT でチェックされます。
 */
void OpenSession(GpioPadSession* pOutSession, GpioPadName pad) NN_NOEXCEPT;
#endif

/**
 * @brief 指定した GPIO パッドとのセッションをクローズします。
 *
 * @param[in]   pSession       GPIO パッドのセッション用パラメータ
 * @pre         紐付いている SystemEvent がある場合は待機しているスレッドがいない状態で実行してください。
 * @post        pSession で指定された GPIO パッドとの通信を終了します。
 * @details     pSession で指定された GPIO パッドとの通信を終了します。紐付られていたイベントは破棄され、紐付解除状態となります。
 */
void CloseSession(GpioPadSession* pSession) NN_NOEXCEPT;
//! @}
// ~初期化と終了処理

//! @name GPIO パッドに対する設定
//! @{
/**
 * @brief       GPIO の入出力方向を設定します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   direction        session で指定された GPIO パッドの入出力方向を設定します。
 * @pre         指定したセッションはオープン状態である必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの入出力方向を設定します。
 */
void SetDirection(GpioPadSession* pSession, Direction direction) NN_NOEXCEPT;

/**
 * @brief       GPIO パッドの入出力方向を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの入出力方向を返します。
 * @pre         指定したセッションはオープン状態である必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの入出力方向を取得します。
 */
Direction GetDirection(GpioPadSession* pSession) NN_NOEXCEPT;

/**
 * @brief       GPIO パッドの割り込みモードを設定します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   mode             session で指定された GPIO パッドの割り込みモードを設定します。
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 *              また、サポートされていない InterruptMode を指定した際も ASSERT で落ちます。
 * @details     pSession で指定された GPIO パッドの割り込みモードを設定します。
 */
void SetInterruptMode(GpioPadSession* pSession, InterruptMode mode) NN_NOEXCEPT;

/**
 * @brief       GPIO パッドの割り込みモードを取得します。
 *　
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの割り込みモードを返します。
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの割り込みモードを取得します。
 */
InterruptMode GetInterruptMode(GpioPadSession* pSession) NN_NOEXCEPT;

/**
 * @brief       GPIO パッドの割り込み許可状態を設定します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   enable      gpioPad で指定された GPIO パッドの割り込み許可状態を設定します。
 *                          false：禁止
 *                          true：許可
 * @pre         設定対象の GPIO パッドを指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの割り込み許可状態を設定します。
 */
void SetInterruptEnable(GpioPadSession* pSession, bool enable) NN_NOEXCEPT;

/**
 * @brief       GPIO パッドの割り込み許可状態を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの割り込み許可状態を返します。
 * @retval      false    禁止
 * @retval      true     許可
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの割り込み許可状態を取得します。
 */
bool GetInterruptEnable(GpioPadSession* pSession) NN_NOEXCEPT;


/**
 * @brief       GPIO パッドの割り込み検出ステータスを取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      pSession で指定された GPIO パッドの割り込み検出ステータスを返します。
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの割り込み検出ステータスを取得します。
 */
InterruptStatus GetInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT;


/**
 * @brief       GPIO パッドの割り込み検出ステータスをクリアします。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された GPIO パッドの割り込み検出ステータスをクリアします。
 */
void ClearInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT;

/**
 * @brief チャタリング防止機能の設定状態を切り替えます。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   isEnable          チャタリング防止機能の使用の有無
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @post        チャタリング防止機能の設定状態が切り替わります。
 * @details     pSession で指定されたピンのチャタリング防止機能の設定状態を切り替えます。
 */
void SetDebounceEnabled(GpioPadSession* pSession, bool isEnable) NN_NOEXCEPT;

/**
 * @brief チャタリング防止機能の設定状態を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      チャタリング防止状態を返します。
 * @retval      true        チャタリング防止機能が設定されています。
 * @retval      false       チャタリング防止機能が設定されていません。
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     チャタリング防止機能の設定状態を取得します。
 */
bool GetDebounceEnabled(GpioPadSession* pSession) NN_NOEXCEPT;

/**
 * @brief チャタリングとして判定する時間を設定します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @param[in]   msecTime          チャタリングとして判定する時間 (mSec)
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @post        チャタリングの判定時間が設定されます。
 * @details     pSession で指定された対象のチャタリング防止機能の判定時間を設定します。@n
 *              msecTime 以内の変化は割り込みが発生しなくなります。@n
 *              本機能はボードによっては他のピンの設定も書き換える可能性があります。@n
 *              許可されたドライバのみ使用するようにしてください。
 *
 * @platformbegin{nx}
 *  - msecTime の範囲は 0 以上 128 以下です。
 *  - 本機能は 8 ピンまとめてチャタリング時間の設定を行います。
 *  - 本 API を使用する場合は担当窓口までお問い合わせください。
 *
 * @platformend
 */
void SetDebounceTime(GpioPadSession* pSession, int msecTime) NN_NOEXCEPT;

/**
 * @brief チャタリングとして判定する時間を取得します。
 *
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
 * @return      チャタリングとして判定する時間を返します。単位は mSec です。
 * @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
 *              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @details     pSession で指定された対象のチャタリング防止の判定時間を取得します。
 */
int GetDebounceTime(GpioPadSession* pSession) NN_NOEXCEPT;
//! @}
// ~GPIO パッドに対する設定


//! @name GPIO パッドのコントロール
//! @{
/**
* @brief GPIO パッドの状態(H/L)を確認します。
*
 * @param[in]   pSession          GPIO パッドのセッション用パラメータ
* @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
*              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
* @return      pSession で指定した GPIO パッドの状態を返します。
* @details     pSession で指定された GPIO パッド状態(H/L)を確認します。
*/
GpioValue GetValue(GpioPadSession* pSession) NN_NOEXCEPT;

/**
* @brief GPIO パッドの状態(H/L)を設定します。
*
* @param[in]   pSession          GPIO パッドのセッション用パラメータ
* @param[in]   value            pSession で指定した GPIO パッドの状態を設定します。
* @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
*              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
* @details     pSession で指定された GPIO パッドの状態(H/L)を設定します。
*/
void SetValue(GpioPadSession* pSession, GpioValue value) NN_NOEXCEPT;

/**
* @brief スリープ中の GPIO パッドの状態(H/L)を設定します。
*
* @param[in]   pSession          GPIO パッドのセッション用パラメータ
* @param[in]   value            pSession で指定した GPIO パッドのスリープ中の状態を設定します。
* @pre         設定対象の GPIO パッド を指定して初期化されている必要があります。
*              (オープン状態かどうか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
* @details     pSession で指定された GPIO パッドのスリープ中の状態(H/L)を設定します。@n
*              スリープ前に本関数を呼び出すと、指定された値が内部に保存され、スリープの直前に設定されます。@n
*              一度本関数を呼び出すと、毎回のスリープの度に保存した値に書き換えることにご注意ください。@n
*              本関数を呼び出していない場合は、各ドライバが設定したままの状態となります。
*/
void SetValueForSleepState(GpioPadSession* pSession, GpioValue value) NN_NOEXCEPT;
//! @}
// ~GPIO パッドのコントロール


//! @name GPIO パッドの割り込みとイベントの紐付けと紐付解除
//! @{
/**
 * @brief        GPIO パッドの割り込みとイベントを紐付けます。
 *
 * @param[in]    pEvent           割込みが発生した際にシグナル化するイベントを指定します。
 * @param[in]    pSession          GPIO パッドのセッション用パラメータ
 * @return       処理の結果を返します。
 * @retval       ResultSuccess        成功しました。
 * @retval       ResultAlreadyBound   session で指定した GPIO パッドがすでに特定イベントと紐付けされています。
 * @pre          pSession で指定した GPIO パッドが他のイベントと紐付されていない必要があります。@n
 *               また、pEvent は本関数内で初期化されるため、初期化せずに本関数へ渡してください。@n
 *               (初期化されていない pEvent が渡されているか、また、アクセスが許可されている GPIO パッドかどうかは ASSERT　でチェックされます。)
 * @post         pSession で指定した GPIO パッドと pEvent が紐付けられた状態となります。(pEvent は本関数内で初期化されます。)@n
 *               指定された GPIO パッドに割り込みが発生すると pEvent がシグナルされます。@n
 *               シグナル後は紐付けられた GPIO パッドに対する割り込みが割り込み禁止状態となる点にご注意ください。 @n
 *               再度割り込みを受けたい場合は SetInterruptEnable() を呼んで割り込み許可状態にする必要があります。@n
 * @details      pSession で指定した GPIO パッド に割り込みが発生した場合に pEvent で指定したイベントをシグナル状態にセットします。@n
 *               シグナル後は紐付けられた GPIO パッドの割り込みが割り込み禁止状態となります。@n
 *               再度割り込みを受けたい場合は SetInterruptEnable() を呼んで割り込み許可状態にする要があります。@n
 *               1つの GPIO パッドに対して紐付できるイベントは一つまでです。
 */
Result BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT;


/**
 * @brief        GPIO パッドの割り込みとイベントの紐付を解除します。
 *
 * @param[in]    pSession         GPIO パッドのセッション用パラメータ
 *
 * @pre          BindInterrupt で SystemEvent が pSession と紐付けされている必要があります。@n
 *               pSession に紐付いている SystemEvent があるかどうかは ASSERT でチェックされます。@n
 *               紐付いている SystemEvent を待機しているスレッドがいない状態で実行してください。
 * @post         pSession と紐付けられた SystemEvent が紐付解除状態となります。@n
 *               また、紐付いていた SystemEvent は破棄され、未初期化状態となります。
 * @details      pSession と紐付けられた SystemEvent が紐付解除状態となります。
 */
void UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT;
//! @}
// ~割り込みのバインドとアンバインド

//! @name 起床要因の取得
//! @{

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
/**
 * @brief        特定 GPIO パッドが起床要因として使われたかどうかを取得します。
 *
 * @param[in]    name         アクセス対象の GPIO パッドの名前
 *
 * @return      処理の結果を返します。
 * @retresult
 *      @handleresult{nn::gpio::ResultDeviceNotFound, 指定した GPIO パッドが見つかりませんでした。}
 * @endretresult
 *
 * @pre          ライブラリが初期化済みである。
 *
 * @details      name で指定された GPIO パッドがスリープからの復帰に使用されたかどうかを取得します。
 *               本関数は OpenSession することなく使用可能です。@n
 *               スリープからの復帰ごとに最新の状態に更新され、前回のスリープ時の起床要因は取れなくなります。@n
 */
nn::Result IsWakeEventActive(bool* pOutIsActive, nn::DeviceCode name) NN_NOEXCEPT;
#endif
#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
/**
 * @brief        特定 GPIO パッドが起床要因として使われたかどうかを取得します。
 *
 * @param[in]    name         アクセス対象の GPIO パッドの名前
 *
 * @pre          ライブラリが初期化済みである。
 *
 * @details      name で指定された GPIO パッドがスリープからの復帰に使用されたかどうかを取得します。
 *               本関数は OpenSession することなく使用可能です。@n
 *               スリープからの復帰ごとに最新の状態に更新され、前回のスリープ時の起床要因は取れなくなります。@n
 */
bool IsWakeEventActive(GpioPadName name) NN_NOEXCEPT;
#endif

#if defined(NN_DETAIL_GPIO_ENABLE_GEN_2)
// TODO: 定義番号に依存している仕様なので、要検討
#endif
#if defined(NN_DETAIL_GPIO_ENABLE_GEN_1)
/**
* @brief        どの GPIO パッドが起床要因として使われたかを WakeBitFlag を返り値として取得します。
*
* @return       WakeBitFlag の形で起床要因を返します。
*
* @pre          ライブラリが初期化済みである。
*
* @details      どの GPIO パッドがスリープからの復帰に使用されたか WakeBitFlag の形で取得します。
*               本関数は OpenSession することなく使用可能です。@n
*               スリープからの復帰ごとに最新の状態に更新され、前回のスリープ時の起床要因は取れなくなります。@n
*               WakeBitFlag の 各 bit は GpioPadName の定義番号と対応します。
*/
WakeBitFlag GetWakeEventActiveFlagSet() NN_NOEXCEPT;
#endif

//! @}
// ~起床要因の取得

} // driver
} // gpio
} // nn


