﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

// include/rapidjson/msinttypes が削除されているため、事前定義が必要。
#define RAPIDJSON_NO_INT64DEFINE

#define RAPIDJSON_NAMESPACE         nne::rapidjson::friends
#define RAPIDJSON_NAMESPACE_BEGIN   namespace nne { namespace rapidjson { namespace friends {
#define RAPIDJSON_NAMESPACE_END     }}}

#define RAPIDJSON_ASSERT(x) NN_SDK_ASSERT(x)

#define RAPIDJSON_HAS_CXX11_RVALUE_REFS 1 // NOLINT(preprocessor/const)
#define RAPIDJSON_HAS_CXX11_TYPETRAITS  1 // NOLINT(preprocessor/const)

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif

#include <rapidjson/reader.h>
#include <rapidjson/writer.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

namespace nn { namespace friends { namespace detail { namespace service { namespace json {

/*!
    @brief      メモリの動的確保を禁止したアロケータです。
*/
class NullAllocator
{
public:
    static const bool kNeedFree = false;

public:
    // Do not call.
    void* Malloc(size_t) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false, "Memory allocation is not allowed.");
        return nullptr;
    }

    // Do not call.
    void* Realloc(void*, size_t, size_t) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false, "Memory re-allocation is not allowed.");
        return nullptr;
    }

    static void Free(void*) NN_NOEXCEPT
    {
    }
};

/*!
    @brief      固定長のメモリを返すアロケータです。
*/
class FixedSizeAllocator
{
public:
    static const bool kNeedFree = false;

public:
    /*!
        @brief      コンストラクタです。
    */
    FixedSizeAllocator() NN_NOEXCEPT : m_IsAllocated(false)
    {
    }

    void* Malloc(size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(size <= sizeof (m_Buffer));
        NN_UNUSED(size);

        NN_SDK_ASSERT(!m_IsAllocated);

        return m_Buffer;
    }

    void* Realloc(void* p, size_t oldSize, size_t newSize) NN_NOEXCEPT
    {
        NN_UNUSED(oldSize);

        if (p)
        {
            NN_SDK_ASSERT(p == m_Buffer);
            m_IsAllocated = false;
        }
        return Malloc(newSize);
    }

    static void Free(void*) NN_NOEXCEPT
    {
    }

private:
    //
    char m_Buffer[256];
    //
    bool m_IsAllocated;
};

/*!
    @brief      メモリ上にデータを書き込む出力ストリームです。
*/
class JsonMemoryOutputStream
{
public:
    typedef char Ch;

public:
    /*!
        @brief      コンストラクタです。
    */
    JsonMemoryOutputStream() NN_NOEXCEPT :
        m_Memory(nullptr),
        m_MemorySize(0),
        m_FilledSize(0),
        m_IsOpended(false)
    {
    }

    /*!
        @brief      出力ストリームを開きます。

        @param[in]  memory  出力データ。
        @param[in]  size    出力データのサイズ。

        @pre
            - memory != nullptr
            - size > 0

        @return     処理結果。
    */
    void Open(Ch* memory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(memory);
        NN_SDK_REQUIRES(size > 0);

        NN_SDK_ASSERT(!m_IsOpended);

        m_Memory = memory;
        m_MemorySize = size;
        m_FilledSize = 0;

        m_IsOpended = true;
    }

    /*!
        @brief      出力ストリームを閉じます。
    */
    void Close() NN_NOEXCEPT
    {
        m_IsOpended = false;
    }

    /*!
        @brief      データの入力を開始します。
    */
    void PutBegin() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsOpended);

        m_FilledSize = 0;
    }

    /*!
        @brief      データを入力します。

        @param[in]  c   文字。
    */
    void Put(Ch c) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsOpended);
        NN_SDK_ASSERT(m_FilledSize < m_MemorySize);

        m_Memory[m_FilledSize++] = c;
    }

    /*!
        @brief      データの入力を終了します。

        @return     入力した文字数。
    */
    size_t PutEnd() NN_NOEXCEPT
    {
        if (m_FilledSize < m_MemorySize)
        {
            m_Memory[m_FilledSize] = '\0';
        }

        return m_FilledSize;
    }

    void Flush() NN_NOEXCEPT
    {
    }

private:
    //
    Ch* m_Memory;
    size_t m_MemorySize;
    //
    size_t m_FilledSize;
    //
    bool m_IsOpended;
};

/*!
    @brief      ライブラリで利用する JsonReader オブジェクトです。
*/
typedef RAPIDJSON_NAMESPACE::GenericReader
    <RAPIDJSON_NAMESPACE::UTF8<>, RAPIDJSON_NAMESPACE::UTF8<>, NullAllocator> JsonReader;

/*!
    @brief      ライブラリで利用する JsonWriter オブジェクトです。
*/
typedef RAPIDJSON_NAMESPACE::Writer
    <JsonMemoryOutputStream, RAPIDJSON_NAMESPACE::UTF8<>, RAPIDJSON_NAMESPACE::UTF8<>, FixedSizeAllocator> JsonWriter;

/*!
    @brief      整数型です。

    @details
                任意の整数型が格納されます。
*/
struct AnyInteger
{
    union
    {
        int64_t s;  //!< 64 ビット符号付き整数型。
        uint64_t u; //!< 64 ビット符号なし整数型。
    };
    bool isSigned;
};

}}}}}
