﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      プレイログの管理モジュールです。
*/
class PlayLogManager
{
private:
    NN_DISALLOW_COPY(PlayLogManager);
    NN_DISALLOW_MOVE(PlayLogManager);

private:
    /*!
        @brief      コンストラクタです。
    */
    PlayLogManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static PlayLogManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(PlayLogManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      プレイログリストを取得します。

        @param[out] outCount    取得した数。
        @param[out] outPlayLogs プレイログリスト。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  count       プレイログリストの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outPlayLogs != nullptr
            - count > 0
    */
    nn::Result GetPlayLogList(int* outCount, PlayLog* outPlayLogs,
        const nn::account::Uid& uid, int count) NN_NOEXCEPT;

    /*!
        @brief      送信すべきプレイログを保持しているかどうかを確認します。

        @param[out] out 送信すべきプレイログを保持しているかどうか。
        @param[in]  uid ユーザーアカウント。

        @return     処理結果。

        @pre
            - out != nullptr
    */
    nn::Result HasPlayLog(bool* out, const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      アップロード済みのプレイログを削除します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  playLogs    プレイログリスト。
        @param[in]  count       プレイログリストの要素数。

        @return     処理結果。

        @details
                    指定したプレイログとすべてのパラメータが一致するプレイログを削除します。@n
                    アプリケーション ID が一致しても、他のパラメータに更新がある場合、送信処理中に新たに追加されたものであると判定します。
    */
    nn::Result RemoveUploadedPlayLog(const nn::account::Uid& uid, const PlayLog* playLogs, int count) NN_NOEXCEPT;

    /*!
        @brief      プレイログをクリアします。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。
    */
    nn::Result Clear(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      メモリキャッシュを無効化します。
    */
    void Invalidate() NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの追加イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの削除イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの Open イベントを通知します。

        @param[in]  uid     ユーザーアカウント。
        @param[in]  appInfo ユーザーを Open したアプリケーションのアプリケーション情報。
    */
    void NotifyUserOpened(const nn::account::Uid& uid, const ApplicationInfo& appInfo) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの Close イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserClosed(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      スリープを通知します。
    */
    void NotifySleep() NN_NOEXCEPT;

    /*!
        @brief      スリープ復帰を通知します。
    */
    void NotifySleepAwaked() NN_NOEXCEPT;

private:
    //
    struct Record
    {
        PlayLog playLog;
    };

    //
    struct CurrentApplicationInfo
    {
        nn::account::Uid uid;
        ApplicationInfo appInfo;
    };

    NN_STATIC_ASSERT(sizeof (Record) == 40);

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_CurrentUid;
    //
    Record m_Records[PlayLogCountMax];
    int m_Count;
    //
    CurrentApplicationInfo m_CurrentApplications[nn::account::UserCountMax];

private:
    //
    nn::Result Load(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    nn::Result Save() NN_NOEXCEPT;
    //
    int SearchUser(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    nn::Result Update(const nn::account::Uid& uid, const ApplicationInfo& appInfo) NN_NOEXCEPT;
};

}}}}}
