﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      いっしょに遊んだユーザーの管理モジュールです。
*/
class PlayHistoryManager
{
public:
    /*!
        @brief      いっしょに遊んだユーザーの登録キーです。
    */
    struct PlayHistoryRegistrationKeyData
    {
        // 1
        Bit8 version;
        // 1
        Bit8 platform;
        // 1
        Bit8 hashKeyIndex;
        // 1
        bool hasNetworkServiceAccountId;
        // 1
        bool isLocalPlayed;
        // 3
        NN_PADDING3;
        // 8
        nn::account::NetworkServiceAccountId accountId;
        // 16
        nn::util::Uuid uuid;
        // 32
        Bit8 hash[32];
    };

    NN_STATIC_ASSERT(sizeof (PlayHistoryRegistrationKeyData) == sizeof (PlayHistoryRegistrationKey));

    /*!
        @brief      レコードです。
    */
    struct Record
    {
        // 8
        nn::account::NetworkServiceAccountId accountId;
        // 16
        nn::util::Uuid uuid;
        // 168
        PlayRecord playRecord;
        // 1
        bool hasNetworkServiceAccountId;
        // 1
        bool isLocalPlayed;
        // 6
        NN_PADDING6;
        // 56
        char reserved[56];
    };

    NN_STATIC_ASSERT(sizeof (Record) == 256);

private:
    /*!
        @brief      コンストラクタです。
    */
    PlayHistoryManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static PlayHistoryManager& GetInstance() NN_NOEXCEPT
    {
        static PlayHistoryManager instance;
        return instance;
    }

public:
    /*!
        @brief      いっしょに遊んだユーザーの登録キーを取得します。

        @param[out] outKey          登録キー。
        @param[in]  uid             ユーザーアカウント。
        @param[in]  isLocalPlayed   ローカル通信でいっしょに遊んだかどうか。

        @return     処理結果。

        @pre
            - outKey != nullptr
    */
    nn::Result GetPlayHistoryRegistrationKey(PlayHistoryRegistrationKey* outKey,
        const nn::account::Uid& uid, bool isLocalPlayed) NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーの登録キーを取得します。

        @param[out] outKey          登録キー。
        @param[in]  accountId       ネットワークサービスアカウント ID。
        @param[in]  isLocalPlayed   ローカル通信でいっしょに遊んだかどうか。

        @return     処理結果。

        @pre
            - outKey != nullptr
    */
    nn::Result GetPlayHistoryRegistrationKey(PlayHistoryRegistrationKey* outKey,
        nn::account::NetworkServiceAccountId accountId, bool isLocalPlayed) NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーの登録キーからネットワークサービスアカウント ID を取得します。

        @param[out] outAccountId    ネットワークサービスアカウント ID。
        @param[in]  key             登録キー。

        @return     処理結果。

        @pre
            - outAccountId != nullptr
    */
    nn::Result GetAccountIdFromPlayHistoryRegistrationKey(nn::account::NetworkServiceAccountId* outAccountId,
        const PlayHistoryRegistrationKey& key) NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーを追加します。

        @param[in]  uid     ユーザーアカウント。
        @param[in]  key     登録キー。
        @param[in]  appInfo アプリケーション情報。
        @param[in]  name    相手のアプリ内表示名。
        @param[in]  myName  自分のアプリ内表示名。

        @return     処理結果。

        @details
                    登録しようとしたユーザーがすでに存在する場合、登録順が更新されます。
    */
    nn::Result AddPlayHistory(const nn::account::Uid& uid, const PlayHistoryRegistrationKey& key,
        const ApplicationInfo& appInfo, const InAppScreenName& name, const InAppScreenName& myName) NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーを保存します。

        @return     処理結果。

        @details
                    別のユーザーアカウントのデータを読み込む時や、ユーザーアカウントを Close する時に本 API を呼び出してください。
    */
    nn::Result Save() NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーを削除します。

        @param[in]  uid ユーザーアカウント。

        @return     処理結果。
    */
    nn::Result Delete(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーを取得します。

        @param[out] outCount        取得した数。
        @param[out] outHistories    いっしょに遊んだユーザー。
        @param[in]  uid             ユーザーアカウント。
        @param[in]  offset          オフセット。
        @param[in]  count           いっしょに遊んだユーザーの要素数。

        @return     処理結果。

        @pre
            - outCount != nullptr
            - outHistories != nullptr
    */
    nn::Result GetPlayHistoryList(int* outCount, PlayHistoryImpl* outHistories,
        const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT;

    /*!
        @brief      いっしょに遊んだユーザーの統計情報を取得します。

        @param[out] outStatistics   統計情報。
        @param[in]  uid             ユーザーアカウント。

        @return     処理結果。

        @pre
            - outStatistics != nullptr
    */
    nn::Result GetPlayHistoryStatistics(PlayHistoryStatistics* outStatistics,
        const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      メモリキャッシュを無効化します。
    */
    void Invalidate() NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::account::Uid m_CurrentUid;
    //
    Record m_Records[PlayHistoryCountMax];
    int m_Count;
    //
    bool m_IsDirty;
    //
    PlayHistoryStatistics m_Statistics;

private:
    //
    nn::Result Load(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadHistory(const nn::account::Uid& uid) NN_NOEXCEPT;
    nn::Result LoadHistoryStats(const nn::account::Uid& uid) NN_NOEXCEPT;
    //
    nn::Result SaveHistory() NN_NOEXCEPT;
    nn::Result SaveHistoryStats() NN_NOEXCEPT;
    //
    int SearchSameUser(const PlayHistoryRegistrationKey& key) NN_NOEXCEPT;
    //
    bool VerifyPlayHistoryRegistrationKey(const PlayHistoryRegistrationKey& key) NN_NOEXCEPT;
    bool VerifyRecord(int count) NN_NOEXCEPT;
};

}}}}}
