﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/friends/detail/service/friends_Common.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

/*!
    @brief      フレンドプレゼンスの管理モジュールです。
*/
class FriendPresenceManager
{
private:
    NN_DISALLOW_COPY(FriendPresenceManager);
    NN_DISALLOW_MOVE(FriendPresenceManager);

private:
    /*!
        @brief      コンストラクタです。
    */
    FriendPresenceManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static FriendPresenceManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(FriendPresenceManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      フレンドプレゼンスを取得します。

        @param[out] outPresence フレンドプレゼンス。
        @param[in]  uid         ユーザーアカウント。
        @param[in]  accountId   フレンドのネットワークサービスアカウント ID。

        @pre
            - outPresence != nullptr
    */
    void GetPresence(FriendPresenceImpl* outPresence,
        const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) const NN_NOEXCEPT;

    /*!
        @brief      プレゼンス状態でフィルターした人数を取得します。

        @param[out] outCount        人数。
        @param[in]  uid             ユーザーアカウント。
        @param[in]  statusFilter    フィルター。

        @pre
            - outCount != nullptr
    */
    void GetCount(int* outCount, const nn::account::Uid& uid, PresenceStatusFilter statusFilter) const NN_NOEXCEPT;

    /*!
        @brief      フレンドリストを更新します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  resources   フレンドリスト。
        @param[in]  count       フレンドリストの要素数。

        @pre
            - resources != nullptr

        @details
                    フレンドリストの同期によってフレンドの増減が発生する場合、本関数を呼び出す必要があります。
    */
    void UpdateFriendList(const nn::account::Uid& uid, const FriendResource* resources, int count) NN_NOEXCEPT;

    /*!
        @brief      フレンドプレゼンスを更新します。

        @param[in]  uid         ユーザーアカウント。
        @param[in]  resource    フレンドリソース。

        @details
                    フレンドリソースで指定されたユーザーが存在しない場合、本関数は何も行いません。
    */
    void UpdatePresence(const nn::account::Uid& uid, const FriendResource& resource) NN_NOEXCEPT;

    /*!
        @brief      フレンドプレゼンスを更新します。

        @param[in]  uid             ユーザーアカウント。
        @param[in]  accountId       フレンドのネットワークサービスアカウント ID。
        @param[in]  presence        フレンドプレゼンス。
        @param[in]  creationTime    フレンドプレゼンスの作成時刻。

        @details
                    指定したフレンドのネットワークサービスアカウント ID が存在しない場合、本関数は何も行いません。

                    フレンドプレゼンスの作成時刻が不明な場合、 creationTime.value は 0 に指定してください。
    */
    void UpdatePresence(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId,
        const FriendPresenceImpl& presence, nn::time::PosixTime creationTime) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの追加イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserAdded(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      ユーザーアカウントの削除イベントを通知します。

        @param[in]  uid ユーザーアカウント。
    */
    void NotifyUserRemoved(const nn::account::Uid& uid) NN_NOEXCEPT;

    /*!
        @brief      NPNS の接続状態の変化を通知します。

        @param[in]  isConnected 接続されているかどうか。
    */
    void NotifyNpnsStateChanged(bool isConnected) NN_NOEXCEPT;

private:
    //
    struct Record
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountIds[FriendCountMax];
        FriendPresenceImpl presences[FriendCountMax];
        int32_t count;
    };

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    Record m_Records[nn::account::UserCountMax];
    bool m_IsNetworkConnected;

private:
    //
    int FindUser(const nn::account::Uid& uid) const NN_NOEXCEPT;
    int FindFriend(int userIndex, nn::account::NetworkServiceAccountId accountId) const NN_NOEXCEPT;
};

}}}}}
