﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitFlagSet.h>
#include <string>

namespace nn { namespace fatalsrv {

#define NN_FATALSRV_AARCH64_VALUES(ENTRY) \
    ENTRY(R0,    0)\
    ENTRY(R1,    1)\
    ENTRY(R2,    2)\
    ENTRY(R3,    3)\
    ENTRY(R4,    4)\
    ENTRY(R5,    5)\
    ENTRY(R6,    6)\
    ENTRY(R7,    7)\
    ENTRY(R8,    8)\
    ENTRY(R9,    9)\
    ENTRY(R10,  10)\
    ENTRY(R11,  11)\
    ENTRY(R12,  12)\
    ENTRY(R13,  13)\
    ENTRY(R14,  14)\
    ENTRY(R15,  15)\
    ENTRY(R16,  16)\
    ENTRY(R17,  17)\
    ENTRY(R18,  18)\
    ENTRY(R19,  19)\
    ENTRY(R20,  20)\
    ENTRY(R21,  21)\
    ENTRY(R22,  22)\
    ENTRY(R23,  23)\
    ENTRY(R24,  24)\
    ENTRY(R25,  25)\
    ENTRY(R26,  26)\
    ENTRY(R27,  27)\
    ENTRY(R28,  28)\
    ENTRY(R29,  29)\
    ENTRY(R30,  30)\
    ENTRY(Sp,   31)\
    ENTRY(Pc,   32)\
    ENTRY(Pstate,   33)\
    ENTRY(Afsr0,    34)\
    ENTRY(Afsr1,    35)\
    ENTRY(Esr,      36)\
    ENTRY(Far,      37)\


#define NN_FATALSRV_GENERATE_ENUM(NAME, NUMBER) NAME=NUMBER,
#define NN_FATALSRV_GENERATE_STRING(NAME, NUMBER) #NAME,
    struct Aarch64Context
    {
        static const int MaxBacktraceCount = 32;

        union
        {
            struct
            {
                union
                {
                    struct
                    {
                        Bit64   r[31];      //!< x0～x30 の値が格納されます
                    };
                    struct
                    {
                        Bit64   _padding[30];
                        Bit64   lr;         //!< x30 と等価です。リンクレジスタです
                    };
                };
                Bit64   sp;                 //!< スタックポインタです
                Bit64   pc;                 //!< プログラムカウンタです

                                            // 例外情報
                Bit64   pstate; //!< pstate の値が格納されます。ビット配置は SPSR のフォーマットと同じです。上位32ビットは無効値です。
                Bit64   afsr0;  //!< AFSR0 の値が格納されます。上位32ビットは無効値です。
                Bit64   afsr1;  //!< AFSR1 の値が格納されます。上位32ビットは無効値です。
                Bit64   esr;    //!< ESR の値が格納されます。上位32ビットは無効値です。
                Bit64   far;    //!< FAR の値が格納されます。
            };
            struct
            {
                Bit64 values[38];
            };
        };


        // バックトレース
        Bit64   backtrace[MaxBacktraceCount];

        // プログラムのマップされているアドレス
        Bit64   programMappedAddr;

        // 各項目のセット/非セットを表す
        nn::util::BitFlagSet<64> setFlag;

        int32_t backtraceCount;

        enum ValueNames
        {
            NN_FATALSRV_AARCH64_VALUES(NN_FATALSRV_GENERATE_ENUM)
            ValueNameCount,
        };

        static const char* ValueNameStrings[];

        void Clear() NN_NOEXCEPT
        {
            std::memset(this, 0, sizeof(*this));
        }

        void SetValue(ValueNames target, Bit64 value) NN_NOEXCEPT
        {
            this->values[target] = value;
            setFlag.Set(target, true);
        }

        void SetProgramMappedAddr(Bit64 addr) NN_NOEXCEPT
        {
            this->programMappedAddr = addr;
        }
    };
}} // namespace nn::fatalsrv
