﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ファン制御ライブラリの開発用途限定の制御機能の API 宣言。
 */

#pragma once

#include <nn/nn_Common.h>

#include <nn/fan/fan_Control.h>

namespace nn {
namespace fan {

//! @name 回転速度の直接指定による制御（開発用途限定）
//! @{

/**
 * @brief 回転速度レートで指定可能な最大値です。
 */
const int MaxRotationRate = 255;

/**
 * @brief ファンの回転速度レートを設定します。
 *
 * @param[in]   pController     コントローラ
 * @param[in]   rotationRate    回転速度レート
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *  @a rotationRate は [0 - MaxRotationRate] の整数値である必要があります。
 *
 * @post
 *  ファンの回転が @a rotationRate に対応した速度になります。
 *
 * @details
 *  ファンの回転速度レートを設定します。@n
 *  @a rotationRate が 0 のときファンは停止状態となり、 MaxRotationRate のとき最大速度で回転します。@n
 *  本関数は開発目的での利用を想定しており、製品環境では利用できません。@n
 *  製品環境では @ref SetRotationSpeedLevel() や @ref SetRotationSpeedLevelByDiff() を使用してください。@n
 */
void SetRotationRate(Controller* pController, int rotationRate) NN_NOEXCEPT;

/**
 * @brief ファンの現在の回転速度レートを取得します。
 *
 * @param[in]   pController    コントローラ
 *
 * @return      回転速度レート
 *
 * @pre
 *  ライブラリは初期化済の状態である必要があります。@n
 *  @a pController は有効なコントローラである必要があります。@n
 *
 * @details
 *  ファンの現在の回転速度レートを取得します。@n
 *  本関数は開発目的での利用を想定しており、製品環境では利用できません。@n
 *  製品環境では @ref GetRotationSpeedLevel() を使用してください。@n
 */
int GetRotationRate(Controller* pController) NN_NOEXCEPT;

//! @}
// ~回転速度の直接指定による制御（開発用途限定）

} // fan
} // nn


