﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/es/es_TypesForInner.h>
#include <nn/es/es_ELicenseTypes.h>

namespace nn { namespace es {

/**
*   @brief 内部実装用の ELicense のに関する属性を表すフラグです。
*/
enum class ELicenseInfoForSystemFlags : uint8_t
{
    RecommendsServerInteraction = 1 << 0,   //!< 起動時にネットワークでの再認証を推奨するか
};

constexpr ELicenseInfoForSystemFlags operator&(ELicenseInfoForSystemFlags lhs, ELicenseInfoForSystemFlags rhs)
{
    return static_cast<ELicenseInfoForSystemFlags>(static_cast<std::underlying_type<ELicenseInfoForSystemFlags>::type>(lhs) & static_cast<std::underlying_type<ELicenseInfoForSystemFlags>::type>(rhs));
}

constexpr ELicenseInfoForSystemFlags operator|(ELicenseInfoForSystemFlags lhs, ELicenseInfoForSystemFlags rhs)
{
    return static_cast<ELicenseInfoForSystemFlags>(static_cast<std::underlying_type<ELicenseInfoForSystemFlags>::type>(lhs) | static_cast<std::underlying_type<ELicenseInfoForSystemFlags>::type>(rhs));
}

/**
*   @brief 内部実装用の ELicense の情報を表す構造体です。
*/
struct ELicenseInfoForSystem
{
    ELicenseId                  eLicenseId;
    ELicenseArchiveId           archiveId;
    RightsId                    rightsId;
    TicketId                    ticketId;
    ELicenseOwnerId             ownerNaId;
    ExpireDate                  expireDate;
    AccountId                   ticketOwnerVaId;
    ELicenseScope               scope;
    ELicenseInfoForSystemFlags  flags;
    uint8_t                     padding[2];

    bool IsServerInteractionRecommended() const NN_NOEXCEPT
    {
        return (this->flags & ELicenseInfoForSystemFlags::RecommendsServerInteraction) == ELicenseInfoForSystemFlags::RecommendsServerInteraction;
    }
};
NN_STATIC_ASSERT(sizeof(ELicenseInfoForSystem) == 72);

/**
*   @brief 内部実装用の ELicense と付随する情報を含んだ構造体です。
*/
struct ELicenseInfoForSystemWrapper
{
    /**
    *   @brief 内部実装用の ELicense の情報です。
    */
    ELicenseInfoForSystem info;

    /**
    *   @brief 付随する情報を格納するフラグです。
    */
    ELicenseStatusFlag _flags;

    /**
    *   @brief 拡張のための予約領域です。
    */
    uint8_t _reserved[4];

    /**
    *   @brief 対応するチケットを保有しているかを判定します。
    */
    bool HasTicket() const NN_NOEXCEPT
    {
        return _flags.Test(ELicenseStatusFlag_HasTicket::Index);
    }
};
NN_STATIC_ASSERT(sizeof(ELicenseInfoForSystemWrapper) == 80);

/**
*   @brief ELicenseInfo を取得します。
*/
ELicenseInfo GetELicenseInfo(ELicenseInfoForSystem eLicenseInfoForSystem) NN_NOEXCEPT;

}} // namespace nn::es
