﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
*   @file
*   @brief  使用中権利リストの管理に関する機能
*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/account/account_NintendoAccountTypes.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/es/es_Types.h>
#include <nn/es/es_RightsTypes.h>
#include <nn/es/es_ELicenseTypes.h>
#include <nn/es/es_ActiveRightsContextTypes.h>

namespace nn { namespace es {

//-----------------------------------------------------------------------------

//! @name   ns 向けの API の宣言
//! @{

/**
* @brief  使用中権利リストのコンテキストを生成し、そのハンドルを取得します。
*
* @pre    事前に nn::es::Initialize() が呼ばれていること
*
* @return   対象コンテキストのハンドルが返ります。
*
* @details
*   使用中権利リストのコンテキストを生成し、そのハンドルを返します。
*
*   作成されたコンテキストには、特定のアカウントは紐づけられておらず、
*   当該デバイスに登録されている全アカウントが許可されたものになっています。
*   特定のアカウントに権利を限定する場合は、本 API の返値であるハンドルを使って
*   es::RegisterAccountForActiveRightsContext() を呼んでください。
*
*   本 API はアプリケーションの起動前に呼び出す必要があります。
*   同時に作成できるコンテキストはとりあえず最大 2 個までです。
*   （1 つは通常アプリ用、もう 1 つは Encounter 用を想定。）
*
*/
ActiveRightsContextHandle CreateActiveRightsContext() NN_NOEXCEPT;


/**
* @brief  使用中権利リストのコンテキストを破棄し、ハンドルを閉じます。
*
* @param[in] handle 対象コンテキストのハンドル
*
* @details
*   本 API の呼出しにより対象コンテキストは破棄され、
*   ハンドルは使用不可になります。
*
*   内部的に es::UnregisterAllTitleKey() 相当の処理も行われます。
*
*/
void CloseActiveRightsContextHandle(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  権利コンテキストにアカウント限定権利の利用者アカウントを設定します。
*
* @param[in] handle     権利コンテキストのハンドル
* @param[in] naIdList   登録するアカウントの NintendoAccountId の配列
* @param[in] naIdCount  登録するアカウントの NintendoAccountId の数
* @param[in] anyAccount アカウントを限定するか否か
*
* @return   処理の結果が返ります。
*
* @details
*   使用中権利リストのコンテキストにアカウント限定権利の利用者アカウントを
*   account::NintendoAccountId のリストで設定します。
*
*   本 API を呼び出すと、それまでに設定されていたアカウントリストを一旦破棄し、
*   新たなアカウントリストで上書きします。本 API を呼ばなかった場合の
*   デフォルトは、アカウント限定権利を使用しない状態となっています。
*
*   anyAccount == false の場合、対象コンテキストでは naIdList と naIdCount で
*   指定されたアカウント限定権利の RightsId リストを扱うようになります。
*
*   anyAccount == true の場合は naIdList と naIdCount は無視され、
*   デフォルトの状態、つまり、アカウント限定権利を扱わないようになります。
*
*/
Result SetUsersOfAccountRestrictedRights(ActiveRightsContextHandle handle, const account::NintendoAccountId naIdList[], int naIdCount, bool anyAccount) NN_NOEXCEPT;


/**
* @brief  権利コンテキストから、アカウント限定権利の利用者リストを取得します。
*
* @param[out] outLength     outNaIdList に格納した NintendoAccountId の要素数
* @param[out] outNaIdList   登録されたアカウント限定権利の利用者を格納する配列
* @param[in]  listLength    outNaIdList の長さ（格納可能な最大要素数）
* @param[in]  handle        権利コンテキストのハンドル
*
* @return   アカウント限定権利の使用者リストの取得可否を bool で返します。
*
* @details
*   対象権利コンテキストに対して、
*   SetUsersOfAccountRestrictedRights() で登録された
*   アカウント限定権利の利用者リストを返します。
*
*   対象権利コンテキストがアカウント限定権利を扱っている場合、true を返します。
*   この時、outLength および outNaIdList に利用者アカウント情報が格納されます。
*
*   対象権利コンテキストがアカウント限定権利を扱っていない場合、false を返します。
*   この時、outLength および outNaIdList には何も格納されません。
*
*/
bool GetUsersOfAccountRestrictedRights(int* outLength, account::NintendoAccountId outNaIdList[], int listLength, ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  使用予定の RightsId を登録します。
*
* @param[in] handle         使用中権利リストのハンドル
* @param[in] rightsIdList   使用予定の 64bit RightsId の配列
* @param[in] listLength     rightsIdList の長さ（要素数）
*
* @return   処理の結果が返ります。
*
* @details
*   指定された使用中権利リストに、rightsIdList で指定された権利を登録します。
*   既に登録済みの RightsId と重複する分はスキップされます。
*
*   本 API ではリストへの登録のみ行い、対応する eLicense の検索や照合等は
*   行ないません。個別の RightsId の有効性をチェックする場合には、
*   CheckRightsIdListValidity() を発行して下さい。
*
*/
Result RegisterRightsIdList(ActiveRightsContextHandle handle, const nn::es::RightsId rightsIdList[], int listLength) NN_NOEXCEPT;


/**
* @brief  登録済みの各 RightsId の有効性をチェックします。
*
* @param[in] handle         使用中権利リストのハンドル
*
* @details
*   RegisterRightsIdList() で登録済みの権利リストの全 RightsId に対して、
*   それに対応する eLicense を検索して対応付けを行います。
*   利用可能な eLicense が複数存在する場合には、
*   es 内のポリシーに従って唯一の eLicense が選定されます。
*
*   本 API により、権利リストの各 RightsId の有効・無効が判定されると共に、
*   権利リスト全体としての有効・無効および、最も近い将来の有効期限も
*   算出されます。
*
*/
void CheckRightsIdListValidity(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  登録済みの各 RightsId のうち、対応する eLicense がないものを権利リストから削除します。
*
* @param[in] handle         使用中権利リストのハンドル
*
* @pre
*   - 事前に CheckRightsIdListValidity() が発行されていること
*   - 事前に BeginUsingActiveRightsContext() が発行されていないこと
*
* @details
*   CheckRightsIdListValidity() の判定によって、対応する eLicense が
*   見つからなかった RightsId を権利リスト上から削除します。
*
*   本 API により、権利リスト全体としての有効・無効および、
*   最も近い将来の有効期限が更新されます。
*
*   本 API 発行後に権利リスト上に残った RightsId リストを取得したい場合は、
*   ListUsingRightsIds() を利用して下さい。
*
*/
void RemoveUnavailableRightsIds(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  使用中権利リストの使用開始を宣言します。
*
* @param[in] handle     使用中権利リストのハンドル
*
* @return   処理の結果が返ります。
*
* @details
*   使用中権利リストの使用開始を es に対して宣言します。
*   これにより es は、使用中権利リスト内の個々の RightsId に関して、
*   過去に登録されたことのあるタイトル鍵を再登録します。
*
*   このとき、各 RightsId に付随する eLicenseId の失効時間などを考慮し、
*   利用可能なものについてのみ、タイトル鍵の再登録を行ないます。
*
*   es::EndUsingActiveRightsContext() で使用終了宣言していたものを
*   再び使用開始する場合にも本 API を使用して下さい。
*
*/
Result BeginUsingActiveRightsContext(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  使用中権利リストの使用中止を宣言します。
*
* @param[in] handle     使用中権利リストのハンドル
*
* @details
*   使用中権利リストの使用中止を es に対して宣言します。
*
*   本 API ではタイトル鍵の解除は行いません。
*
*/
void EndUsingActiveRightsContext(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  アプリ終了時の、使用中権利リストの強制使用開始を宣言します。
*
* @param[in] handle     使用中権利リストのハンドル
*
* @return   処理の結果が返ります。
*
* @details
*   アプリ終了処理を目的とした使用中権利リストの強制使用開始を es に対して
*   宣言します。これにより es は、使用中権利リスト内の個々の RightsId に関して、
*   過去に登録されたことのあるタイトル鍵を再登録します。
*
*   本 API では、各 RightsId に付随する eLicenseId の失効時間などは考慮せず、
*   過去の登録有無によってのみタイトル鍵の再登録の可否が決定されます。
*
*/
Result ForceActivateRightsContextForExit(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  使用中権利リストから、使用中の RightsId をリストで取得します。
*
* @param[out] outList   使用中の RightsId を格納する配列
* @param[in] listLength outList の長さ（格納可能な最大要素数）
* @param[in] handle     使用中権利リストのハンドル
* @param[in] isTemporaryOnly    有効期限ありの eLicense のみを返すか否か
*
* @return   outList に格納した使用中の権利情報の要素数
*
* @details
*   使用中の権利リストを es::RightsId の配列で返します。
*
*   isTemporaryOnly = false の場合は、RegisterRightsIdList() で登録した
*   全ての RightsId が outList[] のリストに返されます。
*
*   isTemporaryOnly = true の場合、以下の条件を満たす RightsId が
*   outList[] のリストに返されます。
*
*   - 有効な eLicense がないもの（権利が剥奪されているなど）
*   - 有効な eLicense があり、アカウント限定権利ではなく、有効期限があるもの
*   - 有効な eLicense があり、アカウント限定権利であり、そのいずれかひとつにでも有効期限があるもの
*
*/
int ListUsingRightsIds(RightsId outList[], int listLength, ActiveRightsContextHandle handle, bool isTemporaryOnly = false) NN_NOEXCEPT;


/**
* @brief  使用中権利リストから、指定された権利オーナーの eLicenseId をリストで取得します。
*
* @param[out] outList           使用中の eLicenseId を格納する配列
* @param[in] listLength         outList の長さ（格納可能な最大要素数）
* @param[in] handle             使用中権利リストのハンドル
* @param[in] isTemporaryOnly    有効期限ありの eLicense のみを返すか否か
* @param[in] ownerId            検索対象とする権利オーナーの NintendoAccountId
*
* @return   outList に格納した eLicenseId の要素数
*
* @pre
*   - listLength > 0
*   - ownerId != nn::account::InvalidNintendoAccountId
*
* @details
*   使用中権利リストの中で、ownerId で指定された権利オーナーの
*   eLicenseId リストを指定された配列に返します。
*   本 API は、eLicense 有効期限の延長要求や利用状況報告を行なうための
*   eLicenseId リストを収集するために使用される想定です。
*
*   isTemporaryOnly = true の場合、有効期限を持つ eLicense の Id だけを
*   配列に格納します（有効期限が切れているものも格納します）。
*   isTemporaryOnly = false の場合、有効期限の有無に関係なく、
*   全ての eLicense の Id を配列に格納します。
*
*/
int ListUsingELicenseIds(ELicenseId outList[], int listLength, ActiveRightsContextHandle handle, bool isTemporaryOnly, account::NintendoAccountId ownerId) NN_NOEXCEPT;


/**
* @brief  使用中権利リストから、使用中 eLicense の所有者 NintendoAccountId リストを取得します。
*
* @param[out] outNaIdList   アカウント限定権利の所有者リストを格納する配列
* @param[in]  listLength    outNaIdList の長さ（格納可能な最大要素数）
* @param[in]  handle        使用中権利リストのハンドル
*
* @return   outList に格納した NintendoAccountId の要素数
*
* @details
*   対象権利コンテキストにおいて、使用中の各 eLicense における
*   ライセンス所有者の NintendoAccountId のリストを返します。
*
*/
int ListUsingELicenseOwnerIds(account::NintendoAccountId outNaIdList[], int listLength, ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  使用中権利リスト全体の状態を参照します。
*
* @param[in] handle     使用中権利リストのハンドル
*
* @return   使用中権利リストの状態を ActiveRightsContextStatus で返します。
*
* @details
*   使用中権利リスト全体の総合的な状態を ActiveRightsContextStatus で返します。
*
*   権利リストの中に、登録だけされて未チェックの RightsId のものがある場合、
*   ActiveRightsContextStatus_NotChecked が返ります。この場合には、
*   CheckRightsIdListValidity() および RemoveUnavailableRightsIds() を
*   順番に発行した上で、再度本 API を発行して下さい。
*
*/
ActiveRightsContextStatus GetActiveRightsContextStatus(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  使用中権利リスト全体での最も近い将来の有効期限を取得します。
*
* @param[in] handle     使用中権利リストのハンドル
*
* @pre
*   - GetActiveRightsContextStatus() != ActiveRightsContextStatus_NotChecked
*
* @return   最も近い将来の有効期限を示す Tick 値を返します。
*
* @details
*   使用中権利リスト全体の中で、最も近い将来の有効期限を示す Tick 値を返します。
*   有効期限がない場合、遠い未来を示す十分に大きい値の os::Tick 値が返ります。
*
*/
os::Tick GetActiveRightsContextExpiredTime(ActiveRightsContextHandle handle) NN_NOEXCEPT;


/**
* @brief  対象権利コンテキストの権利有無や有効期限に変化があったことを通知するシステムイベントを取得します。
*
* @param[in]  handle    使用中権利コンテキストのハンドル
*
* @return   システムイベントの ReadableHandle を表す sf::NativeHandle
*
* @details
*   対象権利コンテキストの権利の有無や有効期限に変化があったことを通知する
*   システムイベントの Readable ハンドルを sf::NativeHandle 型のまま返します。
*
*   対象の SystemEvent は、当該権利コンテキストに上記のような変化があった時に
*   Signal() されます。es プロセス側では Clear() は一切行いません。
*   当該 SystemEvent を構築する際の os::EventClearMode の指定や
*   Clear() のタイミングは利用する側で適切に指定して下さい。
*
*   なお、es プロセスによる Signal() 処理は spurious に行われる場合があります。
*
*/
sf::NativeHandle GetActiveRightsContextExpiredTimeChangedEvent(ActiveRightsContextHandle handle) NN_NOEXCEPT;


//! @}


}}  // namespace nn::es
