﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @file
 * @brief システム向けのエラーAPIで利用される型の定義
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/err/err_Types.h>
#include <nn/nn_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>

namespace nn { namespace err {

//! @name システム向けのエラーAPIで利用される型
//! @{

/**
 * @brief エラーメッセージデータベース（システムデータ）のバージョン情報を表す構造体です。
 */
struct ErrorMessageDatabaseVersion
{
    /**
    * @brief    メジャーバージョン。
    */
    uint16_t majorVersion;
    /**
    * @brief    マイナーバージョン。
    */
    uint16_t minorVersion;
};

/**
* @brief    エラービューアからのジャンプ先を表します。現在の定義・値は仮のものです。
*/
enum class ErrorViewerJumpDestination : uint8_t
{
    // TODO: 下記は適当な値です。システムデータが持つ値（エラーコードDB から吐き出される値）に合わせて更新します。
    Nowhere                 = 0, //!< ジャンプ先なし（「とじる」など）
    Set_SystemUpdate        = 1, //!< 本体設定 -> 本体更新
    Set_DataManagement      = 2, //!< 本体設定 -> データ管理
    Set_InternetSetting     = 3, //!< 本体設定 -> インターネット設定
    Cabinet_NfpDataSetting  = 4, //!< amiibo 設定 -> データの設定
};

/**
* @brief    エラーの種類を表します。
*/
enum class ErrorType : Bit8
{
    SystemData,                 //!< システムデータから情報を読み取るエラーです。
    SystemError,                //!< エラーコードとメッセージを指定するエラーです。（ショップ向け）
    ApplicationError,           //!< アプリケーション独自のエラーです。
    Eula,                       //!< EULA です。
    ParentalControl,            //!< ペアレンタルコントロールの制限によるエラーです。
    RecordedSystemData,         //!< システムデータから情報を読み取るエラーの記録です。
    RecordedSystemError,        //!< エラーコードとメッセージを指定するエラーの記録です。
    RecordedApplicationError,   //!< アプリケーション独自のエラーの記録です。
    SystemUpdateEula,           //!< 本体更新データの EULA です。
};

/**
* @brief    エラービューアアプレットに渡される入力パラメータの最大サイズです。
*/
const int ErrorViewerStartupParamSizeMax = 5 * 1024;

/**
* @brief    エラーの記録に必要なバッファサイズです。
* @details  記録内容の取得のためにアプリケーション管理データの取得を行うため、
*           アイコンサイズ（最大128KiB) + アプリケーション管理データのサイズが必要となります。
*/
const size_t RequiredBufferSizeForRecordError = 128 * 1024 + sizeof(ns::ApplicationControlProperty);

/**
* @brief    障害解析用に保存する Result のバックトレースを表します。
*/
struct ResultBacktrace
{
    static const int CountMax = 32;
    int count;
    Result results[CountMax];

    /**
    * @brief        障害解析用に保存する Result のバックトレースを作成します。
    * @param[out]   pOutValue 出力。
    * @param[in]    results 保存する Result の配列。
    * @param[in]    count results に格納された Result の数。
    * @pre
    *               - pOutValue != nullptr
    *               - results != nullptr
    *               - count > 0 && count <= ResultBacktrace::CountMax
    */
    static void Make(ResultBacktrace* pOutValue, const Result results[], int count) NN_NOEXCEPT;
};

//! @}
//-----------------------------------------------------------------------------

}}
