﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief Error report class.
*/

#include <nn/os.h>
#include <nn/erpt/erpt_Types.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/sfdl/erpt.sfdl.h>

namespace nn   {
namespace erpt {

/**
 *  @brief Provides interface for report retrieval. User must be provisioned for access to 'erpt:r' port.
 */
class Report
{
    NN_DISALLOW_COPY(Report);
    NN_DISALLOW_MOVE(Report);

private:
    nn::sf::SharedPointer<nn::erpt::sf::IReport> m_pImpl;

public:
    /**
     *  @brief      Retrieve report.
     *  @param[out] pReadCountOut                       Pointer to store number of bytes read from the report.
     *  @param[in]  pBufferIn                           Pointer to the buffer.
     *  @param[in]  bufferLength                        Size of the buffer.
     *  @retval     nn::erpt::ResultNotInitialized      Open has not been called.
     *  @retval     nn::erpt::ResultInvalidArgument     There is a problem with the passed in arguments.
     *  @retval     nn::erpt::ResultPowerStateViolation Operation can not be completed in the current power state.
     *  @details    Reads up to @a bufferLength bytes from the report starting at current position.
     *              First call will set report position to 0.
     *              After operation completes, report's position will be advanced
     *              by the number of bytes read from the report. A successfull read that returns 0 bytes
     *              indicates end of report file.
     *  @pre        @ref Open must be called.
    */
    nn::Result Read(uint32_t* pReadCountOut, uint8_t* pBufferIn, uint32_t bufferLength)
    NN_NOEXCEPT;

    /**
     *  @brief      Set flags for this report.
     *  @param[in]  reportFlags                         Set of flags, see nn::erpt::ReportFlagSet.
     *  @retval     nn::erpt::ResultNotInitialized      Open has not been called, or failed.
     *  @retval     nn::erpt::ResultPowerStateViolation Operation can not be completed in the current power state.
     *  @details    Allows client to set flags for this report. For example, after report has succesfully been
     *              transmitted to remote server, client may set a flag to indicate that further transmission
     *              of this report is not required.
     *              Note that flags can only be set, it is not possible to clear a flag using this or any other API.
     *  @pre        @ref Open must be called.
    */
    nn::Result SetFlags(ReportFlagSet reportFlags)
    NN_NOEXCEPT;

    /**
     *  @brief      Get flags for this report.
     *  @param[in]  p_ReportFlags                       Pointer to store nn::erpt::ReportFlagSet.
     *  @retval     nn::erpt::ResultNotInitialized      Open has not been called.
     *  @details    Allows client to get flags for this report.
     *  @pre        @ref Open must be called.
    */
    nn::Result GetFlags(ReportFlagSet* p_ReportFlags)
    NN_NOEXCEPT;

    /**
     *  @brief      Get size of this report.
     *  @param[in]  pSize                               Pointer to the buffer to store size of the report.
     *  @retval     nn::erpt::ResultNotInitialized      Open has not been called.
     *  @retval     nn::erpt::ResultPowerStateViolation Operation can not be completed in the current power state.
     *  @details    Allows client to get size of the report.
     *  @pre        @ref Open must be called.
    */
    nn::Result GetSize(int64_t* pSize)
    NN_NOEXCEPT;

    /**
     *  @brief      Open report stream.
     *  @param[in]  reportId                            Report ID, see @ref nn::erpt::ReportId .
     *  @retval     nn::erpt::ResultAlreadyInitialized  This instance has already been initialized.
     *  @retval     nn::erpt::ResultNotFound            Could not find report with this ID.
     *  @retval     nn::erpt::ResultOutOfMemory         Report server is out of memory.
     *  @retval     nn::erpt::ResultPowerStateViolation Operation can not be completed in the current power state.
     *  @details    Establishes connection to a report.
     *  @pre        @ref Open should not have been called.
    */
    nn::Result Open(ReportId reportId)
    NN_NOEXCEPT;

    /**
     *  @brief      Close report stream.
     *  @retval     nn::erpt::ResultNotInitialized      Open has not been called, or failed.
     *  @details    Terminates connection to a report.
     *  @pre        @ref Open must be called.
    */
    nn::Result Close()
    NN_NOEXCEPT;

    Report()  NN_NOEXCEPT;
    ~Report() NN_NOEXCEPT;
};

}}
