﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 *  @file
 *  @brief Error report context class.
*/

#include <nn/os.h>
#include <nn/erpt/erpt_Types.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/err/err_Types.h>

namespace nn   {
namespace erpt {

/**
   @brief       Container for context, keeps field-value pairs.
   @details     This class may be used to construct a set of field-value pairs for
                transmission to error report service.

                Sample code:
                @code
                #include <nn/erpt.h>
                nn::Result SubmitI64Sample(int64_t data)
                {
                    nn::Result result;

                    // Prepare for submission to 'Test' category (for all categories see nn::erpt::CategoryId).
                    nn::erpt::Context context(nn::erpt::Test);

                    // set 'TestI64' field to 'data' (for all fields see nn::erpt::FieldId).
                    if ((result = context.Add(nn::erpt::TestI64, data)).IsFailure() ||
                        (result = context.SubmitContext()).IsFailure())
                    {
                        return result;
                    }

                    return ResultSuccess();
                }

                nn::Result CreateReportSample(const char* pErrorString, uint32_t errorStringLength)
                {
                    nn::Result result;

                    // Prepare for submission to 'ErrorInfo' category (for all categories see nn::erpt::CategoryId).
                    // Usage of 'ErrorInfo' is required when creating reports.
                    nn::erpt::Context context(nn::erpt::ErrorInfo);

                    // Set 'ErrorCode' field to error string (for all fields see nn::erpt::FieldId).
                    // At least 'ErrorCode' field has to be set in order to generate a report.
                    if ((result = context.Add(nn::erpt::ErrorCode, pErrorString, errorStringLength)).IsFailure() ||
                        (result = context.CreateReport()).IsFailure())
                    {
                        return result;
                    }

                    return ResultSuccess();
                }
                @endcode
 */
class Context
{
private:
    ContextEntry m_Context;
    uint8_t      m_ArrayBuffer[ArrayBufferLength];

    nn::Result Add(FieldId id, const uint8_t* pBuffer, uint32_t bufferLength, FieldType type)
    NN_NOEXCEPT;

public:
    /**
     *  @brief Add BOOL.
     *  @param[in]  id                                      Field id.
     *  @param[in]  valueBool                               BOOL value.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @details    Assigns BOOL to a field with @a id.
     */
    nn::Result Add(FieldId id, bool valueBool)
    NN_NOEXCEPT;

    /**
     *  @brief Add 64 bit unsigned number.
     *  @param[in]  id                                      Field id.
     *  @param[in]  valueU64                                Unsigned 64 bit value.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @details    Assigns 64 bit unsigned integer to a field with @a id.
     */
    nn::Result Add(FieldId id, uint64_t valueU64)
    NN_NOEXCEPT;

    /**
     *  @brief Add 32 bit unsigned number.
     *  @param[in]  id                                      Field id.
     *  @param[in]  valueU32                                Unsigned 32 bit value.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @details    Assigns 32 bit unsigned integer to a field with @a id.
     */
    nn::Result Add(FieldId id, uint32_t valueU32)
    NN_NOEXCEPT;

    /**
    *  @brief Add 16 bit unsigned number.
    *  @param[in]  id                                      Field id.
    *  @param[in]  valueU16                                Unsigned 16 bit value.
    *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
    *  @details    Assigns 16 bit unsigned integer to a field with @a id.
    */
    nn::Result Add(FieldId id, uint16_t valueU16)
    NN_NOEXCEPT;

    /**
     *  @brief Add 8 bit unsigned number.
     *  @param[in]  id                                      Field id.
     *  @param[in]  valueU8                                 Unsigned 8 bit value.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @details    Assigns 8 bit unsigned integer to a field with @a id.
     */
    nn::Result Add(FieldId id, uint8_t valueU8)
    NN_NOEXCEPT;

    /**
     *  @brief Add 64 bit signed number.
     *  @param[in]  id                                      Field id.
     *  @param[in]  valueI64                                Signed 64 bit value.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @details    Assigns 64 bit signed integer to a field with @a id.
     */
    nn::Result Add(FieldId id, int64_t valueI64)
    NN_NOEXCEPT;

    /**
     *  @brief Add 32 bit signed number.
     *  @param[in]  id                                      Field id.
     *  @param[in]  valueI32                                Signed 32 bit value.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @details    Assigns 32 bit signed integer to a field with @a id.
     */
    nn::Result Add(FieldId id, int32_t valueI32)
    NN_NOEXCEPT;

    /**
    *  @brief Add 16 bit signed number.
    *  @param[in]  id                                      Field id.
    *  @param[in]  valueI16                                Signed 16 bit value.
    *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
    *  @details    Assigns 16 bit signed integer to a field with @a id.
    */
    nn::Result Add(FieldId id, int16_t valueI16)
    NN_NOEXCEPT;

    /**
    *  @brief Add 8 bit signed number.
    *  @param[in]  id                                      Field id.
    *  @param[in]  valueI8                                 Signed 8 bit value.
    *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
    *  @details    Assigns 8 bit signed integer to a field with @a id.
    */
    nn::Result Add(FieldId id, int8_t valueI8)
    NN_NOEXCEPT;

    /**
     *  @brief Add string.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pBuffer                                 String buffer.
     *  @param[in]  bufferLength                            Length of the string.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for string storage.
     *  @details    Assigns string to a field with @a id.
     */
    nn::Result Add(FieldId id, const char* pBuffer, uint32_t bufferLength)
    NN_NOEXCEPT;

    /**
     *  @brief Add array of 8 bit unsigned numbers.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pBuffer                                 Array buffer.
     *  @param[in]  bufferLength                            Length of the buffer.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for array storage.
     *  @details    Assigns array of 8 bit unsigned numbers to a field with @a id.
     */
    nn::Result Add(FieldId id, const uint8_t* pBuffer, uint32_t bufferLength)
    NN_NOEXCEPT;

    /**
     *  @brief Add array of 32 bit unsigned numbers.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pArray                                  Array buffer.
     *  @param[in]  elementCount                            Number of values in the array.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for array storage.
     *  @details    Assigns array of 32 bit unsigned numbers to a field with @a id.
     */
    nn::Result Add(FieldId id, const uint32_t* pArray, uint32_t elementCount)
    NN_NOEXCEPT;

    /**
     *  @brief Add array of 64 bit unsigned numbers.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pArray                                  Array buffer.
     *  @param[in]  elementCount                            Number of values in the array.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for array storage.
     *  @details    Assigns array of 64 bit unsigned numbers to a field with @a id.
     */
    nn::Result Add(FieldId id, const uint64_t* pArray,  uint32_t elementCount)
    NN_NOEXCEPT;

    /**
     *  @brief Add array of 8 bit signed numbers.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pArray                                  Array buffer.
     *  @param[in]  elementCount                            Number of values in the array.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for array storage.
     *  @details    Assigns array of 8 bit signed numbers to a field with @a id.
     */
    nn::Result Add(FieldId id, const int8_t*  pArray, uint32_t elementCount)
    NN_NOEXCEPT;

    /**
     *  @brief Add array of 32 bit signed numbers.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pArray                                  Array buffer.
     *  @param[in]  elementCount                            Number of values in the array.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for array storage.
     *  @details    Assigns array of 32 bit signed numbers to a field with @a id.
     */
    nn::Result Add(FieldId id, const int32_t*  pArray,  uint32_t elementCount)
    NN_NOEXCEPT;

    /**
     *  @brief Add array of 64 bit signed numbers.
     *  @param[in]  id                                      Field id.
     *  @param[in]  pArray                                  Array buffer.
     *  @param[in]  elementCount                            Number of values in the array.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for array storage.
     *  @details    Assigns array of 64 bit signed numbers to a field with @a id.
     */
    nn::Result Add(FieldId id, const int64_t*  pArray,  uint32_t elementCount) NN_NOEXCEPT;

    /**
     *  @brief Add error code with 'ErrorCode' field id.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for string storage.
     *  @details    Assigns error code with 'ErrorCode' field id.
     */
    nn::Result AddErrorCode(const nn::err::ErrorCode& errorCode) NN_NOEXCEPT;

    /**
     *  @brief Add error code converted from result with 'ErrorCode' field id.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for string storage.
     *  @details    Assigns error code with 'ErrorCode' field id.
     */
    nn::Result AddErrorCode(nn::Result result) NN_NOEXCEPT;

    /**
     *  @brief Add name of current thread with 'ThreadName' field id.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for string storage.
     *  @details    Assigns name of current thread with 'ThreadName' field id.
     */
    nn::Result AddCurrentThreadName() NN_NOEXCEPT;

    /**
     *  @brief Add name of specified thread with 'ThreadName' field id.
     *  @param[in]  pThread                                 Pointer to os::ThreadType.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for string storage.
     *  @details    Assigns name of the specified thread with 'ThreadName' field id.
     */
    nn::Result AddThreadName(const os::ThreadType* pThread) NN_NOEXCEPT;

    /**
     *  @brief Add ErrorContext's fields with their corresponding field id.
     *  @param[in]  errorContext                            ErrorContext.
     *  @retval     nn::erpt::ResultOutOfFieldSpace         nn::erpt::FieldsPerContext number of entries has already been added to context.
     *  @retval     nn::erpt::ResultOutOfArraySpace         No more space for string storage.
     *  @details    Assigns ErrorContext's fields with their corresponding field id.
     */
    nn::Result AddErrorContext(const err::ErrorContext& errorContext) NN_NOEXCEPT;

    /**
     *  @brief      Reset context fields
     *  @param[in]  category                                New category ID for this context.
     *  @details    Clients wishing to re-use context for a different category, may
     *              re-initialize this instance by calling this function and passing
     *              new category id. All fields in this instance of context will be reset.
     *              If context was created using constructor with user buffer,
     *              that buffer will be cleared and re-used.
     */
    nn::Result ResetContext(CategoryId category)
    NN_NOEXCEPT;

    /**
     *  @brief      Transmit context to the error process.
     *  @retval     nn::erpt::ResultCategoryNotFound        Category id used by this context is not valid.
     *  @retval     nn::erpt::ResultOutOfMemory             Error service can not allocate storage for this context.
     *  @retval     nn::erpt::ResultFieldTypeMismatch       Value of the specified type can not be used with this field id.
     *  @retval     nn::erpt::ResultFieldCategoryMismatch   Field id can not be used with this category.
     *  @retval     nn::erpt::ResultArrayFieldSizeTooLarge  There is a string/array field whose size exceeds @ref MaxArrayFieldSize.
     *  @details    This function transfers context to the error process.
     *              Field-value information from this context will be kept in
     *              memory inside error process until such time when error report is generated.
     *              At that time, this information will be appended to the error
     *              report.
     */
    nn::Result SubmitContext()
    NN_NOEXCEPT;

    /**
     *  @brief      Create error report.
     *  @param[in]  type                                    Type of the report.
     *  @param[in]  meta                                    User provided meta data for this report, stored together with the report.
     *  @retval     nn::erpt::ResultOutOfMemory             Error service can not allocate storage for this context.
     *  @retval     nn::erpt::ResultFieldTypeMismatch       Value of the specified type can not be used with this field id.
     *  @retval     nn::erpt::ResultFieldCategoryMismatch   Field id can not be used with this category.
     *  @retval     nn::erpt::ResultArrayFieldSizeTooLarge  There is a string/array field whose size exceeds @ref MaxArrayFieldSize.
     *  @retval     nn::erpt::ResultRequiredContextMissing  Required context (ErrorInfo) was not provided.
     *  @retval     nn::erpt::ResultRequiredFieldMissing    Required field ErrorCode was not provided.
     *  @retval     nn::erpt::ResultFormatterError          Conversion of fields to msgpack failed.
     *  @details    This function will generate report from this and all previously transmitted contexts.
     *              @a pReportMetaData provides user defined meta data that is stored together
     *              with this report. The meta data can be retrieved separately from the report
     *              and may be used to make decisions on whether to display, transmit, etc...
     *              Context provided with this call should be of ErrorInfo category and should contain at least
     *              ErrorCode fields.
     */
    nn::Result CreateReport(ReportType type, ReportMetaData meta)
    NN_NOEXCEPT;

    /**
     *  @brief      Create error report.
     *  @param[in]  type                                    Type of the report.
     *  @retval     nn::erpt::ResultOutOfMemory             Error service can not allocate storage for this context.
     *  @retval     nn::erpt::ResultFieldCategoryMismatch   Field id can not be used with this category.
     *  @retval     nn::erpt::ResultRequiredContextMissing  Required context (ErrorInfo) was not provided.
     *  @retval     nn::erpt::ResultRequiredFieldMissing    Required field ErrorCode was not provided.
     *  @retval     nn::erpt::ResultFormatterError          Conversion of fields to msgpack failed.
     *  @retval     nn::erpt::ResultPowerStateViolation     Operation can not be completed in the current power state.
     *  @details    This function will generate report from this and all previously transmitted contexts.
     *              Context provided with this call should be of ErrorInfo category and should contain at least
     *              ErrorCode field.
     */
    nn::Result CreateReport(ReportType type)
    NN_NOEXCEPT;

    /**
     * @brief       Constructor.
     * @param[in]   category                                Category ID.
     * @details     This constructor will intialize this instance with passed in category id,
     *              and a default buffer of size nn::erpt::ArrayBufferLength. The buffer will be
     *              used for string/array storage.
     */
    explicit Context(CategoryId category)
    NN_NOEXCEPT;

    /**
     * @brief       Constructor.
     * @param[in]   category                                Category ID.
     * @param[in]   arrayBuffer                             Buffer for array/string storage.
     * @param[in]   arrayBufferSize                         Size of the buffer.
     * @details     This constructor will intialize this instance with passed in category id,
     *              and a user provided buffer for string/array storage. Maximum size of the buffer is
     *              nn::erpt::MaxArrayBufferLength.
     */
    Context(CategoryId category, uint8_t* arrayBuffer, uint32_t arrayBufferSize)
    NN_NOEXCEPT;
};

}}
