﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// NOLINT(build/header_guard)

/**
 * @file
 * @brief Master file from which field/category strings/ids and various supporting structs are generated.
 */

#define ERRVERSION 0 // NOLINT

/**
 * @enum nn::erpt::FieldType
 * @brief Supported Storage Types.
 * Type                   |  Description
 * -----------------------|-------------------------
 * FieldType_NumericU64   | unsigned 64 bit number
 * FieldType_NumericU32   | unsigned 32 bit number
 * FieldType_NumericI64   | signed 64 bit number
 * FieldType_NumericI32   | signed 32 bit number
 * FieldType_String       | char string
 * FieldType_U8Array      | array of 8 bit unsigned numbers
 * FieldType_U32Array     | array of 32 bit unsigned numbers
 * FieldType_U64Array     | array of 64 bit unsigned nbumbers
 * FieldType_I32Array     | array of 32 bit signed numbers
 * FieldType_I64Array     | array of 64 bit signed numbers
 * FieldType_Bool         | bool
 * FieldType_NumericU16   | unsigned 16 bit number
 * FieldType_NumericU8    | unsigned 8 bit number
 * FieldType_NumericI16   | signed 16 bit number
 * FieldType_NumericI8    | signed 8 bit number
 */

#define FOREACH_TYPE(ENTRY)\
        ENTRY(FieldType_NumericU64, 0)\
        ENTRY(FieldType_NumericU32, 1)\
        ENTRY(FieldType_NumericI64, 2)\
        ENTRY(FieldType_NumericI32, 3)\
        ENTRY(FieldType_String,     4)\
        ENTRY(FieldType_U8Array,    5)\
        ENTRY(FieldType_U32Array,   6)\
        ENTRY(FieldType_U64Array,   7)\
        ENTRY(FieldType_I32Array,   8)\
        ENTRY(FieldType_I64Array,   9)\
        ENTRY(FieldType_Bool,       10)\
        ENTRY(FieldType_NumericU16, 11)\
        ENTRY(FieldType_NumericU8,  12)\
        ENTRY(FieldType_NumericI16, 13)\
        ENTRY(FieldType_NumericI8,  14)\
        ENTRY(FieldType_I8Array,    15)\

/*
 * @enum nn::erpt::srv::FieldFlag
 * @brief Supported Flags.
 * Type                   |  Description
 * -----------------------|-------------------------
 * FieldFlag_None         | field does not have any flags set
 * FieldFlag_Encrypt      | encrypt field prior to storing in report
 */
#define FOREACH_FLAG(ENTRY)\
        ENTRY(FieldFlag_None, 0)\
        ENTRY(FieldFlag_Encrypt, 1)\

/**
 * @enum nn::erpt::CategoryId
 * @brief   Supported Category IDs.
 * @details A category groups together one or more items which should be
 *          added to report as a set. When context is added to report,
 *          all existing items from the same category are discarded and replaced with
 *          the items contained in the new context. It is possible that new context
 *          may contain fewer items, only the items provided in the most recent
 *          context will be output into report. Fields which are added together
 *          may share the same category, fields which may be added individually
 *          should use a unique category. Please note that Category is not shared
 *          with remote server, it's a construct used only inside error report
 *          service on the device.
 *
 *          Example:
 *          Since network information is always added as a set,
 *          network related fields - ip, gw, mask - may share the same category.
 *          On the other hand EnableWifi field may be changed separately from other
 *          WIFI related items, and therefore it has to be assigned a unique category.
 *
 * Category ID                          |  Description
 * -------------------------------------|-----------------------------
 * Test                                 | Used for testing.
 * ErrorInfo                            | Error related information.
 * ErrorInfoAuto                        | Error related information, auto filled by error report service.
 * ErrorInfoDefaults                    | Error related information, default values
 * ConnectionStatusInfo                 | Network connection status.
 * NetworkInfo                          | Network connection information.
 * NXMacAddressInfo                     | MAC address.
 * StealthNetworkInfo                   | Information related to hidden SSID.
 * LimitHighCapacityInfo                | Information related to upload when storage is almost full.
 * NATTypeInfo                          | NAT type.
 * WirelessAPMacAddressInfo             | MAC address of access point.
 * GlobalIPAddressInfo                  | Global IP address.
 * EnableWirelessInterfaceInfo          | Related to all wireless communicaionts.
 * EnableWifiInfo                       | Related to WIFI.
 * EnableBluetoothInfo                  | Related to Bluetooth.
 * EnableNFCInfo                        | Related to NFC.
 * NintendoZoneSSIDListVersionInfo      | Nintendo Zone Configuration.
 * LANAdapterMacAddressInfo             | LAN adapter information.
 * ApplicationInfo                      | Application which shows/causes the error.
 * OccurrenceInfo                       | Task the system is performing at the time of error.
 * ProductModelInfo                     | Model related information.
 * CurrentLanguageInfo                  | Language related configuration.
 * UseNetworkTimeProtocolInfo           | Network time related configuration.
 * TimeZoneInfo                         | Time setting.
 * ControllerFirmwareInfo               | Controller related information.
 * VideoOutputInfo                      | Display related information.
 * NANDFreeSpaceInfo                    | NAND usage related information.
 * SDCardFreeSpaceInfo                  | SDCARD usage related
 * ScreenBrightnessInfo                 | Screen brightness configuration.
 * AudioFormatInfo                      | Audio format at time of error.
 * MuteOnHeadsetUnpluggedInfo           | Mute related configuration.
 * NumUserRegisteredInfo                | User account related information.
 * DataDeletionInfo                     | Data storage configuration.
 * ControllerVibrationInfo              | Controller vibration setting.
 * LockScreenInfo                       | Lock screen configuration.
 * InternalBatteryLotNumberInfo         | Battery information.
 * LeftControllerSerialNumberInfo       | Left controller information
 * RightControllerSerialNumberInfo      | Right controller information
 * NotificationInfo                     | Notification setting.
 * TVInfo                               | TV information.
 * SleepInfo                            | Sleep related settings.
 * ConnectionInfo                       | Network test related information.
 * NetworkErrorInfo                     | Network error related information.
 * FileAccessPathInfo                   | File system access error information.
 * GameCardCIDInfo                      | Game card information.
 * NANDCIDInfo                          | NAND information.
 * MicroSDCIDInfo                       | Micro SD information.
 * NANDSpeedModeInfo                    | NAND information.
 * MicroSDSpeedModeInfo                 | Micro SD information.
 * GameCardSpeedModeInfo                | Game card information.
 * UserAccountInternalIDInfo            | User account information.
 * NetworkServiceAccountInternalIDInfo  | Nintendo account information.
 * NintendoAccountInternalIDInfo        | Nitnendo account information.
 * USB3AvailableInfo                    | USB3 related information.
 * CallStackInfo                        | Crash callstack.
 * SystemStartupLogInfo                 | Startup log.
 * RegionSettingInfo                    | Region information.
 * NintendoZoneConnectedInfo            | Nintendo zone information.
 * ForceSleepInfo                       | Forced sleep related information.
 * ChargerInfo                          | Charger operation.
 * SystemPowerStateInfo                 | System power state information.
 * PerformanceInfo                      | Current performance mode and performance configuration.
 * ThrottlingInfo                       | Current soctherm throttling status.
 * GameCardErrorInfo                    | Game card error information.
 * EdidInfo                             | EDID of the last connected display.
 * ThermalInfo                          | Current temperatures and duty rate of fan.
 * CradleFirmwareInfo                   | Cradle firmware info.
 * RunningApplicationInfo               | Currently running application.
 * RunningAppletInfo                    | Currently running applet.
 * FocusedAppletHistoryInfo             | History of focused applet.
 * CompositorInfo                       | (RESERVED) not used
 * BatteryChargeInfo                    | Battery charge information.
 * NANDExtendedCsd                      | NAND Extended CSD information.
 * NANDPatrolInfo                       | NAND patrol information.
 * NANDErrorInfo                        | NAND control error information.
 * NANDDriverLog                        | NAND driver log.
 * SdCardSizeSpec                       | SD Card size specification information.
 * SdCardErrorInfo                      | SD Card control error information.
 * SdCardDriverLog                      | SD Card driver log.
 * FsProxyErrorInfo                     | FileSystem Proxy error information.
 * SystemAppletSceneInfo                | SystemApplet scene information.
 * VideoInfo                            | Information related to video decoder/encoder.
 * GpuErrorInfo                         | GPU error related information.
 * PowerClockInfo                       | Power and clock related information.
 * AdspErrorInfo                        | Adsp error related information.
 * NvDispDeviceInfo                     | nvdisp driver deviceinfo.
 * NvDispDcWindowInfo                   | nvdisp DC Window info.
 * NvDispDpModeInfo                     | nvdisp DP Mode info.
 * NvDispDpLinkSpec                     | nvdisp DP Link Spec.
 * NvDispDpLinkStatus                   | nvdisp DP Link Status.
 * NvDispDpHdcpInfo                     | nvdisp DP Hdcp Info.
 * NvDispDpAuxCecInfo                   | nvdisp DPAUX and CEC Info.
 * NvDispDcInfo                         | nvdisp DC status.
 * NvDispDsiInfo                        | nvdisp DSI status.
 * NvDispErrIDInfo                      | nvdisp ErrorID information.
 * SdCardMountInfo                      | SD Card Mount Info.
 * RetailInteractiveDisplayInfo         | Retail Interactive Display Info.
 * CompositorStateInfo                  | nvnflinger state info.
 * CompositorLayerInfo                  | nvnflinger layer info.
 * CompositorDisplayInfo                | nvnflinger display info.
 * CompositorHWCInfo                    | nvnflinger HWComposerinfo.
 * MonitorCapability                    | Connected Monitor capability.
 * ErrorReportSharePermissionInfo       | User permission state to share error reports with other companies.
 * MultimediaInfo                       | Multimedia info.
 * ConnectedControllerInfo              | Connected Controller Info.
 * FsMemoryInfo                         | FileSystem Memory information.
 * UserClockContextInfo                 | SystemClockContext info of standard user clock.
 * NetworkClockContextInfo              | SystemClockContext info of standard network clock.
 * MonitorSettings                      | Connected monitor mode information.
 * RebootlessSystemUpdateVersionInfo    | Rebootless system update version information.
 * PcieLoggedStateInfo                  | Pcie logged state information.
 */

/*
 * Macro for adding new categories.
 * This macro may be used to declare new categories.
 *          Macro takes three arguments:
 *          - Name,
 *          - Enum number assigned to the category, should be sequential,
 *          - Number of records to keep and store in report, for example it may be
 *            possible to store N last known network configurations and output them
 *            into report when report is generated.
 *
 * When adding new categories, please update doxygen comments for @enum nn::erpt::CategoryId.
 */
#define FOREACH_CATEGORY(ENTRY)\
        ENTRY(Test,                                 0,              1)\
        ENTRY(ErrorInfo,                            1,              1)\
        ENTRY(ConnectionStatusInfo,                 2,              1)\
        ENTRY(NetworkInfo,                          3,              1)\
        ENTRY(NXMacAddressInfo,                     4,              1)\
        ENTRY(StealthNetworkInfo,                   5,              1)\
        ENTRY(LimitHighCapacityInfo,                6,              1)\
        ENTRY(NATTypeInfo,                          7,              1)\
        ENTRY(WirelessAPMacAddressInfo,             8,              1)\
        ENTRY(GlobalIPAddressInfo,                  9,              1)\
        ENTRY(EnableWirelessInterfaceInfo,          10,             1)\
        ENTRY(EnableWifiInfo,                       11,             1)\
        ENTRY(EnableBluetoothInfo,                  12,             1)\
        ENTRY(EnableNFCInfo,                        13,             1)\
        ENTRY(NintendoZoneSSIDListVersionInfo,      14,             1)\
        ENTRY(LANAdapterMacAddressInfo,             15,             1)\
        ENTRY(ApplicationInfo,                      16,             1)\
        ENTRY(OccurrenceInfo,                       17,             1)\
        ENTRY(ProductModelInfo,                     18,             1)\
        ENTRY(CurrentLanguageInfo,                  19,             1)\
        ENTRY(UseNetworkTimeProtocolInfo,           20,             1)\
        ENTRY(TimeZoneInfo,                         21,             1)\
        ENTRY(ControllerFirmwareInfo,               22,             1)\
        ENTRY(VideoOutputInfo,                      23,             1)\
        ENTRY(NANDFreeSpaceInfo,                    24,             1)\
        ENTRY(SDCardFreeSpaceInfo,                  25,             1)\
        ENTRY(ScreenBrightnessInfo,                 26,             1)\
        ENTRY(AudioFormatInfo,                      27,             1)\
        ENTRY(MuteOnHeadsetUnpluggedInfo,           28,             1)\
        ENTRY(NumUserRegisteredInfo,                29,             1)\
        ENTRY(DataDeletionInfo,                     30,             1)\
        ENTRY(ControllerVibrationInfo,              31,             1)\
        ENTRY(LockScreenInfo,                       32,             1)\
        ENTRY(InternalBatteryLotNumberInfo,         33,             1)\
        ENTRY(LeftControllerSerialNumberInfo,       34,             1)\
        ENTRY(RightControllerSerialNumberInfo,      35,             1)\
        ENTRY(NotificationInfo,                     36,             1)\
        ENTRY(TVInfo,                               37,             1)\
        ENTRY(SleepInfo,                            38,             1)\
        ENTRY(ConnectionInfo,                       39,             1)\
        ENTRY(NetworkErrorInfo,                     40,             1)\
        ENTRY(FileAccessPathInfo,                   41,             1)\
        ENTRY(GameCardCIDInfo,                      42,             1)\
        ENTRY(NANDCIDInfo,                          43,             1)\
        ENTRY(MicroSDCIDInfo,                       44,             1)\
        ENTRY(NANDSpeedModeInfo,                    45,             1)\
        ENTRY(MicroSDSpeedModeInfo,                 46,             1)\
        ENTRY(GameCardSpeedModeInfo,                47,             1)\
        ENTRY(UserAccountInternalIDInfo,            48,             1)\
        ENTRY(NetworkServiceAccountInternalIDInfo,  49,             1)\
        ENTRY(NintendoAccountInternalIDInfo,        50,             1)\
        ENTRY(USB3AvailableInfo,                    51,             1)\
        ENTRY(CallStackInfo,                        52,             1)\
        ENTRY(SystemStartupLogInfo,                 53,             1)\
        ENTRY(RegionSettingInfo,                    54,             1)\
        ENTRY(NintendoZoneConnectedInfo,            55,             1)\
        ENTRY(ForceSleepInfo,                       56,             1)\
        ENTRY(ChargerInfo,                          57,             1)\
        ENTRY(RadioStrengthInfo,                    58,             1)\
        ENTRY(ErrorInfoAuto,                        59,             1)\
        ENTRY(AccessPointInfo,                      60,             1)\
        ENTRY(ErrorInfoDefaults,                    61,             1)\
        ENTRY(SystemPowerStateInfo,                 62,             1)\
        ENTRY(PerformanceInfo,                      63,             1)\
        ENTRY(ThrottlingInfo,                       64,             1)\
        ENTRY(GameCardErrorInfo,                    65,             1)\
        ENTRY(EdidInfo,                             66,             1)\
        ENTRY(ThermalInfo,                          67,             1)\
        ENTRY(CradleFirmwareInfo,                   68,             1)\
        ENTRY(RunningApplicationInfo,               69,             1)\
        ENTRY(RunningAppletInfo,                    70,             1)\
        ENTRY(FocusedAppletHistoryInfo,             71,             1)\
        ENTRY(CompositorInfo,                       72,             1)\
        ENTRY(BatteryChargeInfo,                    73,             1)\
        ENTRY(NANDExtendedCsd,                      74,             1)\
        ENTRY(NANDPatrolInfo,                       75,             1)\
        ENTRY(NANDErrorInfo,                        76,             1)\
        ENTRY(NANDDriverLog,                        77,             1)\
        ENTRY(SdCardSizeSpec,                       78,             1)\
        ENTRY(SdCardErrorInfo,                      79,             1)\
        ENTRY(SdCardDriverLog ,                     80,             1)\
        ENTRY(FsProxyErrorInfo,                     81,             1)\
        ENTRY(SystemAppletSceneInfo,                82,             1)\
        ENTRY(VideoInfo,                            83,             1)\
        ENTRY(GpuErrorInfo,                         84,             1)\
        ENTRY(PowerClockInfo,                       85,             1)\
        ENTRY(AdspErrorInfo,                        86,             1)\
        ENTRY(NvDispDeviceInfo,                     87,             1)\
        ENTRY(NvDispDcWindowInfo,                   88,             1)\
        ENTRY(NvDispDpModeInfo,                     89,             1)\
        ENTRY(NvDispDpLinkSpec,                     90,             1)\
        ENTRY(NvDispDpLinkStatus,                   91,             1)\
        ENTRY(NvDispDpHdcpInfo,                     92,             1)\
        ENTRY(NvDispDpAuxCecInfo,                   93,             1)\
        ENTRY(NvDispDcInfo,                         94,             1)\
        ENTRY(NvDispDsiInfo,                        95,             1)\
        ENTRY(NvDispErrIDInfo,                      96,             1)\
        ENTRY(SdCardMountInfo,                      97,             1)\
        ENTRY(RetailInteractiveDisplayInfo,         98,             1)\
        ENTRY(CompositorStateInfo,                  99,             1)\
        ENTRY(CompositorLayerInfo,                  100,            1)\
        ENTRY(CompositorDisplayInfo,                101,            1)\
        ENTRY(CompositorHWCInfo,                    102,            1)\
        ENTRY(MonitorCapability,                    103,            1)\
        ENTRY(ErrorReportSharePermissionInfo,       104,            1)\
        ENTRY(MultimediaInfo,                       105,            1)\
        ENTRY(ConnectedControllerInfo,              106,            1)\
        ENTRY(FsMemoryInfo,                         107,            1)\
        ENTRY(UserClockContextInfo,                 108,            1)\
        ENTRY(NetworkClockContextInfo,              109,            1)\
        ENTRY(AcpGeneralSettingsInfo,               110,            1)\
        ENTRY(AcpPlayLogSettingsInfo,               111,            1)\
        ENTRY(AcpAocSettingsInfo,                   112,            1)\
        ENTRY(AcpBcatSettingsInfo,                  113,            1)\
        ENTRY(AcpStorageSettingsInfo,               114,            1)\
        ENTRY(AcpRatingSettingsInfo,                115,            1)\
        ENTRY(MonitorSettings,                      116,            1)\
        ENTRY(RebootlessSystemUpdateVersionInfo,    117,            1)\
        ENTRY(NifmConnectionTestInfo,               118,            1)\
        ENTRY(PcieLoggedStateInfo,                  119,            1)\
// ~FOREACH_CATEGORY(ENTRY)


/**
 * @enum  nn::erpt::FieldId
 * @brief Supported Field IDs.
 * Field ID                              | Type                  | Category ID                         | Description
 * --------------------------------------|-----------------------|-------------------------------------|----------------------------
 * TestU64                               | FieldType_NumericU64  | Test                                | Test value.
 * TestU32                               | FieldType_NumericU32  | Test                                | Test value.
 * TestI64                               | FieldType_NumericI64  | Test                                | Test value.
 * TestI32                               | FieldType_NumericI32  | Test                                | Test value.
 * TestString                            | FieldType_String      | Test                                | Test value.
 * TestU8Array                           | FieldType_U8Array     | Test                                | Test value.
 * TestU32Array                          | FieldType_U32Array    | Test                                | Test value.
 * TestU64Array                          | FieldType_U64Array    | Test                                | Test value.
 * TestI32Array                          | FieldType_I32Array    | Test                                | Test value.
 * TestI64Array                          | FieldType_I64Array    | Test                                | Test value.
 * ErrorCode                             | FieldType_String      | ErrorInfo                           | Error code.
 * ErrorDescription                      | FieldType_String      | ErrorInfo                           | Description of error.
 * ReportIdentifier                      | FieldType_String      | ErrorInfoAuto                       | Report id, auto filled by error report service.
 * OccurrenceTimestamp                   | FieldType_NumericI64  | ErrorInfoAuto                       | Timestamp, auto filled by error report service.
 * ConnectionStatus                      | FieldType_String      | ConnectionStatusInfo                | Network connection status.
 * AccessPointSSID                       | FieldType_String      | AccessPointInfo                     | SSID of WIFI access point.
 * AccessPointSecurityType               | FieldType_String      | AccessPointInfo                     | Type of WIFI security.
 * AccessPointChannel                    | FieldType_NumericU16  | AccessPointInfo                     | Channel of WIFI access point.
 * RadioStrength                         | FieldType_NumericU32  | RadioStrengthInfo                   | WIFI signal strength.
 * AccessPointRssi                       | FieldType_NumericI32  | RadioStrengthInfo                   | Received Signal Strength Indicator
 * NXMacAddress                          | FieldType_String      | NXMacAddressInfo                    | MAC address.
 * IPAddressAcquisitionMethod            | FieldType_NumericU32  | NetworkInfo                         | IP configuration type: manual(0) or auto(1).
 * CurrentIPAddress                      | FieldType_String      | NetworkInfo                         | Local IP address.
 * SubnetMask                            | FieldType_String      | NetworkInfo                         | Subnet Mask.
 * GatewayIPAddress                      | FieldType_String      | NetworkInfo                         | Gateway IP address.
 * DNSType                               | FieldType_NumericU32  | NetworkInfo                         | DNS configuration type: manual(0) or auto(1).
 * PriorityDNSIPAddress                  | FieldType_String      | NetworkInfo                         | Primary DNS address.
 * AlternateDNSIPAddress                 | FieldType_String      | NetworkInfo                         | Secondary DNS address.
 * UseProxyFlag                          | FieldType_Bool        | NetworkInfo                         | Proxy state: disabled(0) or enabled(1).
 * ProxyIPAddress                        | FieldType_String      | NetworkInfo                         | Proxy IP address.
 * ProxyPort                             | FieldType_NumericU32  | NetworkInfo                         | Proxy port.
 * ProxyAutoAuthenticateFlag             | FieldType_Bool        | NetworkInfo                         | Proxy auto authentication: disabled(0) or enabled(1).
 * MTU                                   | FieldType_NumericU32  | NetworkInfo                         | MTU.
 * ConnectAutomaticallyFlag              | FieldType_Bool        | NetworkInfo                         | Auto connect to SSID: disabled(0) or enabled(1).
 * UseStealthNetworkFlag                 | FieldType_Bool        | StealthNetworkInfo                  | Auto connect to hidden SSID: disabled(0) or enabled(1).
 * LimitHighCapacityFlag                 | FieldType_Bool        | LimitHighCapacityInfo               | Limit downloads if storage capacity is low: disable(0) or enable(1).
 * NATType                               | FieldType_String      | NATTypeInfo                         | Information from NAT check server.
 * WirelessAPMacAddress                  | FieldType_String      | WirelessAPMacAddressInfo            | MAC of WIFI access point.
 * GlobalIPAddress                       | FieldType_String      | GlobalIPAddressInfo                 | Global IP address assigned to system.
 * EnableWirelessInterfaceFlag           | FieldType_Bool        | EnableWirelessInterfaceInfo         | All wireless comms disabled(0) or enabled(1).
 * EnableWifiFlag                        | FieldType_Bool        | EnableWifiInfo                      | WIFI disabled(0) or enabled(1).
 * EnableBluetoothFlag                   | FieldType_Bool        | EnableBluetoothInfo                 | Bluetooth disabled(0) or enabled(1).
 * EnableNFCFlag                         | FieldType_Bool        | EnableNFCInfo                       | NFC disabled(0) or enabled(1).
 * NintendoZoneSSIDListVersion           | FieldType_String      | NintendoZoneSSIDListVersionInfo     | Version of Nintendo zone SSID list.
 * LANAdapterMacAddress                  | FieldType_String      | LANAdapterMacAddressInfo            | MAC address of LAN adapter attached to the cradle.
 * ApplicationID                         | FieldType_String      | ApplicationInfo                     | Title ID of the application which shows/causes the error.
 * ApplicationTitle                      | FieldType_String      | ApplicationInfo                     | Name of the application which shows/causes the error.
 * ApplicationVersion                    | FieldType_String      | ApplicationInfo                     | Version of the application which shows/causes the error.
 * ApplicationStorageLocation            | FieldType_String      | ApplicationInfo                     | Storage location of the application which shows/causes the error.
 * DownloadContentType                   | FieldType_String      | OccurrenceInfo                      | Type of content being downloaded at the time of error (if any).
 * InstallContentType                    | FieldType_String      | OccurrenceInfo                      | Type of content being installed at the time of error (if any).
 * ConsoleStartingUpFlag                 | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened during console startup.
 * SystemStartingUpFlag                  | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened during system startup.
 * ConsoleFirstInitFlag                  | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened during initial system setup.
 * HomeMenuScreenDisplayedFlag           | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened in HOME Menu applet.
 * DataManagementScreenDisplayedFlag     | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened in Data Management applet.
 * ConnectionTestingFlag                 | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened during connection testing.
 * ApplicationRunningFlag                | FieldType_Bool        | OccurrenceInfo                      | Indicates that error happened during application run.
 * DataCorruptionDetectedFlag            | FieldType_Bool        | OccurrenceInfo                      | Indicates that save data corruption was detected.
 * ProductModel                          | FieldType_String      | ProductModelInfo                    | System product model.
 * CurrentLanguage                       | FieldType_String      | CurrentLanguageInfo                 | Currently selected language.
 * UseNetworkTimeProtocolFlag            | FieldType_Bool        | UseNetworkTimeProtocolInfo          | Adjust clock using time server: disabled(0) or enabled(1).
 * TimeZone                              | FieldType_String      | TimeZoneInfo                        | Currently configured time zone.
 * ControllerFirmware                    | FieldType_String      | ControllerFirmwareInfo              | FW version of attached controller(s).
 * VideoOutputSetting                    | FieldType_String      | VideoOutputInfo                     | Display type at time of error: TV or built in screen.
 * NANDFreeSpace                         | FieldType_NumericU64  | NANDFreeSpaceInfo                   | Free space in embedded storage.
 * SDCardFreeSpace                       | FieldType_NumericU64  | SDCardFreeSpaceInfo                 | Free space in SD card.
 * SerialNumber                          | FieldType_String      | ErrorInfoAuto                       | System serial number, auto filled by error report service.
 * OsVersion                             | FieldType_String      | ErrorInfoAuto                       | System firmware version, auto filled by error report service.
 * ScreenBrightnessAutoAdjustFlag        | FieldType_Bool        | ScreenBrightnessInfo                | Auto brightness adjustment disabled(0) or enabled(1).
 * HdmiAudioOutputMode                   | FieldType_String      | AudioFormatInfo                     | Audio output format: 1ch, 2ch, 5.1ch, 7ch.
 * SpeakerAudioOutputMode                | FieldType_String      | AudioFormatInfo                     | Audio output format: 1ch, 2ch, 5.1ch, 7ch.
 * HeadphoneAudioOutputMode              | FieldType_String      | AudioFormatInfo                     | Audio output format: 1ch, 2ch, 5.1ch, 7ch.
 * MuteOnHeadsetUnpluggedFlag            | FieldType_Bool        | MuteOnHeadsetUnpluggedInfo          | Mute when headphones unplugged disabled(0) or enabled(1).
 * NumUserRegistered                     | FieldType_NumericI32  | NumUserRegisteredInfo               | Number of local accounts.
 * StorageAutoOrganizeFlag               | FieldType_Bool        | DataDeletionInfo                    | Auto delete rarely used SW disabled(0) or enabled(1).
 * ControllerVibrationVolume             | FieldType_String      | ControllerVibrationInfo             | Controller vibrator volume.
 * LockScreenFlag                        | FieldType_Bool        | LockScreenInfo                      | Multiple button press is required for wakeup: disabled(0) or enabled(1).
 * InternalBatteryLotNumber              | FieldType_String      | InternalBatteryLotNumberInfo        | Battery lot number.
 * LeftControllerSerialNumber            | FieldType_String      | LeftControllerSerialNumberInfo      | Left controller serial number.
 * RightControllerSerialNumber           | FieldType_String      | RightControllerSerialNumberInfo     | Right controller serial number.
 * NotifyInGameDownloadCompletionFlag    | FieldType_Bool        | NotificationInfo                    | Notify about completed downloads: disabled(0) or enabled(1).
 * NotificationSoundFlag                 | FieldType_Bool        | NotificationInfo                    | Play notification sounds: disabled(0) or enabled(1).
 * TVResolutionSetting                   | FieldType_String      | TVInfo                              | Current TV resolution.
 * RGBRangeSetting                       | FieldType_String      | TVInfo                              | RGB range.
 * ReduceScreenBurnFlag                  | FieldType_Bool        | TVInfo                              | Dim screen if not used: disabled(0) or enabled(1).
 * TVAllowsCecFlag                       | FieldType_Bool        | TVInfo                              | Turn on/off TV when concole is turned on/off: disabled(0) or enabled(1).
 * HandheldModeTimeToScreenSleep         | FieldType_String      | SleepInfo                           | Idle time (minutes) after which to enter sleep mode.
 * ConsoleModeTimeToScreenSleep          | FieldType_String      | SleepInfo                           | Idle time (minutes) after which to enter sleep mode in docked state.
 * StopAutoSleepDuringContentPlayFlag    | FieldType_Bool        | SleepInfo                           | Disable auto sleep: disabled(0) or enabled(1).
 * LastConnectionTestDownloadSpeed       | FieldType_NumericU32  | ConnectionInfo                      | Communication speed during connection testing.
 * LastConnectionTestUploadSpeed         | FieldType_NumericU32  | ConnectionInfo                      | Communication speed during connection testing.
 * DEPRECATED_ServerFQDN                 | FieldType_String      | NetworkErrorInfo                    | (Do not use this. Use ServerFqdn or create another field.)
 * HTTPRequestContents                   | FieldType_String      | NetworkErrorInfo                    | HTTP request at the time of network error.
 * HTTPRequestResponseContents           | FieldType_String      | NetworkErrorInfo                    | HTTP response at the time of network error.
 * EdgeServerIPAddress                   | FieldType_String      | NetworkErrorInfo                    | CDN IP address at the time of error.
 * CDNContentPath                        | FieldType_String      | NetworkErrorInfo                    | CDN content path at the time of error.
 * FileAccessPath                        | FieldType_String      | FileAccessPathInfo                  | File path at the time of error.
 * GameCardCID                           | FieldType_U8Array     | GameCardCIDInfo                     | Game card HW info.
 * NANDCID                               | FieldType_U8Array     | NANDCIDInfo                         | NAND HW info.
 * MicroSDCID                            | FieldType_U8Array     | MicroSDCIDInfo                      | Micro SD HW info.
 * NANDSpeedMode                         | FieldType_String      | NANDSpeedModeInfo                   | NAND speed mode.
 * MicroSDSpeedMode                      | FieldType_String      | MicroSDSpeedModeInfo                | Micro SD speed mode.
 * GameCardSpeedMode                     | FieldType_String      | GameCardSpeedModeInfo               | Game card speed mode.
 * UserAccountInternalID                 | FieldType_String      | UserAccountInternalIDInfo           | User account ID.
 * NetworkServiceAccountInternalID       | FieldType_String      | NetworkServiceAccountInternalIDInfo | Network account ID.
 * NintendoAccountInternalID             | FieldType_String      | NintendoAccountInternalIDInfo       | Nintendo account ID.
 * USB3AvailableFlag                     | FieldType_Bool        | USB3AvailableInfo                   | USB3 available: no(0), yes(1).
 * CallStack                             | FieldType_String      | CallStackInfo                       | Crash call stack.
 * SystemStartupLog                      | FieldType_String      | SystemStartupLogInfo                | Startup log.
 * RegionSetting                         | FieldType_String      | RegionSettingInfo                   | Region info.
 * NintendoZoneConnectedFlag             | FieldType_Bool        | NintendoZoneConnectedInfo           | Connected to Nintendo Zone (1) or not (0).
 * ForcedSleepHighTemperatureReading     | FieldType_NumericU32  | ForceSleepInfo                      | Temperature reading at the time of forced sleep.
 * ForcedSleepFanSpeedReading            | FieldType_NumericU32  | ForceSleepInfo                      | Fan speed reading at the time of forced sleep.
 * ForcedSleepHWInfo                     | FieldType_String      | ForceSleepInfo                      | Other HW information at the time of forced sleep.
 * AbnormalPowerStateInfo                | FieldType_NumericU32  | ChargerInfo                         | Charger related info.
 * ScreenBrightnessLevel                 | FieldType_String      | ScreenBrightnessInfo                | Screen brightness.
 * ProgramId                             | FieldType_String      | ErrorInfo                           | Id of the process that caused the error.
 * AbortFlag                             | FieldType_Bool        | ErrorInfo                           | If set, error is due to abort.
 * ReportVisibilityFlag                  | FieldType_Bool        | ErrorInfoAuto                       | If set, report is visible to the user
 * FatalFlag                             | FieldType_Bool        | ErrorInfo                           | If set, report is generated by fatal process.
 * OccurrenceTimestampNet                | FieldType_NumericI64  | ErrorInfoAuto                       | Network based timestamp, auto filled by error report service.
 * ResultBacktrace                       | FieldType_U32Array    | ErrorInfo                           | Backtrace of results.
 * GeneralRegisterAarch32                | FieldType_U32Array    | ErrorInfo                           | Register dump for aarch32.
 * StackBacktrace32                      | FieldType_U32Array    | ErrorInfo                           | Backtrace of stack.
 * ExceptionInfoAarch32                  | FieldType_U32Array    | ErrorInfo                           | Exception information for aarch32.
 * GeneralRegisterAarch64                | FieldType_U64Array    | ErrorInfo                           | Register dump for aarch64.
 * StackBacktrace64                      | FieldType_U64Array    | ErrorInfo                           | Backtrace of stack.
 * ExceptionInfoAarch64                  | FieldType_U64Array    | ErrorInfo                           | Exception information for aarch64.
 * RegisterSetFlag32                     | FieldType_NumericU32  | ErrorInfo                           | If bit[N] is set, GeneralRegisterAarch32 value[N] is valid.
 * RegisterSetFlag64                     | FieldType_NumericU64  | ErrorInfo                           | If bit[N] is set, GeneralRegisterAarch64 value[N] is valid.
 * ProgramMappedAddr32                   | FieldType_NumericU32  | ErrorInfo                           | Address of program entry point.
 * ProgramMappedAddr64                   | FieldType_NumericU64  | ErrorInfo                           | Address of program entry point.
 * AbortType                             | FieldType_NumericU32  | ErrorInfo                           | 0 = NN_ABORT called. Other value = exception occured.
 * PrivateOsVersion                      | FieldType_String      | ErrorInfoAuto                       | Detailed system firmware version, auto filled by error report service.
 * CurrentSystemPowerState               | FieldType_NumericU32  | SystemPowerStateInfo                | Current system power state.
 * PreviousSystemPowerState              | FieldType_NumericU32  | SystemPowerStateInfo                | Previous system power state.
 * DestinationSystemPowerState           | FieldType_NumericU32  | SystemPowerStateInfo                | Destination system power state. 0 if not on transition
 * PscTransitionCurrentState             | FieldType_NumericU32  | ErrorInfo                           | Destination PmState of the last transition.
 * PscTransitionPreviousState            | FieldType_NumericU32  | ErrorInfo                           | Starting PmState of the last transition.
 * PscInitializedList                    | FieldType_U8Array     | ErrorInfo                           | Module registration status for each PmModuleId.
 * PscCurrentPmStateList                 | FieldType_U32Array    | ErrorInfo                           | Current PmState for each PmModuleId.
 * PscNextPmStateList                    | FieldType_U32Array    | ErrorInfo                           | Destination PmState for each PmModuleId.
 * PerformanceMode                       | FieldType_NumericI32  | PerformanceInfo                     | Performance mode.
 * PerformanceConfiguration              | FieldType_NumericU32  | PerformanceInfo                     | Performance configuration.
 * Throttled                             | FieldType_Bool        | ThrottlingInfo                      | Throttled(True) or not(False).
 * ThrottlingDuration                    | FieldType_NumericI64  | ThrottlingInfo                      | The last throttling duration(nano seconds).
 * ThrottlingTimestamp                   | FieldType_NumericI64  | ThrottlingInfo                      | Timestamp of ThrottlingInfo.
 * GameCardCrcErrorCount                 | FieldType_NumericU32  | GameCardErrorInfo                   | Crc error count between asic and gamecard.
 * GameCardAsicCrcErrorCount             | FieldType_NumericU32  | GameCardErrorInfo                   | Crc error count between SoC and asic.
 * GameCardRefreshCount                  | FieldType_NumericU32  | GameCardErrorInfo                   | Gamecard refresh count.
 * GameCardReadRetryCount                | FieldType_NumericU32  | GameCardErrorInfo                   | Read retry count to gamecard.
 * EdidBlock                             | FieldType_U8Array     | EdidInfo                            | EDID Block.
 * EdidExtensionBlock                    | FieldType_U8Array     | EdidInfo                            | EDID Extension Block.
 * CreateProcessFailureFlag              | FieldType_Bool        | ErrorInfo                           | If set, report is due to failure of process creation.
 * TemperaturePcb                        | FieldType_NumericI32  | ThermalInfo                         | Temperature of PCB.
 * TemperatureSoc                        | FieldType_NumericI32  | ThermalInfo                         | Temperature of SoC.
 * CurrentFanDuty                        | FieldType_NumericI32  | ThermalInfo                         | Current duty rate of fan.
 * LastDvfsThresholdTripped              | FieldType_NumericI32  | ThermalInfo                         | Last DVFS threshold tripped (applicable DVFS thermal range).
 * CradlePdcHFwVersion                   | FieldType_NumericU32  | CradleFirmwareInfo                  | Firmware version of cradle PDC-H.
 * CradlePdcAFwVersion                   | FieldType_NumericU32  | CradleFirmwareInfo                  | Firmware version of cradle PDC-A.
 * CradleMcuFwVersion                    | FieldType_NumericU32  | CradleFirmwareInfo                  | Firmware version of cradle MCU.
 * CradleDp2HdmiFwVersion                | FieldType_NumericU32  | CradleFirmwareInfo                  | Firmware version of cradle DP2HDMI.
 * RunningApplicationId                  | FieldType_String      | RunningApplicationInfo              | Title ID of the application running at the time of error.
 * RunningApplicationTitle               | FieldType_String      | RunningApplicationInfo              | Name of the application running at the time of error.
 * RunningApplicationVersion             | FieldType_String      | RunningApplicationInfo              | Version of the application running at the time of error.
 * RunningApplicationStorageLocation     | FieldType_String      | RunningApplicationInfo              | Storage location of the application running at the time of error.
 * RunningAppletList                     | FieldType_U64Array    | RunningAppletInfo                   | List of applet running at the time of error.
 * FocusedAppletHistory                  | FieldType_U64Array    | FocusedAppletHistoryInfo            | History of foreground applet.
 * CompositorState                       | FieldType_String      | CompositorStateInfo                 | Dump of general process state of nvnflinger.
 * CompositorLayerState                  | FieldType_String      | CompositorLayerInfo                 | Dump of the layers.
 * CompositorDisplayState                | FieldType_String      | CompositorDisplayInfo               | Dump of the displays.
 * CompositorHWCState                    | FieldType_String      | CompositorHWCInfo                   | Dump of hwcomposer state.
 * InputCurrentLimit                     | FieldType_NumericI32  | BatteryChargeInfo                   | Input current limit(milli ampere).
 * BoostModeCurrentLimit                 | FieldType_NumericI32  | BatteryChargeInfo                   | Output current limit(milli ampere).
 * FastChargeCurrentLimit                | FieldType_NumericI32  | BatteryChargeInfo                   | Fast charge current limit(milli ampere).
 * ChargeVoltageLimit                    | FieldType_NumericI32  | BatteryChargeInfo                   | Charge voltage limit(milli voltage).
 * ChargeConfiguration                   | FieldType_NumericI32  | BatteryChargeInfo                   | Charge configuration.
 * HizMode                               | FieldType_Bool        | BatteryChargeInfo                   | Hi-Z mode.
 * ChargeEnabled                         | FieldType_Bool        | BatteryChargeInfo                   | nCE status.
 * PowerSupplyPath                       | FieldType_NumericI32  | BatteryChargeInfo                   | Power supply path.
 * BatteryTemperature                    | FieldType_NumericI32  | BatteryChargeInfo                   | Battery temperature(milli celsius).
 * BatteryChargePercent                  | FieldType_NumericI32  | BatteryChargeInfo                   | Raw value of battery charge percentage(percent * 1000).
 * BatteryChargeVoltage                  | FieldType_NumericI32  | BatteryChargeInfo                   | Battery charge milli voltage.
 * BatteryAge                            | FieldType_NumericI32  | BatteryChargeInfo                   | Battery age percentage(percent * 1000).
 * PowerRole                             | FieldType_NumericI32  | BatteryChargeInfo                   | Power role.
 * PowerSupplyType                       | FieldType_NumericI32  | BatteryChargeInfo                   | Power supply type.
 * PowerSupplyVoltage                    | FieldType_NumericI32  | BatteryChargeInfo                   | Power supply voltage.
 * PowerSupplyCurrent                    | FieldType_NumericI32  | BatteryChargeInfo                   | Power supply current.
 * FastBatteryChargingEnabled            | FieldType_Bool        | BatteryChargeInfo                   | Fast battery charging was enabled.
 * ControllerPowerAcquired               | FieldType_Bool        | BatteryChargeInfo                   | Controller power was acquired.
 * OtgRequested                          | FieldType_Bool        | BatteryChargeInfo                   | OTG was requested.
 * NANDPreEolInfo                        | FieldType_NumericU32  | NANDExtendedCsd                     | NAND Pre EOL information.
 * NANDDeviceLifeTimeEstTypA             | FieldType_NumericU32  | NANDExtendedCsd                     | NAND Device life time estimation type A.
 * NANDDeviceLifeTimeEstTypB             | FieldType_NumericU32  | NANDExtendedCsd                     | NAND Device life time estimation type B.
 * NANDPatrolCount                       | FieldType_NumericU32  | NANDPatrolInfo                      | NAND Patrol count.
 * NANDNumActivationFailures             | FieldType_NumericU32  | NANDErrorInfo                       | number of NAND activation failures.
 * NANDNumActivationErrorCorrections     | FieldType_NumericU32  | NANDErrorInfo                       | number of NAND activation error corrections.
 * NANDNumReadWriteFailures              | FieldType_NumericU32  | NANDErrorInfo                       | number of NAND read/write failures.
 * NANDNumReadWriteErrorCorrections      | FieldType_NumericU32  | NANDErrorInfo                       | number of NAND read/write error corrections.
 * NANDErrorLog                          | FieldType_String      | NANDDriverLog                       | NAND driver error log.
 * SdCardUserAreaSize                    | FieldType_NumericI64  | SdCardSizeSpec                      | SD Card User Area Size.
 * SdCardProtectedAreaSize               | FieldType_NumericI64  | SdCardSizeSpec                      | SD Card Protected Area Size.
 * SdCardNumActivationFailures           | FieldType_NumericU32  | SdCardErrorInfo                     | number of SD Card activation failures.
 * SdCardNumActivationErrorCorrections   | FieldType_NumericU32  | SdCardErrorInfo                     | number of SD Card activation error corrections.
 * SdCardNumReadWriteFailures            | FieldType_NumericU32  | SdCardErrorInfo                     | number of SD Card read/write failures.
 * SdCardNumReadWriteErrorCorrections    | FieldType_NumericU32  | SdCardErrorInfo                     | number of SD Card read/write error corrections.
 * SdCardErrorLog                        | FieldType_String      | SdCardDriverLog                     | SD Card driver error log.
 * EncryptionKey                         | FieldType_U8Array     | ErrorInfo                           | Key info encrypted by RSA.
 * EncryptedExceptionInfo                | FieldType_U8Array     | ErrorInfo                           | Encrypted Exception information[0].
 * EncryptedExceptionInfo1               | FieldType_U8Array     | ErrorInfo                           | Encrypted Exception information[1].
 * EncryptedExceptionInfo2               | FieldType_U8Array     | ErrorInfo                           | Encrypted Exception information[2].
 * EncryptedExceptionInfo3               | FieldType_U8Array     | ErrorInfo                           | Encrypted Exception information[3].
 * EncryptedDyingMessage                 | FieldType_U8Array     | ErrorInfo                           | Encrypted Dying Message.
 * GameCardTimeoutRetryErrorCount        | FieldType_NumericU32  | GameCardErrorInfo                   | GameCard Timeout And Retry Count.
 * FsRemountForDataCorruptCount          | FieldType_NumericU32  | FsProxyErrorInfo                    | FileSytem remount count by data corruption.
 * FsRemountForDataCorruptRetryOutCount  | FieldType_NumericU32  | FsProxyErrorInfo                    | FileSytem remount retry out count by data corruption.
 * GameCardInsertionCount                | FieldType_NumericU32  | GameCardErrorInfo                   | Count of GameCard insertion.
 * GameCardRemovalCount                  | FieldType_NumericU32  | GameCardErrorInfo                   | Count of GameCard removal.
 * GameCardAsicInitializeCount           | FieldType_NumericU32  | GameCardErrorInfo                   | Count of GameCard Asic Initialize.
 * TestU16                               | FieldType_NumericU16  | Test                                | Test value.
 * TestU8                                | FieldType_NumericU8   | Test                                | Test value.
 * TestI16                               | FieldType_NumericI16  | Test                                | Test value.
 * TestI8                                | FieldType_NumericI8   | Test                                | Test value.
 * SystemAppletScene                     | FieldType_NumericU8   | SystemAppletSceneInfo               | SystemApplet scene (HOME Menu / Settings / News).
 * CodecType                             | FieldType_NumericU32  | VideoInfo                           | Type of decoder/encoder used
 * DecodeBuffers                         | FieldType_NumericU32  | VideoInfo                           | Number of buffers required for decoding
 * FrameWidth                            | FieldType_NumericI32  | VideoInfo                           | Width of frame
 * FrameHeight                           | FieldType_NumericI32  | VideoInfo                           | Height of frame
 * ColorPrimaries                        | FieldType_NumericU8   | VideoInfo                           | Color primaries
 * TransferCharacteristics               | FieldType_NumericU8   | VideoInfo                           | TransferCharacteristics
 * MatrixCoefficients                    | FieldType_NumericU8   | VideoInfo                           | Matrix Coefficients
 * DisplayWidth                          | FieldType_NumericI32  | VideoInfo                           | Display widht
 * DisplayHeight                         | FieldType_NumericI32  | VideoInfo                           | Display height
 * DARWidth                              | FieldType_NumericI32  | VideoInfo                           | Display aspect ratio width
 * DARHeight                             | FieldType_NumericI32  | VideoInfo                           | Display aspect ratio height
 * ColorFormat                           | FieldType_NumericU32  | VideoInfo                           | Color format
 * ColorSpace                            | FieldType_String      | VideoInfo                           | Color space
 * SurfaceLayout                         | FieldType_String      | VideoInfo                           | Surface layout of surface
 * BitStream                             | FieldType_U8Array     | VideoInfo                           | Frame data at which error occured
 * VideoState                            | FieldType_String      | VideoInfo                           | Status of video decoder/Encoder
 * VideoLog                              | FieldType_String      | VideoInfo                           | The log of video info
 * GpuErrorChannelId                     | FieldType_NumericU32  | GpuErrorInfo                        | Channel ID in context during GPU error.
 * GpuErrorAruId                         | FieldType_NumericU64  | GpuErrorInfo                        | ARU ID in context during GPU error.
 * GpuErrorType                          | FieldType_NumericU32  | GpuErrorInfo                        | Type of GPU error.
 * GpuErrorFaultInfo                     | FieldType_NumericU32  | GpuErrorInfo                        | GPU MMU fault information.
 * GpuErrorWriteAccess                   | FieldType_Bool        | GpuErrorInfo                        | Write access for the failed GPU MMU error.
 * GpuErrorFaultAddress                  | FieldType_NumericU64  | GpuErrorInfo                        | Faulting address for GPU MMU error.
 * GpuErrorFaultUnit                     | FieldType_NumericU32  | GpuErrorInfo                        | Faulting unit value for GPU MMU error.
 * GpuErrorFaultType                     | FieldType_NumericU32  | GpuErrorInfo                        | Fault type for GPU MMU error.
 * GpuErrorHwContextPointer              | FieldType_NumericU64  | GpuErrorInfo                        | Queue Hardware Context Pointer for GPU MMU error.
 * GpuErrorContextStatus                 | FieldType_NumericU32  | GpuErrorInfo                        | GPU scheduling error context status.
 * GpuErrorPbdmaIntr                     | FieldType_NumericU32  | GpuErrorInfo                        | GPU PBDMA error interrupt.
 * GpuErrorPbdmaErrorType                | FieldType_NumericU32  | GpuErrorInfo                        | GPU PBDMA error type information.
 * GpuErrorPbdmaHeaderShadow             | FieldType_NumericU32  | GpuErrorInfo                        | GPU PBDMA header shadow information.
 * GpuErrorPbdmaHeader                   | FieldType_NumericU32  | GpuErrorInfo                        | GPU PBDMA header information.
 * GpuErrorPbdmaGpShadow0                | FieldType_NumericU32  | GpuErrorInfo                        | GPU PBDMA GP shadow 0 information.
 * GpuErrorPbdmaGpShadow1                | FieldType_NumericU32  | GpuErrorInfo                        | GPU PBDMA GP shadow 1 information.
 * ThreadName                            | FieldType_String      | ErrorInfo                           | Name of error reporting thread.
 * AdspExceptionRegisters                | FieldType_U32Array    | AdspErrorInfo                       | Adsp registers (r0-r13) at the time of the crash.
 * AdspExceptionSpsr                     | FieldType_NumericU32  | AdspErrorInfo                       | Adsp spsr at time of the crash.
 * AdspExceptionProgramCounter           | FieldType_NumericU32  | AdspErrorInfo                       | Adsp program counter at time of the crash.
 * AdspExceptionLinkRegister             | FieldType_NumericU32  | AdspErrorInfo                       | Adsp link register at time of the crash.
 * AdspExceptionStackPointer             | FieldType_NumericU32  | AdspErrorInfo                       | Adsp stack pointer at time of the crash.
 * AdspExceptionArmModeRegisters         | FieldType_U32Array    | AdspErrorInfo                       | Adsp Arm mode registers at time of the crash.
 * AdspExceptionStackAddress             | FieldType_NumericU32  | AdspErrorInfo                       | Adsp stack address at time of the crash.
 * AdspExceptionStackDump                | FieldType_U32Array    | AdspErrorInfo                       | Adsp stack dump at time of the crash.
 * AdspExceptionReason                   | FieldType_NumericU32  | AdspErrorInfo                       | Adsp exception reason.
 * OscillatorClock                       | FieldType_NumericU32  | PowerClockInfo                      | Clock frequency of oscillator (Hz).
 * CpuDvfsTableClocks                    | FieldType_U32Array    | PowerClockInfo                      | Clock data of CPU DVFS table.
 * CpuDvfsTableVoltages                  | FieldType_I32Array    | PowerClockInfo                      | Voltage data of CPU DVFS table.
 * GpuDvfsTableClocks                    | FieldType_U32Array    | PowerClockInfo                      | Clock data of GPU DVFS table.
 * GpuDvfsTableVoltages                  | FieldType_I32Array    | PowerClockInfo                      | Voltage data of GPU DVFS table.
 * EmcDvfsTableClocks                    | FieldType_U32Array    | PowerClockInfo                      | Clock data of EMC DVFS table.
 * EmcDvfsTableVoltages                  | FieldType_I32Array    | PowerClockInfo                      | Voltage data of EMC DVFS table.
 * ModuleClockFrequencies                | FieldType_U32Array    | PowerClockInfo                      | Clock frequency of each PCV module.
 * ModuleClockEnableFlags                | FieldType_U8Array     | PowerClockInfo                      | Clock enable flag of each PCV module.
 * ModulePowerEnableFlags                | FieldType_U8Array     | PowerClockInfo                      | Power enable flag of each PCV module.
 * ModuleResetAssertFlags                | FieldType_U8Array     | PowerClockInfo                      | Reset assert flag of each PCV module.
 * ModuleMinimumVoltageClockRates        | FieldType_U32Array    | PowerClockInfo                      | Minimum Voltage clock rate of each PCV module.
 * PowerDomainEnableFlags                | FieldType_U8Array     | PowerClockInfo                      | Enable flag of each Power domain.
 * PowerDomainVoltages                   | FieldType_I32Array    | PowerClockInfo                      | Voltage of each Power domain (micro Volt).
 * FuseInfo                              | FieldType_U32Array    | PowerClockInfo                      | Speedo and IDDQ information in fuse.
 * DramId                                | FieldType_U32Array    | PowerClockInfo                      | Dram Id.
 * GameCardDeviceId                      | FieldType_U8Array     | GameCardCIDInfo                     | Game card HW Device Unique Id.
 * GameCardAsicReinitializeCount         | FieldType_NumericU16  | GameCardErrorInfo                   | Game card Asic Reinitialize Count because of card error.
 * GameCardAsicReinitializeFailureCount  | FieldType_NumericU16  | GameCardErrorInfo                   | Game card Asic Reinitialize Failure Count.
 * GameCardAsicReinitializeFailureDetail | FieldType_NumericU16  | GameCardErrorInfo                   | Result Description of game card Asic Reinitialize Failure.
 * GameCardRefreshSuccessCount           | FieldType_NumericU16  | GameCardErrorInfo                   | Game card refresh success count.
 * GameCardAwakenCount                   | FieldType_NumericU32  | GameCardErrorInfo                   | Game card awaken count.
 * GameCardAwakenFailureCount            | FieldType_NumericU16  | GameCardErrorInfo                   | Game card awaken failure count.
 * GameCardReadCountFromInsert           | FieldType_NumericU32  | GameCardErrorInfo                   | Game card read count from game card inserted.
 * GameCardReadCountFromAwaken           | FieldType_NumericU32  | GameCardErrorInfo                   | Game card read count from game card awaken.
 * GameCardLastReadErrorPageAddress      | FieldType_NumericU32  | GameCardErrorInfo                   | Game Card Page Address last error occured.
 * GameCardLastReadErrorPageCount        | FieldType_NumericU32  | GameCardErrorInfo                   | Game Card Page Count last error occured.
 * AppletManagerContextTrace             | FieldType_I32Array    | ErrorInfo                           | Context trace of Applet Manager.
 * NvDispIsRegistered                    | FieldType_Bool        | NvDispDeviceInfo                    | NvDisp Registration status.
 * NvDispIsSuspend                       | FieldType_Bool        | NvDispDeviceInfo                    | NvDisp Suspend/resume status.
 * NvDispDC0SurfaceNum                   | FieldType_I32Array    | NvDispDeviceInfo                    | NvDisp Surface Number on DC0.
 * NvDispDC1SurfaceNum                   | FieldType_I32Array    | NvDispDeviceInfo                    | NvDisp Surface Number on DC1.
 * NvDispWindowSrcRectX                  | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Source Rectangle position X.
 * NvDispWindowSrcRectY                  | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Source Rectangle position Y.
 * NvDispWindowSrcRectWidth              | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Source Rectangle width.
 * NvDispWindowSrcRectHeight             | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Source Rectangle height.
 * NvDispWindowDstRectX                  | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Destination Rectangle position X.
 * NvDispWindowDstRectY                  | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Destination Rectangle position Y.
 * NvDispWindowDstRectWidth              | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Destination Rectangle width.
 * NvDispWindowDstRectHeight             | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp Destination Rectangle height.
 * NvDispWindowIndex                     | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp DC Window index.
 * NvDispWindowBlendOperation            | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp DC Window Blend Operation status.
 * NvDispWindowAlphaOperation            | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp DC Window Alpha Operation status.
 * NvDispWindowDepth                     | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp DC Window Depth.
 * NvDispWindowAlpha                     | FieldType_NumericU8   | NvDispDcWindowInfo                  | NvDisp DC Window Alpha.
 * NvDispWindowHFilter                   | FieldType_Bool        | NvDispDcWindowInfo                  | NvDisp DC Window Horizontal Filtering.
 * NvDispWindowVFilter                   | FieldType_Bool        | NvDispDcWindowInfo                  | NvDisp DC Window Vertical Filtering.
 * NvDispWindowOptions                   | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp DC Window Options.
 * NvDispWindowSyncPointId               | FieldType_NumericU32  | NvDispDcWindowInfo                  | NvDisp DC Window Sync point Id.
 * NvDispDPSorPower                      | FieldType_Bool        | NvDispDpModeInfo                    | NvDisp DisplayPort Power status.
 * NvDispDPClkType                       | FieldType_NumericU8   | NvDispDpModeInfo                    | NvDisp DisplayPort Clock Type.
 * NvDispDPEnable                        | FieldType_NumericU32  | NvDispDpModeInfo                    | NvDisp DisplayPort Enable state.
 * NvDispDPState                         | FieldType_NumericU32  | NvDispDpModeInfo                    | NvDisp DisplayPort State.
 * NvDispDPEdid                          | FieldType_U8Array     | NvDispDpModeInfo                    | NvDisp DisplayPort Edid.
 * NvDispDPEdidSize                      | FieldType_NumericU32  | NvDispDpModeInfo                    | NvDisp DisplayPort Edid size.
 * NvDispDPEdidExtSize                   | FieldType_NumericU32  | NvDispDpModeInfo                    | NvDisp DisplayPort Edid Extsize.
 * NvDispDPFakeMode                      | FieldType_Bool        | NvDispDpModeInfo                    | NvDisp DisplayPort Fake Mode state.
 * NvDispDPModeNumber                    | FieldType_NumericU32  | NvDispDpModeInfo                    | NvDisp DisplayPort Number of modes.
 * NvDispDPPlugInOut                     | FieldType_Bool        | NvDispDpModeInfo                    | NvDisp Plug In Out.
 * NvDispDPAuxIntHandler                 | FieldType_Bool        | NvDispDpModeInfo                    | NvDisp DP AUX Interrupt Handler.
 * NvDispDPForceMaxLinkBW                | FieldType_Bool        | NvDispDpModeInfo                    | NvDisp DP Forcible Max Link BandWidth.
 * NvDispDPIsConnected                   | FieldType_Bool        | NvDispDpModeInfo                    | NvDisp DisplayPort Connected.
 * NvDispDPLinkValid                     | FieldType_Bool        | NvDispDpLinkSpec                    | NvDisp DisplayPort Link config state.
 * NvDispDPLinkMaxBW                     | FieldType_NumericU8   | NvDispDpLinkSpec                    | NvDisp DisplayPort Link Max BandWidth.
 * NvDispDPLinkMaxLaneCount              | FieldType_NumericU8   | NvDispDpLinkSpec                    | NvDisp DisplayPort Link Max Lane Count.
 * NvDispDPLinkDownSpread                | FieldType_Bool        | NvDispDpLinkSpec                    | NvDisp DisplayPort Link Down Spread.
 * NvDispDPLinkSupportEnhancedFraming    | FieldType_Bool        | NvDispDpLinkSpec                    | NvDisp DisplayPort Link Support Enhanced Framing.
 * NvDispDPLinkBpp                       | FieldType_NumericU32  | NvDispDpLinkSpec                    | NvDisp DisplayPort Link Bits per pixel.
 * NvDispDPLinkScaramberCap              | FieldType_Bool        | NvDispDpLinkSpec                    | NvDisp DisplayPort Scramber Reset Cap.
 * NvDispDPLinkBW                        | FieldType_NumericU8   | NvDispDpLinkStatus                  | NvDisp DisplayPort Link BandWidth.
 * NvDispDPLinkLaneCount                 | FieldType_NumericU8   | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Lane Count.
 * NvDispDPLinkEnhancedFraming           | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Enhanced Frame.
 * NvDispDPLinkScrambleEnable            | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Scramble Enable.
 * NvDispDPLinkActivePolarity            | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Active Polarity.
 * NvDispDPLinkActiveCount               | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Active Count.
 * NvDispDPLinkTUSize                    | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link TU size.
 * NvDispDPLinkActiveFrac                | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Active Frac.
 * NvDispDPLinkWatermark                 | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Watermark.
 * NvDispDPLinkHBlank                    | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link H Blank.
 * NvDispDPLinkVBlank                    | FieldType_NumericU32  | NvDispDpLinkStatus                  | NvDisp DisplayPort Link V Blank.
 * NvDispDPLinkOnlyEnhancedFraming       | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Only Enhanced Framing.
 * NvDispDPLinkOnlyEdpCap                | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Only Edp Cap.
 * NvDispDPLinkSupportFastLT             | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Support Fast LT.
 * NvDispDPLinkLTDataValid               | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link LT Data Valid.
 * NvDispDPLinkTsp3Support               | FieldType_Bool        | NvDispDpLinkStatus                  | NvDisp DisplayPort Link Tsp3 Support.
 * NvDispDPLinkAuxInterval               | FieldType_NumericU8   | NvDispDpLinkStatus                  | NvDisp DisplayPort AUX interval.
 * NvDispHdcpCreated                     | FieldType_Bool        | NvDispDpHdcpInfo                    | NvDisp HDCP Is Created.
 * NvDispHdcpUserRequest                 | FieldType_Bool        | NvDispDpHdcpInfo                    | NvDisp HDCP User request.
 * NvDispHdcpPlugged                     | FieldType_Bool        | NvDispDpHdcpInfo                    | NvDisp HDCP plugged.
 * NvDispHdcpState                       | FieldType_NumericU32  | NvDispDpHdcpInfo                    | NvDisp HDCP State.
 * NvDispHdcpFailCount                   | FieldType_NumericI32  | NvDispDpHdcpInfo                    | NvDisp HDCP Failure count.
 * NvDispHdcpHdcp22                      | FieldType_NumericI8   | NvDispDpHdcpInfo                    | NvDisp HDCP HDCP2.2 support.
 * NvDispHdcpMaxRetry                    | FieldType_NumericU8   | NvDispDpHdcpInfo                    | NvDisp HDCP Max Retry.
 * NvDispHdcpHpd                         | FieldType_NumericU8   | NvDispDpHdcpInfo                    | NvDisp HDCP HPD.
 * NvDispHdcpRepeater                    | FieldType_NumericU8   | NvDispDpHdcpInfo                    | NvDisp HDCP Repeater.
 * NvDispCecRxBuf                        | FieldType_U8Array     | NvDispDpAuxCecInfo                  | NvDisp HDCP CEC Rx Buffer.
 * NvDispCecRxLength                     | FieldType_NumericI32  | NvDispDpAuxCecInfo                  | NvDisp Cec Rx Length.
 * NvDispCecTxBuf                        | FieldType_U8Array     | NvDispDpAuxCecInfo                  | NvDisp HDCP CEC Tx Buffer.
 * NvDispCecTxLength                     | FieldType_NumericI32  | NvDispDpAuxCecInfo                  | NvDisp Cec Tx Length.
 * NvDispCecTxRetry                      | FieldType_NumericI32  | NvDispDpAuxCecInfo                  | NvDisp Cec Tx Retry.
 * NvDispCecState                        | FieldType_NumericU32  | NvDispDpAuxCecInfo                  | NvDisp Cec State.
 * NvDispCecTxInfo                       | FieldType_NumericU8   | NvDispDpAuxCecInfo                  | NvDisp Cec Tx Info.
 * NvDispCecRxInfo                       | FieldType_NumericU8   | NvDispDpAuxCecInfo                  | NvDisp Cec Rx Info.
 * NvDispDCIndex                         | FieldType_NumericU32  | NvDispDcInfo                        | NvDisp Display Controller Index.
 * NvDispDCInitialize                    | FieldType_Bool        | NvDispDcInfo                        | NvDisp Display Controller Initialized.
 * NvDispDCClock                         | FieldType_Bool        | NvDispDcInfo                        | NvDisp Display Controller Clock.
 * NvDispDCFrequency                     | FieldType_NumericU32  | NvDispDcInfo                        | NvDisp Display Controller Frequency.
 * NvDispDCFailed                        | FieldType_Bool        | NvDispDcInfo                        | NvDisp Display Controller Fail.
 * NvDispDCModeWidth                     | FieldType_NumericI32  | NvDispDcInfo                        | NvDisp Display Controller Mode Width.
 * NvDispDCModeHeight                    | FieldType_NumericI32  | NvDispDcInfo                        | NvDisp Display Controller Mode Height.
 * NvDispDCModeBpp                       | FieldType_NumericU32  | NvDispDcInfo                        | NvDisp Display Controller Bit per pix.
 * NvDispDCPanelFrequency                | FieldType_NumericU32  | NvDispDcInfo                        | NvDisp Display Controller Panel Frequency.
 * NvDispDCWinDirty                      | FieldType_NumericU32  | NvDispDcInfo                        | NvDisp Display Controller Win Dirty.
 * NvDispDCWinEnable                     | FieldType_NumericU32  | NvDispDcInfo                        | NvDisp Display Controller Win Enable.
 * NvDispDCVrr                           | FieldType_Bool        | NvDispDcInfo                        | NvDisp Display Controller VRR.
 * NvDispDCPanelInitialize               | FieldType_Bool        | NvDispDcInfo                        | NvDisp Display Controller Panel Initialize.
 * NvDispDsiDataFormat                   | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Data format.
 * NvDispDsiVideoMode                    | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Video Mode.
 * NvDispDsiRefreshRate                  | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Refresh Rate.
 * NvDispDsiLpCmdModeFrequency           | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI LP CMD Mode Frequency.
 * NvDispDsiHsCmdModeFrequency           | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI HS CMD Mode Frequency.
 * NvDispDsiPanelResetTimeout            | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Panel Reset Timeout.
 * NvDispDsiPhyFrequency                 | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Phy Frequency.
 * NvDispDsiFrequency                    | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Frequency.
 * NvDispDsiInstance                     | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DSI Instance.
 * NvDispDcDsiHostCtrlEnable             | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI Host Controller Enable.
 * NvDispDcDsiInit                       | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI Init.
 * NvDispDcDsiEnable                     | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI Enable.
 * NvDispDcDsiHsMode                     | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI HS Mode.
 * NvDispDcDsiVendorId                   | FieldType_U8Array     | NvDispDsiInfo                       | NvDisp DC DSI Vendor ID.
 * NvDispDcDsiLcdVendorNum               | FieldType_NumericU8   | NvDispDsiInfo                       | NvDisp DC DSI Lcd Vendor Number.
 * NvDispDcDsiHsClockControl             | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DC DSI HS Clock Control.
 * NvDispDcDsiEnableHsClockInLpMode      | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI Enable HS Clock In LP Mode.
 * NvDispDcDsiTeFrameUpdate              | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI TE Frame Update.
 * NvDispDcDsiGangedType                 | FieldType_NumericU32  | NvDispDsiInfo                       | NvDisp DC DSI Ganged Type.
 * NvDispDcDsiHbpInPktSeq                | FieldType_Bool        | NvDispDsiInfo                       | NvDisp DC DSI Hbp In Pkt Seq.
 * NvDispErrID                           | FieldType_NumericU32  | NvDispErrIDInfo                     | NvDisp ErrId Info to indicate err place
 * NvDispErrData0                        | FieldType_NumericU32  | NvDispErrIDInfo                     | NvDisp ErrData0
 * NvDispErrData1                        | FieldType_NumericU32  | NvDispErrIDInfo                     | NvDisp ErrData1
 * SdCardMountStatus                     | FieldType_String      | SdCardMountInfo                     | Result value of ns::CheckSdCardMountStatus().
 * SdCardMountUnexpectedResult           | FieldType_String      | SdCardMountInfo                     | Result value of ns::GetLastSdCardMountUnexpectedResult().
 * NANDTotalSize                         | FieldType_NumericU64  | NANDFreeSpaceInfo                   | Total Size of NAND.
 * SdCardTotalSize                       | FieldType_NumericU64  | SDCardFreeSpaceInfo                 | Total Size of Sd Card.
 * ElapsedTimeSinceInitialLaunch         | FieldType_NumericI64  | ErrorInfoAuto                       | Elapsed time (seconds) since the first initial launch settings.
 * ElapsedTimeSincePowerOn               | FieldType_NumericI64  | ErrorInfoAuto                       | Elapsed time (seconds) since power on.
 * ElapsedTimeSinceLastAwake             | FieldType_NumericI64  | ErrorInfoAuto                       | Elapsed time (seconds) since last awake.
 * OccurrenceTick                        | FieldType_NumericI64  | ErrorInfoAuto                       | System tick value.
 * RetailInteractiveDisplayFlag          | FieldType_Bool        | RetailInteractiveDisplayInfo        | True indicates the system is used at a shop for customers to play.
 * FatFsError                            | FieldType_NumericI32  | FsProxyErrorInfo                    | Last FatFs error.
 * FatFsExtraError                       | FieldType_NumericI32  | FsProxyErrorInfo                    | Last FatFs extra error.
 * FatFsErrorDrive                       | FieldType_NumericI32  | FsProxyErrorInfo                    | Last FatFs error drive ID.
 * FatFsErrorName                        | FieldType_String      | FsProxyErrorInfo                    | Last FatFs error name.
 * MonitorManufactureCode                | FieldType_String      | MonitorCapability                   | Monitor vendor name.
 * MonitorProductCode                    | FieldType_NumericU16  | MonitorCapability                   | Monitor product code.
 * MonitorSerialNumber                   | FieldType_NumericU32  | MonitorCapability                   | Monitor serial number.
 * MonitorManufactureYear                | FieldType_NumericI32  | MonitorCapability                   | Monitor Manufacture year.
 * PhysicalAddress                       | FieldType_NumericU16  | MonitorCapability                   | Physical address of NX in HDMI topology.
 * Is4k60Hz                              | FieldType_Bool        | MonitorCapability                   | If the monitor support 4k60Hz.
 * Is4k30Hz                              | FieldType_Bool        | MonitorCapability                   | If the monitor support 4k30Hz.
 * Is1080P60Hz                           | FieldType_Bool        | MonitorCapability                   | If the monitor supports 1080p60Hz .
 * Is720P60Hz                            | FieldType_Bool        | MonitorCapability                   | If the monitor supports 720p60Hz.
 * PcmChannelMax                         | FieldType_NumericI32  | MonitorCapability                   | Max Audio channel for PCM format (stereo/surround).
 * MonitorCurrentWidth                   | FieldType_NumericU16  | MonitorSettings                     | Monitor's current resolution width in pixels.
 * MonitorCurrentHeight                  | FieldType_NumericU16  | MonitorSettings                     | Monitor's current resolution height in pixels.
 * MonitorCurrentRefreshRate             | FieldType_String      | MonitorSettings                     | Monitor's current refresh rate in Hz.
 * CrashReportHash                       | FieldType_U8Array     | ErrorInfo                           | Hash digest of crash stack trace.
 * ErrorReportSharePermission            | FieldType_NumericU8   | ErrorReportSharePermissionInfo      | User permission state to share error reports with other companies.
 * VideoCodecTypeEnum                    | FieldType_NumericI32  | MultimediaInfo                      |
 * VideoBitRate                          | FieldType_NumericI32  | MultimediaInfo                      |
 * VideoFrameRate                        | FieldType_NumericI32  | MultimediaInfo                      |
 * VideoWidth                            | FieldType_NumericI32  | MultimediaInfo                      |
 * VideoHeight                           | FieldType_NumericI32  | MultimediaInfo                      |
 * AudioCodecTypeEnum                    | FieldType_NumericI32  | MultimediaInfo                      |
 * AudioSampleRate                       | FieldType_NumericI32  | MultimediaInfo                      |
 * AudioChannelCount                     | FieldType_NumericI32  | MultimediaInfo                      |
 * AudioBitRate                          | FieldType_NumericI32  | MultimediaInfo                      |
 * MultimediaContainerType               | FieldType_NumericI32  | MultimediaInfo                      |
 * MultimediaProfileType                 | FieldType_NumericI32  | MultimediaInfo                      |
 * MultimediaLevelType                   | FieldType_NumericI32  | MultimediaInfo                      |
 * MultimediaCacheSizeEnum               | FieldType_NumericI32  | MultimediaInfo                      |
 * MultimediaErrorStatusEnum             | FieldType_NumericI32  | MultimediaInfo                      |
 * MultimediaErrorLog                    | FieldType_U8Array     | MultimediaInfo                      |
 * ServerFqdn                            | FieldType_String      | ErrorInfo                           | FQDN of the server.
 * ServerIpAddress                       | FieldType_String      | ErrorInfo                           | IP address of the server.
 * TestStringEncrypt                     | FieldType_String      | Test                                | Encrypted test value.
 * TestU8ArrayEncrypt                    | FieldType_U8Array     | Test                                | Encrypted test value.
 * TestU32ArrayEncrypt                   | FieldType_U32Array    | Test                                | Encrypted test value.
 * TestU64ArrayEncrypt                   | FieldType_U64Array    | Test                                | Encrypted test value.
 * TestI32ArrayEncrypt                   | FieldType_I32Array    | Test                                | Encrypted test value.
 * TestI64ArrayEncrypt                   | FieldType_I64Array    | Test                                | Encrypted test value.
 * CipherKey                             | FieldType_U8Array     | ErrorInfoAuto                       | Cipher key used for encryption
 * FileSystemPath                        | FiledType_String      | ErrorInfo                           | (Encrypted) File or directory's path.
 * WebMediaPlayerOpenUrl                 | FieldType_String      | ErrorInfo                           | (Encrypted) URL opened by media player.
 * WebMediaPlayerLastSocketErrors        | FieldType_I32Array    | ErrorInfo                           | Last socket errors occurred while playing web media.
 * UnknownControllerCount                | FieldType_NumericU8   | ConnectedControllerInfo             | The number of controllers connected by unknown interface.
 * AttatchedControllerCount              | FieldType_NumericU8   | ConnectedControllerInfo             | The number of controllers physically attached to the console.
 * BluetoothControllerCount              | FieldType_NumericU8   | ConnectedControllerInfo             | The number of controllers connected by bluetooth.
 * UsbControllerCount                    | FieldType_NumericU8   | ConnectedControllerInfo             | The number of controllers connected by usb.
 * ControllerTypeList                    | FieldType_U8Array     | ConnectedControllerInfo             | The list of controller type.
 * ControllerInterfaceList               | FieldType_U8Array     | ConnectedControllerInfo             | The list of controller interface.
 * ControllerStyleList                   | FieldType_U8Array     | ConnectedControllerInfo             | The list of controller style.
 * FsPooledBufferPeakFreeSize            | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsPooledBufferRetriedCount            | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsPooledBufferReduceAllocationCount   | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsBufferManagerPeakFreeSize           | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsBufferManagerRetriedCount           | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsExpHeapPeakFreeSize                 | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsBufferPoolPeakFreeSize              | FieldType_NumericU64  | FsMemoryInfo                        |
 * FsPatrolReadAllocateBufferSuccessCount| FieldType_NumericU64  | FsMemoryInfo                        |
 * FsPatrolReadAllocateBufferFailureCount| FieldType_NumericU64  | FsMemoryInfo                        |
 * SteadyClockInternalOffset,            | FieldType_NumericI64  | ErrorInfoAuto                       | Standard steady clock's internal offset (seconds)
 * SteadyClockCurrentTimePointValue,     | FieldType_NumericI64  | ErrorInfoAuto                       | Standard steady clock's current timepoint value.
 * UserClockContextOffset                | FieldType_NumericI64  | UserClockContextInfo                | SystemClockContext offset of standard user clock.
 * UserClockContextTimeStampValue        | FieldType_NumericI64  | UserClockContextInfo                | SystemClockContext timestamp value of standard user clock.
 * NetworkClockContextOffset             | FieldType_NumericI64  | NetworkClockContextInfo             | SystemClockContext offset of Standard network clock.
 * NetworkClockContextTimeStampValue     | FieldType_NumericI64  | NetworkClockContextInfo             | SystemClockContext timestamp value of standard network clock.
 * SystemAbortFlag                       | FiledType_Bool        | ErrorInfo                           | If set, error is due to system abort (fatal error).
 * ApplicationAbortFlag                  | FiledType_Bool        | ErrorInfo                           | If set, error is due to application abort.
 * NifmErrorCode                         | FieldType_String      | ConnectionStatusInfo                |
 * LcsApplicationId                      | FieldType_String      | ErrorInfo                           |
 * LcsContentMetaKeyIdList               | FieldType_U64Array    | ErrorInfo                           |
 * LcsContentMetaKeyVersionList          | FieldType_U32Array    | ErrorInfo                           |
 * LcsContentMetaKeyTypeList             | FieldType_U8Array     | ErrorInfo                           |
 * LcsContentMetaKeyInstallTypeList      | FieldType_U8Array     | ErrorInfo                           |
 * LcsSenderFlag                         | FieldType_Bool        | ErrorInfo                           |
 * LcsApplicationRequestFlag             | FieldType_Bool        | ErrorInfo                           |
 * LcsHasExFatDriverFlag                 | FieldType_Bool        | ErrorInfo                           |
 * LcsIpAddress                          | FieldType_NumericU32  | ErrorInfo                           |
 * AcpStartupUserAccount                 | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpAocRegistrationType                | FieldType_NumericU8   | AcpAocSettingsInfo                  |
 * AcpAttributeFlag                      | FieldType_NumericU32  | AcpGeneralSettingsInfo              |
 * AcpSupportedLanguageFlag              | FieldType_NumericU32  | AcpGeneralSettingsInfo              |
 * AcpParentalControlFlag                | FieldType_NumericU32  | AcpGeneralSettingsInfo              |
 * AcpScreenShot                         | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpVideoCapture                       | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpDataLossConfirmation               | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpPlayLogPolicy                      | FieldType_NumericU8   | AcpPlayLogSettingsInfo              |
 * AcpPresenceGroupId                    | FieldType_NumericU64  | AcpGeneralSettingsInfo              |
 * AcpRatingAge                          | FieldType_I8Array     | AcpRatingSettingsInfo               |
 * AcpAocBaseId                          | FieldType_NumericU64  | AcpAocSettingsInfo                  |
 * AcpSaveDataOwnerId                    | FieldType_NumericU64  | AcpStorageSettingsInfo              |
 * AcpUserAccountSaveDataSize            | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpUserAccountSaveDataJournalSize     | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpDeviceSaveDataSize                 | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpDeviceSaveDataJournalSize          | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpBcatDeliveryCacheStorageSize       | FieldType_NumericI64  | AcpBcatSettingsInfo                 |
 * AcpApplicationErrorCodeCategory       | FieldType_String      | AcpGeneralSettingsInfo              |
 * AcpLocalCommunicationId               | FieldType_U64Array    | AcpGeneralSettingsInfo              |
 * AcpLogoType                           | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpLogoHandling                       | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpRuntimeAocInstall                  | FieldType_NumericU8   | AcpAocSettingsInfo                  |
 * AcpCrashReport                        | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpHdcp                               | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * AcpSeedForPseudoDeviceId              | FieldType_NumericU64  | AcpGeneralSettingsInfo              |
 * AcpBcatPassphrase                     | FieldType_String      | AcpBcatSettingsInfo                 |
 * AcpUserAccountSaveDataSizeMax         | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpUserAccountSaveDataJournalSizeMax  | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpDeviceSaveDataSizeMax              | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpDeviceSaveDataJournalSizeMax       | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpTemporaryStorageSize               | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpCacheStorageSize                   | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpCacheStorageJournalSize            | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpCacheStorageDataAndJournalSizeMax  | FieldType_NumericI64  | AcpStorageSettingsInfo              |
 * AcpCacheStorageIndexMax               | FieldType_NumericI16  | AcpStorageSettingsInfo              |
 * AcpPlayLogQueryableApplicationId      | FieldType_U64Array    | AcpPlayLogSettingsInfo              |
 * AcpPlayLogQueryCapability             | FieldType_NumericU8   | AcpPlayLogSettingsInfo              |
 * AcpRepairFlag                         | FieldType_NumericU8   | AcpGeneralSettingsInfo              |
 * RunningApplicationPatchStorageLocation| FieldType_String      | RunningApplicationInfo              |
 * RunningApplicationVersionNumber       | FieldType_NumericU32  | RunningApplicationInfo              |
 * FsRecoveredByInvalidateCacheCount     | FieldType_NumericU32  | FsProxyErrorInfo                    |
 * FsSaveDataIndexCount                  | FieldType_NumericU32  | FsProxyErrorInfo                    |
 * FsBufferManagerPeakTotalAllocatableSize | FieldType_NumericU64| FsMemoryInfo                        |
 * RebootlessSystemUpdateVersion         | FieldType_String      | RebootlessSystemUpdateVersionInfo   |
 * NifmConnectionTestRedirectUrl         | FieldType_String,     | NifmConnectionTestInfo              |
 * AcpRequiredNetworkServiceLicenseOnLaunch | FieldType_NumericU8| AcpGeneralSettingsInfo              |
 * PciePort0Flags                        | FieldType_NumericU32  | PcieLoggedStateInfo                 | port0: Bitmapped set of flags to express state of root port
 * PciePort0Speed                        | FieldType_NumericU8   | PcieLoggedStateInfo                 | port0: Negotiated PCIe link speed
 * PciePort0ResetTimeInUs                | FieldType_NumericU32  | PcieLoggedStateInfo                 | port0: Upon enumeration, how long in us did link take to come up?
 * PciePort0IrqCount                     | FieldType_NumericU32  | PcieLoggedStateInfo                 | port0: Interrupt count
 * PciePort0Statistics                   | FieldType_U32Array    | PcieLoggedStateInfo                 | port0: AFI message statistics
 * PciePort1Flags                        | FieldType_NumericU32  | PcieLoggedStateInfo                 | port1: Bitmapped set of flags to express state of root port
 * PciePort1Speed                        | FieldType_NumericU8   | PcieLoggedStateInfo                 | port1: Negotiated PCIe link speed
 * PciePort1ResetTimeInUs                | FieldType_NumericU32  | PcieLoggedStateInfo                 | port1: Upon enumeration, how long in us did link take to come up?
 * PciePort1IrqCount                     | FieldType_NumericU32  | PcieLoggedStateInfo                 | port1: Interrupt count
 * PciePort1Statistics                   | FieldType_U32Array    | PcieLoggedStateInfo                 | port1: AFI message statistics
 * PcieFunction0VendorId                 | FieldType_NumericU16  | PcieLoggedStateInfo                 | function0: Vendor ID
 * PcieFunction0DeviceId                 | FieldType_NumericU16  | PcieLoggedStateInfo                 | function0: Device ID
 * PcieFunction0PmState                  | FieldType_NumericU8   | PcieLoggedStateInfo                 | function0: Power management state
 * PcieFunction0IsAcquired               | FieldType_Bool        | PcieLoggedStateInfo                 | function0: Is acquired by class driver?
 * PcieFunction1VendorId                 | FieldType_NumericU16  | PcieLoggedStateInfo                 | function1: Vendor ID
 * PcieFunction1DeviceId                 | FieldType_NumericU16  | PcieLoggedStateInfo                 | function1: Device ID
 * PcieFunction1PmState                  | FieldType_NumericU8   | PcieLoggedStateInfo                 | function1: Power management state
 * PcieFunction1IsAcquired               | FieldType_Bool        | PcieLoggedStateInfo                 | function1: Is acquired by class driver?
 * PcieGlobalRootComplexStatistics       | FieldType_U32Array    | PcieLoggedStateInfo                 | Global root complex statistics
 * PciePllResistorCalibrationValue       | FieldType_NumericU8   | PcieLoggedStateInfo                 | PLL resistor calibration value
 */

/*
 * Macro for adding new fields.
 * This macro may be used to declare new fields.
 *          Macro takes four arguments:
 *          - Name (this is the string stored in the report identifying this field),
 *          - Enum number assigned to this field, should be sequential,
 *          - Type of the variable stored under this field ID.
 *          - Category ID that this field belongs to,
 *
 * When adding new fields, please update doxygen comments for @enum nn::erpt::FieldId.
 */
#define FOREACH_ID(ENTRY)\
        ENTRY(TestU64,                              0,      FieldType_NumericU64,       Test)\
        ENTRY(TestU32,                              1,      FieldType_NumericU32,       Test)\
        ENTRY(TestI64,                              2,      FieldType_NumericI64,       Test)\
        ENTRY(TestI32,                              3,      FieldType_NumericI32,       Test)\
        ENTRY(TestString,                           4,      FieldType_String,           Test)\
        ENTRY(TestU8Array,                          5,      FieldType_U8Array,          Test)\
        ENTRY(TestU32Array,                         6,      FieldType_U32Array,         Test)\
        ENTRY(TestU64Array,                         7,      FieldType_U64Array,         Test)\
        ENTRY(TestI32Array,                         8,      FieldType_I32Array,         Test)\
        ENTRY(TestI64Array,                         9,      FieldType_I64Array,         Test)\
        ENTRY(ErrorCode,                            10,     FieldType_String,           ErrorInfo)\
        ENTRY(ErrorDescription,                     11,     FieldType_String,           ErrorInfo)\
        ENTRY(OccurrenceTimestamp,                  12,     FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(ReportIdentifier,                     13,     FieldType_String,           ErrorInfoAuto)\
        ENTRY(ConnectionStatus,                     14,     FieldType_String,           ConnectionStatusInfo)\
        ENTRY(AccessPointSSID,                      15,     FieldType_String,           AccessPointInfo)\
        ENTRY(AccessPointSecurityType,              16,     FieldType_String,           AccessPointInfo)\
        ENTRY(RadioStrength,                        17,     FieldType_NumericU32,       RadioStrengthInfo)\
        ENTRY(NXMacAddress,                         18,     FieldType_String,           NXMacAddressInfo)\
        ENTRY(IPAddressAcquisitionMethod,           19,     FieldType_NumericU32,       NetworkInfo)\
        ENTRY(CurrentIPAddress,                     20,     FieldType_String,           NetworkInfo)\
        ENTRY(SubnetMask,                           21,     FieldType_String,           NetworkInfo)\
        ENTRY(GatewayIPAddress,                     22,     FieldType_String,           NetworkInfo)\
        ENTRY(DNSType,                              23,     FieldType_NumericU32,       NetworkInfo)\
        ENTRY(PriorityDNSIPAddress,                 24,     FieldType_String,           NetworkInfo)\
        ENTRY(AlternateDNSIPAddress,                25,     FieldType_String,           NetworkInfo)\
        ENTRY(UseProxyFlag,                         26,     FieldType_Bool,             NetworkInfo)\
        ENTRY(ProxyIPAddress,                       27,     FieldType_String,           NetworkInfo)\
        ENTRY(ProxyPort,                            28,     FieldType_NumericU32,       NetworkInfo)\
        ENTRY(ProxyAutoAuthenticateFlag,            29,     FieldType_Bool,             NetworkInfo)\
        ENTRY(MTU,                                  30,     FieldType_NumericU32,       NetworkInfo)\
        ENTRY(ConnectAutomaticallyFlag,             31,     FieldType_Bool,             NetworkInfo)\
        ENTRY(UseStealthNetworkFlag,                32,     FieldType_Bool,             StealthNetworkInfo)\
        ENTRY(LimitHighCapacityFlag,                33,     FieldType_Bool,             LimitHighCapacityInfo)\
        ENTRY(NATType,                              34,     FieldType_String,           NATTypeInfo)\
        ENTRY(WirelessAPMacAddress,                 35,     FieldType_String,           WirelessAPMacAddressInfo)\
        ENTRY(GlobalIPAddress,                      36,     FieldType_String,           GlobalIPAddressInfo)\
        ENTRY(EnableWirelessInterfaceFlag,          37,     FieldType_Bool,             EnableWirelessInterfaceInfo)\
        ENTRY(EnableWifiFlag,                       38,     FieldType_Bool,             EnableWifiInfo)\
        ENTRY(EnableBluetoothFlag,                  39,     FieldType_Bool,             EnableBluetoothInfo)\
        ENTRY(EnableNFCFlag,                        40,     FieldType_Bool,             EnableNFCInfo)\
        ENTRY(NintendoZoneSSIDListVersion,          41,     FieldType_String,           NintendoZoneSSIDListVersionInfo)\
        ENTRY(LANAdapterMacAddress,                 42,     FieldType_String,           LANAdapterMacAddressInfo)\
        ENTRY(ApplicationID,                        43,     FieldType_String,           ApplicationInfo)\
        ENTRY(ApplicationTitle,                     44,     FieldType_String,           ApplicationInfo)\
        ENTRY(ApplicationVersion,                   45,     FieldType_String,           ApplicationInfo)\
        ENTRY(ApplicationStorageLocation,           46,     FieldType_String,           ApplicationInfo)\
        ENTRY(DownloadContentType,                  47,     FieldType_String,           OccurrenceInfo)\
        ENTRY(InstallContentType,                   48,     FieldType_String,           OccurrenceInfo)\
        ENTRY(ConsoleStartingUpFlag,                49,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(SystemStartingUpFlag,                 50,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(ConsoleFirstInitFlag,                 51,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(HomeMenuScreenDisplayedFlag,          52,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(DataManagementScreenDisplayedFlag,    53,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(ConnectionTestingFlag,                54,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(ApplicationRunningFlag,               55,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(DataCorruptionDetectedFlag,           56,     FieldType_Bool,             OccurrenceInfo)\
        ENTRY(ProductModel,                         57,     FieldType_String,           ProductModelInfo)\
        ENTRY(CurrentLanguage,                      58,     FieldType_String,           CurrentLanguageInfo)\
        ENTRY(UseNetworkTimeProtocolFlag,           59,     FieldType_Bool,             UseNetworkTimeProtocolInfo)\
        ENTRY(TimeZone,                             60,     FieldType_String,           TimeZoneInfo)\
        ENTRY(ControllerFirmware,                   61,     FieldType_String,           ControllerFirmwareInfo)\
        ENTRY(VideoOutputSetting,                   62,     FieldType_String,           VideoOutputInfo)\
        ENTRY(NANDFreeSpace,                        63,     FieldType_NumericU64,       NANDFreeSpaceInfo)\
        ENTRY(SDCardFreeSpace,                      64,     FieldType_NumericU64,       SDCardFreeSpaceInfo)\
        ENTRY(SerialNumber,                         65,     FieldType_String,           ErrorInfoAuto)\
        ENTRY(OsVersion,                            66,     FieldType_String,           ErrorInfoAuto)\
        ENTRY(ScreenBrightnessAutoAdjustFlag,       67,     FieldType_Bool,             ScreenBrightnessInfo)\
        ENTRY(HdmiAudioOutputMode,                  68,     FieldType_String,           AudioFormatInfo)\
        ENTRY(SpeakerAudioOutputMode,               69,     FieldType_String,           AudioFormatInfo)\
        ENTRY(HeadphoneAudioOutputMode,             70,     FieldType_String,           AudioFormatInfo)\
        ENTRY(MuteOnHeadsetUnpluggedFlag,           71,     FieldType_Bool,             MuteOnHeadsetUnpluggedInfo)\
        ENTRY(NumUserRegistered,                    72,     FieldType_NumericI32,       NumUserRegisteredInfo)\
        ENTRY(StorageAutoOrganizeFlag,              73,     FieldType_Bool,             DataDeletionInfo)\
        ENTRY(ControllerVibrationVolume,            74,     FieldType_String,           ControllerVibrationInfo)\
        ENTRY(LockScreenFlag,                       75,     FieldType_Bool,             LockScreenInfo)\
        ENTRY(InternalBatteryLotNumber,             76,     FieldType_String,           InternalBatteryLotNumberInfo)\
        ENTRY(LeftControllerSerialNumber,           77,     FieldType_String,           LeftControllerSerialNumberInfo)\
        ENTRY(RightControllerSerialNumber,          78,     FieldType_String,           RightControllerSerialNumberInfo)\
        ENTRY(NotifyInGameDownloadCompletionFlag,   79,     FieldType_Bool,             NotificationInfo)\
        ENTRY(NotificationSoundFlag,                80,     FieldType_Bool,             NotificationInfo)\
        ENTRY(TVResolutionSetting,                  81,     FieldType_String,           TVInfo)\
        ENTRY(RGBRangeSetting,                      82,     FieldType_String,           TVInfo)\
        ENTRY(ReduceScreenBurnFlag,                 83,     FieldType_Bool,             TVInfo)\
        ENTRY(TVAllowsCecFlag,                      84,     FieldType_Bool,             TVInfo)\
        ENTRY(HandheldModeTimeToScreenSleep,        85,     FieldType_String,           SleepInfo)\
        ENTRY(ConsoleModeTimeToScreenSleep,         86,     FieldType_String,           SleepInfo)\
        ENTRY(StopAutoSleepDuringContentPlayFlag,   87,     FieldType_Bool,             SleepInfo)\
        ENTRY(LastConnectionTestDownloadSpeed,      88,     FieldType_NumericU32,       ConnectionInfo)\
        ENTRY(LastConnectionTestUploadSpeed,        89,     FieldType_NumericU32,       ConnectionInfo)\
        ENTRY(DEPRECATED_ServerFQDN,                90,     FieldType_String,           NetworkErrorInfo)\
        ENTRY(HTTPRequestContents,                  91,     FieldType_String,           NetworkErrorInfo)\
        ENTRY(HTTPRequestResponseContents,          92,     FieldType_String,           NetworkErrorInfo)\
        ENTRY(EdgeServerIPAddress,                  93,     FieldType_String,           NetworkErrorInfo)\
        ENTRY(CDNContentPath,                       94,     FieldType_String,           NetworkErrorInfo)\
        ENTRY(FileAccessPath,                       95,     FieldType_String,           FileAccessPathInfo)\
        ENTRY(GameCardCID,                          96,     FieldType_U8Array,          GameCardCIDInfo)\
        ENTRY(NANDCID,                              97,     FieldType_U8Array,          NANDCIDInfo)\
        ENTRY(MicroSDCID,                           98,     FieldType_U8Array,          MicroSDCIDInfo)\
        ENTRY(NANDSpeedMode,                        99,     FieldType_String,           NANDSpeedModeInfo)\
        ENTRY(MicroSDSpeedMode,                     100,    FieldType_String,           MicroSDSpeedModeInfo)\
        ENTRY(GameCardSpeedMode,                    101,    FieldType_String,           GameCardSpeedModeInfo)\
        ENTRY(UserAccountInternalID,                102,    FieldType_String,           UserAccountInternalIDInfo)\
        ENTRY(NetworkServiceAccountInternalID,      103,    FieldType_String,           NetworkServiceAccountInternalIDInfo)\
        ENTRY(NintendoAccountInternalID,            104,    FieldType_String,           NintendoAccountInternalIDInfo)\
        ENTRY(USB3AvailableFlag,                    105,    FieldType_Bool,             USB3AvailableInfo)\
        ENTRY(CallStack,                            106,    FieldType_String,           CallStackInfo)\
        ENTRY(SystemStartupLog,                     107,    FieldType_String,           SystemStartupLogInfo)\
        ENTRY(RegionSetting,                        108,    FieldType_String,           RegionSettingInfo)\
        ENTRY(NintendoZoneConnectedFlag,            109,    FieldType_Bool,             NintendoZoneConnectedInfo)\
        ENTRY(ForcedSleepHighTemperatureReading,    110,    FieldType_NumericU32,       ForceSleepInfo)\
        ENTRY(ForcedSleepFanSpeedReading,           111,    FieldType_NumericU32,       ForceSleepInfo)\
        ENTRY(ForcedSleepHWInfo,                    112,    FieldType_String,           ForceSleepInfo)\
        ENTRY(AbnormalPowerStateInfo,               113,    FieldType_NumericU32,       ChargerInfo)\
        ENTRY(ScreenBrightnessLevel,                114,    FieldType_String,           ScreenBrightnessInfo)\
        ENTRY(ProgramId,                            115,    FieldType_String,           ErrorInfo)\
        ENTRY(AbortFlag,                            116,    FieldType_Bool,             ErrorInfo)\
        ENTRY(ReportVisibilityFlag,                 117,    FieldType_Bool,             ErrorInfoAuto)\
        ENTRY(FatalFlag,                            118,    FieldType_Bool,             ErrorInfo)\
        ENTRY(OccurrenceTimestampNet,               119,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(ResultBacktrace,                      120,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(GeneralRegisterAarch32,               121,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(StackBacktrace32,                     122,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(ExceptionInfoAarch32,                 123,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(GeneralRegisterAarch64,               124,    FieldType_U64Array,         ErrorInfo)\
        ENTRY(ExceptionInfoAarch64,                 125,    FieldType_U64Array,         ErrorInfo)\
        ENTRY(StackBacktrace64,                     126,    FieldType_U64Array,         ErrorInfo)\
        ENTRY(RegisterSetFlag32,                    127,    FieldType_NumericU32,       ErrorInfo)\
        ENTRY(RegisterSetFlag64,                    128,    FieldType_NumericU64,       ErrorInfo)\
        ENTRY(ProgramMappedAddr32,                  129,    FieldType_NumericU32,       ErrorInfo)\
        ENTRY(ProgramMappedAddr64,                  130,    FieldType_NumericU64,       ErrorInfo)\
        ENTRY(AbortType,                            131,    FieldType_NumericU32,       ErrorInfo)\
        ENTRY(PrivateOsVersion,                     132,    FieldType_String,           ErrorInfoAuto)\
        ENTRY(CurrentSystemPowerState,              133,    FieldType_NumericU32,       SystemPowerStateInfo)\
        ENTRY(PreviousSystemPowerState,             134,    FieldType_NumericU32,       SystemPowerStateInfo)\
        ENTRY(DestinationSystemPowerState,          135,    FieldType_NumericU32,       SystemPowerStateInfo)\
        ENTRY(PscTransitionCurrentState,            136,    FieldType_NumericU32,       ErrorInfo)\
        ENTRY(PscTransitionPreviousState,           137,    FieldType_NumericU32,       ErrorInfo)\
        ENTRY(PscInitializedList,                   138,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(PscCurrentPmStateList,                139,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(PscNextPmStateList,                   140,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(PerformanceMode,                      141,    FieldType_NumericI32,       PerformanceInfo)\
        ENTRY(PerformanceConfiguration,             142,    FieldType_NumericU32,       PerformanceInfo)\
        ENTRY(Throttled,                            143,    FieldType_Bool,             ThrottlingInfo)\
        ENTRY(ThrottlingDuration,                   144,    FieldType_NumericI64,       ThrottlingInfo)\
        ENTRY(ThrottlingTimestamp,                  145,    FieldType_NumericI64,       ThrottlingInfo)\
        ENTRY(GameCardCrcErrorCount,                146,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardAsicCrcErrorCount,            147,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardRefreshCount,                 148,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardReadRetryCount,               149,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(EdidBlock,                            150,    FieldType_U8Array,          EdidInfo)\
        ENTRY(EdidExtensionBlock,                   151,    FieldType_U8Array,          EdidInfo)\
        ENTRY(CreateProcessFailureFlag,             152,    FieldType_Bool,             ErrorInfo)\
        ENTRY(TemperaturePcb,                       153,    FieldType_NumericI32,       ThermalInfo)\
        ENTRY(TemperatureSoc,                       154,    FieldType_NumericI32,       ThermalInfo)\
        ENTRY(CurrentFanDuty,                       155,    FieldType_NumericI32,       ThermalInfo)\
        ENTRY(LastDvfsThresholdTripped,             156,    FieldType_NumericI32,       ThermalInfo)\
        ENTRY(CradlePdcHFwVersion,                  157,    FieldType_NumericU32,       CradleFirmwareInfo)\
        ENTRY(CradlePdcAFwVersion,                  158,    FieldType_NumericU32,       CradleFirmwareInfo)\
        ENTRY(CradleMcuFwVersion,                   159,    FieldType_NumericU32,       CradleFirmwareInfo)\
        ENTRY(CradleDp2HdmiFwVersion,               160,    FieldType_NumericU32,       CradleFirmwareInfo)\
        ENTRY(RunningApplicationId,                 161,    FieldType_String,           RunningApplicationInfo)\
        ENTRY(RunningApplicationTitle,              162,    FieldType_String,           RunningApplicationInfo)\
        ENTRY(RunningApplicationVersion,            163,    FieldType_String,           RunningApplicationInfo)\
        ENTRY(RunningApplicationStorageLocation,    164,    FieldType_String,           RunningApplicationInfo)\
        ENTRY(RunningAppletList,                    165,    FieldType_U64Array,         RunningAppletInfo)\
        ENTRY(FocusedAppletHistory,                 166,    FieldType_U64Array,         FocusedAppletHistoryInfo)\
        ENTRY(CompositorState,                      167,    FieldType_String,           CompositorStateInfo)\
        ENTRY(CompositorLayerState,                 168,    FieldType_String,           CompositorLayerInfo)\
        ENTRY(CompositorDisplayState,               169,    FieldType_String,           CompositorDisplayInfo)\
        ENTRY(CompositorHWCState,                   170,    FieldType_String,           CompositorHWCInfo)\
        ENTRY(InputCurrentLimit,                    171,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(BoostModeCurrentLimit,                172,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(FastChargeCurrentLimit,               173,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(ChargeVoltageLimit,                   174,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(ChargeConfiguration,                  175,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(HizMode,                              176,    FieldType_Bool,             BatteryChargeInfo)\
        ENTRY(ChargeEnabled,                        177,    FieldType_Bool,             BatteryChargeInfo)\
        ENTRY(PowerSupplyPath,                      178,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(BatteryTemperature,                   179,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(BatteryChargePercent,                 180,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(BatteryChargeVoltage,                 181,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(BatteryAge,                           182,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(PowerRole,                            183,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(PowerSupplyType,                      184,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(PowerSupplyVoltage,                   185,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(PowerSupplyCurrent,                   186,    FieldType_NumericI32,       BatteryChargeInfo)\
        ENTRY(FastBatteryChargingEnabled,           187,    FieldType_Bool,             BatteryChargeInfo)\
        ENTRY(ControllerPowerSupplyAcquired,        188,    FieldType_Bool,             BatteryChargeInfo)\
        ENTRY(OtgRequested,                         189,    FieldType_Bool,             BatteryChargeInfo)\
        ENTRY(NANDPreEolInfo,                       190,    FieldType_NumericU32,       NANDExtendedCsd)\
        ENTRY(NANDDeviceLifeTimeEstTypA,            191,    FieldType_NumericU32,       NANDExtendedCsd)\
        ENTRY(NANDDeviceLifeTimeEstTypB,            192,    FieldType_NumericU32,       NANDExtendedCsd)\
        ENTRY(NANDPatrolCount,                      193,    FieldType_NumericU32,       NANDPatrolInfo)\
        ENTRY(NANDNumActivationFailures,            194,    FieldType_NumericU32,       NANDErrorInfo)\
        ENTRY(NANDNumActivationErrorCorrections,    195,    FieldType_NumericU32,       NANDErrorInfo)\
        ENTRY(NANDNumReadWriteFailures,             196,    FieldType_NumericU32,       NANDErrorInfo)\
        ENTRY(NANDNumReadWriteErrorCorrections,     197,    FieldType_NumericU32,       NANDErrorInfo)\
        ENTRY(NANDErrorLog,                         198,    FieldType_String,           NANDDriverLog)\
        ENTRY(SdCardUserAreaSize,                   199,    FieldType_NumericI64,       SdCardSizeSpec)\
        ENTRY(SdCardProtectedAreaSize,              200,    FieldType_NumericI64,       SdCardSizeSpec)\
        ENTRY(SdCardNumActivationFailures,          201,    FieldType_NumericU32,       SdCardErrorInfo)\
        ENTRY(SdCardNumActivationErrorCorrections,  202,    FieldType_NumericU32,       SdCardErrorInfo)\
        ENTRY(SdCardNumReadWriteFailures,           203,    FieldType_NumericU32,       SdCardErrorInfo)\
        ENTRY(SdCardNumReadWriteErrorCorrections,   204,    FieldType_NumericU32,       SdCardErrorInfo)\
        ENTRY(SdCardErrorLog,                       205,    FieldType_String,           SdCardDriverLog )\
        ENTRY(EncryptionKey,                        206,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(EncryptedExceptionInfo,               207,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(GameCardTimeoutRetryErrorCount,       208,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(FsRemountForDataCorruptCount,         209,    FieldType_NumericU32,       FsProxyErrorInfo)\
        ENTRY(FsRemountForDataCorruptRetryOutCount, 210,    FieldType_NumericU32,       FsProxyErrorInfo)\
        ENTRY(GameCardInsertionCount,               211,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardRemovalCount,                 212,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardAsicInitializeCount,          213,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(TestU16,                              214,    FieldType_NumericU16,       Test)\
        ENTRY(TestU8,                               215,    FieldType_NumericU8,        Test)\
        ENTRY(TestI16,                              216,    FieldType_NumericI16,       Test)\
        ENTRY(TestI8,                               217,    FieldType_NumericI8,        Test)\
        ENTRY(SystemAppletScene,                    218,    FieldType_NumericU8,        SystemAppletSceneInfo)\
        ENTRY(CodecType,                            219,    FieldType_NumericU32,       VideoInfo)\
        ENTRY(DecodeBuffers,                        220,    FieldType_NumericU32,       VideoInfo)\
        ENTRY(FrameWidth,                           221,    FieldType_NumericI32,       VideoInfo)\
        ENTRY(FrameHeight,                          222,    FieldType_NumericI32,       VideoInfo)\
        ENTRY(ColorPrimaries,                       223,    FieldType_NumericU8,        VideoInfo)\
        ENTRY(TransferCharacteristics,              224,    FieldType_NumericU8,        VideoInfo)\
        ENTRY(MatrixCoefficients,                   225,    FieldType_NumericU8,        VideoInfo)\
        ENTRY(DisplayWidth,                         226,    FieldType_NumericI32,       VideoInfo)\
        ENTRY(DisplayHeight,                        227,    FieldType_NumericI32,       VideoInfo)\
        ENTRY(DARWidth,                             228,    FieldType_NumericI32,       VideoInfo)\
        ENTRY(DARHeight,                            229,    FieldType_NumericI32,       VideoInfo)\
        ENTRY(ColorFormat,                          230,    FieldType_NumericU32,       VideoInfo)\
        ENTRY(ColorSpace,                           231,    FieldType_String,           VideoInfo)\
        ENTRY(SurfaceLayout,                        232,    FieldType_String,           VideoInfo)\
        ENTRY(BitStream,                            233,    FieldType_U8Array,          VideoInfo)\
        ENTRY(VideoDecState,                        234,    FieldType_String,           VideoInfo)\
        ENTRY(GpuErrorChannelId,                    235,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorAruId,                        236,    FieldType_NumericU64,       GpuErrorInfo)\
        ENTRY(GpuErrorType,                         237,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorFaultInfo,                    238,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorWriteAccess,                  239,    FieldType_Bool,             GpuErrorInfo)\
        ENTRY(GpuErrorFaultAddress,                 240,    FieldType_NumericU64,       GpuErrorInfo)\
        ENTRY(GpuErrorFaultUnit,                    241,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorFaultType,                    242,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorHwContextPointer,             243,    FieldType_NumericU64,       GpuErrorInfo)\
        ENTRY(GpuErrorContextStatus,                244,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorPbdmaIntr,                    245,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorPbdmaErrorType,               246,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorPbdmaHeaderShadow,            247,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorPbdmaHeader,                  248,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorPbdmaGpShadow0,               249,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(GpuErrorPbdmaGpShadow1,               250,    FieldType_NumericU32,       GpuErrorInfo)\
        ENTRY(AccessPointChannel,                   251,    FieldType_NumericU16,       AccessPointInfo)\
        ENTRY(ThreadName,                           252,    FieldType_String,           ErrorInfo)\
        ENTRY(AdspExceptionRegisters,               253,    FieldType_U32Array,         AdspErrorInfo)\
        ENTRY(AdspExceptionSpsr,                    254,    FieldType_NumericU32,       AdspErrorInfo)\
        ENTRY(AdspExceptionProgramCounter,          255,    FieldType_NumericU32,       AdspErrorInfo)\
        ENTRY(AdspExceptionLinkRegister,            256,    FieldType_NumericU32,       AdspErrorInfo)\
        ENTRY(AdspExceptionStackPointer,            257,    FieldType_NumericU32,       AdspErrorInfo)\
        ENTRY(AdspExceptionArmModeRegisters,        258,    FieldType_U32Array,         AdspErrorInfo)\
        ENTRY(AdspExceptionStackAddress,            259,    FieldType_NumericU32,       AdspErrorInfo)\
        ENTRY(AdspExceptionStackDump,               260,    FieldType_U32Array,         AdspErrorInfo)\
        ENTRY(AdspExceptionReason,                  261,    FieldType_NumericU32,       AdspErrorInfo)\
        ENTRY(OscillatorClock,                      262,    FieldType_NumericU32,       PowerClockInfo)\
        ENTRY(CpuDvfsTableClocks,                   263,    FieldType_U32Array,         PowerClockInfo)\
        ENTRY(CpuDvfsTableVoltages,                 264,    FieldType_I32Array,         PowerClockInfo)\
        ENTRY(GpuDvfsTableClocks,                   265,    FieldType_U32Array,         PowerClockInfo)\
        ENTRY(GpuDvfsTableVoltages,                 266,    FieldType_I32Array,         PowerClockInfo)\
        ENTRY(EmcDvfsTableClocks,                   267,    FieldType_U32Array,         PowerClockInfo)\
        ENTRY(EmcDvfsTableVoltages,                 268,    FieldType_I32Array,         PowerClockInfo)\
        ENTRY(ModuleClockFrequencies,               269,    FieldType_U32Array,         PowerClockInfo)\
        ENTRY(ModuleClockEnableFlags,               270,    FieldType_U8Array,          PowerClockInfo)\
        ENTRY(ModulePowerEnableFlags,               271,    FieldType_U8Array,          PowerClockInfo)\
        ENTRY(ModuleResetAssertFlags,               272,    FieldType_U8Array,          PowerClockInfo)\
        ENTRY(ModuleMinimumVoltageClockRates,       273,    FieldType_U32Array,         PowerClockInfo)\
        ENTRY(PowerDomainEnableFlags,               274,    FieldType_U8Array,          PowerClockInfo)\
        ENTRY(PowerDomainVoltages,                  275,    FieldType_I32Array,         PowerClockInfo)\
        ENTRY(AccessPointRssi,                      276,    FieldType_NumericI32,       RadioStrengthInfo)\
        ENTRY(FuseInfo,                             277,    FieldType_U32Array,         PowerClockInfo)\
        ENTRY(VideoLog,                             278,    FieldType_String,           VideoInfo)\
        ENTRY(GameCardDeviceId,                     279,    FieldType_U8Array,          GameCardCIDInfo)\
        ENTRY(GameCardAsicReinitializeCount,        280,    FieldType_NumericU16,       GameCardErrorInfo)\
        ENTRY(GameCardAsicReinitializeFailureCount, 281,    FieldType_NumericU16,       GameCardErrorInfo)\
        ENTRY(GameCardAsicReinitializeFailureDetail,282,    FieldType_NumericU16,       GameCardErrorInfo)\
        ENTRY(GameCardRefreshSuccessCount,          283,    FieldType_NumericU16,       GameCardErrorInfo)\
        ENTRY(GameCardAwakenCount,                  284,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardAwakenFailureCount,           285,    FieldType_NumericU16,       GameCardErrorInfo)\
        ENTRY(GameCardReadCountFromInsert,          286,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardReadCountFromAwaken,          287,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardLastReadErrorPageAddress,     288,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(GameCardLastReadErrorPageCount,       289,    FieldType_NumericU32,       GameCardErrorInfo)\
        ENTRY(AppletManagerContextTrace,            290,    FieldType_I32Array,         ErrorInfo)\
        ENTRY(NvDispIsRegistered,                   291,    FieldType_Bool,             NvDispDeviceInfo)\
        ENTRY(NvDispIsSuspend,                      292,    FieldType_Bool,             NvDispDeviceInfo)\
        ENTRY(NvDispDC0SurfaceNum,                  293,    FieldType_I32Array,         NvDispDeviceInfo)\
        ENTRY(NvDispDC1SurfaceNum,                  294,    FieldType_I32Array,         NvDispDeviceInfo)\
        ENTRY(NvDispWindowSrcRectX,                 295,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowSrcRectY,                 296,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowSrcRectWidth,             297,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowSrcRectHeight,            298,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowDstRectX,                 299,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowDstRectY,                 300,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowDstRectWidth,             301,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowDstRectHeight,            302,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowIndex,                    303,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowBlendOperation,           304,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowAlphaOperation,           305,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowDepth,                    306,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowAlpha,                    307,    FieldType_NumericU8,        NvDispDcWindowInfo)\
        ENTRY(NvDispWindowHFilter,                  308,    FieldType_Bool,             NvDispDcWindowInfo)\
        ENTRY(NvDispWindowVFilter,                  309,    FieldType_Bool,             NvDispDcWindowInfo)\
        ENTRY(NvDispWindowOptions,                  310,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispWindowSyncPointId,              311,    FieldType_NumericU32,       NvDispDcWindowInfo)\
        ENTRY(NvDispDPSorPower,                     312,    FieldType_Bool,             NvDispDpModeInfo)\
        ENTRY(NvDispDPClkType,                      313,    FieldType_NumericU8,        NvDispDpModeInfo)\
        ENTRY(NvDispDPEnable,                       314,    FieldType_NumericU32,       NvDispDpModeInfo)\
        ENTRY(NvDispDPState,                        315,    FieldType_NumericU32,       NvDispDpModeInfo)\
        ENTRY(NvDispDPEdid,                         316,    FieldType_U8Array,          NvDispDpModeInfo)\
        ENTRY(NvDispDPEdidSize,                     317,    FieldType_NumericU32,       NvDispDpModeInfo)\
        ENTRY(NvDispDPEdidExtSize,                  318,    FieldType_NumericU32,       NvDispDpModeInfo)\
        ENTRY(NvDispDPFakeMode,                     319,    FieldType_Bool,             NvDispDpModeInfo)\
        ENTRY(NvDispDPModeNumber,                   320,    FieldType_NumericU32,       NvDispDpModeInfo)\
        ENTRY(NvDispDPPlugInOut,                    321,    FieldType_Bool,             NvDispDpModeInfo)\
        ENTRY(NvDispDPAuxIntHandler,                322,    FieldType_Bool,             NvDispDpModeInfo)\
        ENTRY(NvDispDPForceMaxLinkBW,               323,    FieldType_Bool,             NvDispDpModeInfo)\
        ENTRY(NvDispDPIsConnected,                  324,    FieldType_Bool,             NvDispDpModeInfo)\
        ENTRY(NvDispDPLinkValid,                    325,    FieldType_Bool,             NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkMaxBW,                    326,    FieldType_NumericU8,        NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkMaxLaneCount,             327,    FieldType_NumericU8,        NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkDownSpread,               328,    FieldType_Bool,             NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkSupportEnhancedFraming,   329,    FieldType_Bool,             NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkBpp,                      330,    FieldType_NumericU32,       NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkScaramberCap,             331,    FieldType_Bool,             NvDispDpLinkSpec)\
        ENTRY(NvDispDPLinkBW,                       332,    FieldType_NumericU8,        NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkLaneCount,                333,    FieldType_NumericU8,        NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkEnhancedFraming,          334,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkScrambleEnable,           335,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkActivePolarity,           336,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkActiveCount,              337,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkTUSize,                   338,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkActiveFrac,               339,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkWatermark,                340,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkHBlank,                   341,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkVBlank,                   342,    FieldType_NumericU32,       NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkOnlyEnhancedFraming,      343,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkOnlyEdpCap,               344,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkSupportFastLT,            345,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkLTDataValid,              346,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkTsp3Support,              347,    FieldType_Bool,             NvDispDpLinkStatus)\
        ENTRY(NvDispDPLinkAuxInterval,              348,    FieldType_NumericU8,        NvDispDpLinkStatus)\
        ENTRY(NvDispHdcpCreated,                    349,    FieldType_Bool,             NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpUserRequest,                350,    FieldType_Bool,             NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpPlugged,                    351,    FieldType_Bool,             NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpState,                      352,    FieldType_NumericU32,       NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpFailCount,                  353,    FieldType_NumericI32,       NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpHdcp22,                     354,    FieldType_NumericI8,        NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpMaxRetry,                   355,    FieldType_NumericU8,        NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpHpd,                        356,    FieldType_NumericU8,        NvDispDpHdcpInfo)\
        ENTRY(NvDispHdcpRepeater,                   357,    FieldType_NumericU8,        NvDispDpHdcpInfo)\
        ENTRY(NvDispCecRxBuf,                       358,    FieldType_U8Array,          NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecRxLength,                    359,    FieldType_NumericI32,       NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecTxBuf,                       360,    FieldType_U8Array,          NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecTxLength,                    361,    FieldType_NumericI32,       NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecTxRet,                       362,    FieldType_NumericI32,       NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecState,                       363,    FieldType_NumericU32,       NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecTxInfo,                      364,    FieldType_NumericU8,        NvDispDpAuxCecInfo)\
        ENTRY(NvDispCecRxInfo,                      365,    FieldType_NumericU8,        NvDispDpAuxCecInfo)\
        ENTRY(NvDispDCIndex,                        366,    FieldType_NumericU32,       NvDispDcInfo)\
        ENTRY(NvDispDCInitialize,                   367,    FieldType_Bool,             NvDispDcInfo)\
        ENTRY(NvDispDCClock,                        368,    FieldType_Bool,             NvDispDcInfo)\
        ENTRY(NvDispDCFrequency,                    369,    FieldType_NumericU32,       NvDispDcInfo)\
        ENTRY(NvDispDCFailed,                       370,    FieldType_Bool,             NvDispDcInfo)\
        ENTRY(NvDispDCModeWidth,                    371,    FieldType_NumericI32,       NvDispDcInfo)\
        ENTRY(NvDispDCModeHeight,                   372,    FieldType_NumericI32,       NvDispDcInfo)\
        ENTRY(NvDispDCModeBpp,                      373,    FieldType_NumericU32,       NvDispDcInfo)\
        ENTRY(NvDispDCPanelFrequency,               374,    FieldType_NumericU32,       NvDispDcInfo)\
        ENTRY(NvDispDCWinDirty,                     375,    FieldType_NumericU32,       NvDispDcInfo)\
        ENTRY(NvDispDCWinEnable,                    376,    FieldType_NumericU32,       NvDispDcInfo)\
        ENTRY(NvDispDCVrr,                          377,    FieldType_Bool,             NvDispDcInfo)\
        ENTRY(NvDispDCPanelInitialize,              378,    FieldType_Bool,             NvDispDcInfo)\
        ENTRY(NvDispDsiDataFormat,                  379,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiVideoMode,                   380,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiRefreshRate,                 381,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiLpCmdModeFrequency,          382,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiHsCmdModeFrequency,          383,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiPanelResetTimeout,           384,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiPhyFrequency,                385,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiFrequency,                   386,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDsiInstance,                    387,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDcDsiHostCtrlEnable,            388,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispDcDsiInit,                      389,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispDcDsiEnable,                    390,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispDcDsiHsMode,                    391,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispDcDsiVendorId,                  392,    FieldType_U8Array,          NvDispDsiInfo)\
        ENTRY(NvDispDcDsiLcdVendorNum,              393,    FieldType_NumericU8,        NvDispDsiInfo)\
        ENTRY(NvDispDcDsiHsClockControl,            394,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDcDsiEnableHsClockInLpMode,     395,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispDcDsiTeFrameUpdate,             396,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispDcDsiGangedType,                397,    FieldType_NumericU32,       NvDispDsiInfo)\
        ENTRY(NvDispDcDsiHbpInPktSeq,               398,    FieldType_Bool,             NvDispDsiInfo)\
        ENTRY(NvDispErrID,                          399,    FieldType_NumericU32,       NvDispErrIDInfo)\
        ENTRY(NvDispErrData0,                       400,    FieldType_NumericU32,       NvDispErrIDInfo)\
        ENTRY(NvDispErrData1,                       401,    FieldType_NumericU32,       NvDispErrIDInfo)\
        ENTRY(SdCardMountStatus,                    402,    FieldType_String,           SdCardMountInfo)\
        ENTRY(SdCardMountUnexpectedResult,          403,    FieldType_String,           SdCardMountInfo)\
        ENTRY(NANDTotalSize,                        404,    FieldType_NumericU64,       NANDFreeSpaceInfo)\
        ENTRY(SdCardTotalSize,                      405,    FieldType_NumericU64,       SDCardFreeSpaceInfo)\
        ENTRY(ElapsedTimeSinceInitialLaunch,        406,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(ElapsedTimeSincePowerOn,              407,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(ElapsedTimeSinceLastAwake,            408,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(OccurrenceTick,                       409,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(RetailInteractiveDisplayFlag,         410,    FieldType_Bool,             RetailInteractiveDisplayInfo)\
        ENTRY(FatFsError,                           411,    FieldType_NumericI32,       FsProxyErrorInfo)\
        ENTRY(FatFsExtraError,                      412,    FieldType_NumericI32,       FsProxyErrorInfo)\
        ENTRY(FatFsErrorDrive,                      413,    FieldType_NumericI32,       FsProxyErrorInfo)\
        ENTRY(FatFsErrorName,                       414,    FieldType_String,           FsProxyErrorInfo)\
        ENTRY(MonitorManufactureCode,               415,    FieldType_String,           MonitorCapability)\
        ENTRY(MonitorProductCode,                   416,    FieldType_NumericU16,       MonitorCapability)\
        ENTRY(MonitorSerialNumber,                  417,    FieldType_NumericU32,       MonitorCapability)\
        ENTRY(MonitorManufactureYear,               418,    FieldType_NumericI32,       MonitorCapability)\
        ENTRY(PhysicalAddress,                      419,    FieldType_NumericU16,       MonitorCapability)\
        ENTRY(Is4k60Hz,                             420,    FieldType_Bool,             MonitorCapability)\
        ENTRY(Is4k30Hz,                             421,    FieldType_Bool,             MonitorCapability)\
        ENTRY(Is1080P60Hz,                          422,    FieldType_Bool,             MonitorCapability)\
        ENTRY(Is720P60Hz,                           423,    FieldType_Bool,             MonitorCapability)\
        ENTRY(PcmChannelMax,                        424,    FieldType_NumericI32,       MonitorCapability)\
        ENTRY(CrashReportHash,                      425,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(ErrorReportSharePermission,           426,    FieldType_NumericU8,        ErrorReportSharePermissionInfo)\
        ENTRY(VideoCodecTypeEnum,                   427,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(VideoBitRate,                         428,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(VideoFrameRate,                       429,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(VideoWidth,                           430,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(VideoHeight,                          431,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(AudioCodecTypeEnum,                   432,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(AudioSampleRate,                      433,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(AudioChannelCount,                    434,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(AudioBitRate,                         435,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(MultimediaContainerType,              436,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(MultimediaProfileType,                437,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(MultimediaLevelType,                  438,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(MultimediaCacheSizeEnum,              439,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(MultimediaErrorStatusEnum,            440,    FieldType_NumericI32,       MultimediaInfo)\
        ENTRY(MultimediaErrorLog,                   441,    FieldType_U8Array,          MultimediaInfo)\
        ENTRY(ServerFqdn,                           442,    FieldType_String,           ErrorInfo)\
        ENTRY(ServerIpAddress,                      443,    FieldType_String,           ErrorInfo)\
        ENTRY(TestStringEncrypt,                    444,    FieldType_String,           Test, FieldFlag_Encrypt)\
        ENTRY(TestU8ArrayEncrypt,                   445,    FieldType_U8Array,          Test, FieldFlag_Encrypt)\
        ENTRY(TestU32ArrayEncrypt,                  446,    FieldType_U32Array,         Test, FieldFlag_Encrypt)\
        ENTRY(TestU64ArrayEncrypt,                  447,    FieldType_U64Array,         Test, FieldFlag_Encrypt)\
        ENTRY(TestI32ArrayEncrypt,                  448,    FieldType_I32Array,         Test, FieldFlag_Encrypt)\
        ENTRY(TestI64ArrayEncrypt,                  449,    FieldType_I64Array,         Test, FieldFlag_Encrypt)\
        ENTRY(CipherKey,                            450,    FieldType_U8Array,          ErrorInfoAuto)\
        ENTRY(FileSystemPath,                       451,    FieldType_String,           ErrorInfo, FieldFlag_Encrypt)\
        ENTRY(WebMediaPlayerOpenUrl,                452,    FieldType_String,           ErrorInfo, FieldFlag_Encrypt)\
        ENTRY(WebMediaPlayerLastSocketErrors,       453,    FieldType_I32Array,         ErrorInfo)\
        ENTRY(UnknownControllerCount,               454,    FieldType_NumericU8,        ConnectedControllerInfo)\
        ENTRY(AttachedControllerCount,              455,    FieldType_NumericU8,        ConnectedControllerInfo)\
        ENTRY(BluetoothControllerCount,             456,    FieldType_NumericU8,        ConnectedControllerInfo)\
        ENTRY(UsbControllerCount,                   457,    FieldType_NumericU8,        ConnectedControllerInfo)\
        ENTRY(ControllerTypeList,                   458,    FieldType_U8Array,          ConnectedControllerInfo)\
        ENTRY(ControllerInterfaceList,              459,    FieldType_U8Array,          ConnectedControllerInfo)\
        ENTRY(ControllerStyleList,                  460,    FieldType_U8Array,          ConnectedControllerInfo)\
        ENTRY(FsPooledBufferPeakFreeSize,           461,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsPooledBufferRetriedCount,           462,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsPooledBufferReduceAllocationCount,  463,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsBufferManagerPeakFreeSize,          464,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsBufferManagerRetriedCount,          465,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsExpHeapPeakFreeSize,                466,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsBufferPoolPeakFreeSize,             467,    FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsPatrolReadAllocateBufferSuccessCount,468,   FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(FsPatrolReadAllocateBufferFailureCount,469,   FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(SteadyClockInternalOffset,            470,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(SteadyClockCurrentTimePointValue,     471,    FieldType_NumericI64,       ErrorInfoAuto)\
        ENTRY(UserClockContextOffset,               472,    FieldType_NumericI64,       UserClockContextInfo)\
        ENTRY(UserClockContextTimeStampValue,       473,    FieldType_NumericI64,       UserClockContextInfo)\
        ENTRY(NetworkClockContextOffset,            474,    FieldType_NumericI64,       NetworkClockContextInfo)\
        ENTRY(NetworkClockContextTimeStampValue,    475,    FieldType_NumericI64,       NetworkClockContextInfo)\
        ENTRY(SystemAbortFlag,                      476,    FieldType_Bool,             ErrorInfo)\
        ENTRY(ApplicationAbortFlag,                 477,    FieldType_Bool,             ErrorInfo)\
        ENTRY(NifmErrorCode,                        478,    FieldType_String,           ConnectionStatusInfo)\
        ENTRY(LcsApplicationId,                     479,    FieldType_String,           ErrorInfo)\
        ENTRY(LcsContentMetaKeyIdList,              480,    FieldType_U64Array,         ErrorInfo)\
        ENTRY(LcsContentMetaKeyVersionList,         481,    FieldType_U32Array,         ErrorInfo)\
        ENTRY(LcsContentMetaKeyTypeList,            482,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(LcsContentMetaKeyInstallTypeList,     483,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(LcsSenderFlag,                        484,    FieldType_Bool,             ErrorInfo)\
        ENTRY(LcsApplicationRequestFlag,            485,    FieldType_Bool,             ErrorInfo)\
        ENTRY(LcsHasExFatDriverFlag,                486,    FieldType_Bool,             ErrorInfo)\
        ENTRY(LcsIpAddress,                         487,    FieldType_NumericU32,       ErrorInfo)\
        ENTRY(AcpStartupUserAccount,                488,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpAocRegistrationType,               489,    FieldType_NumericU8,        AcpAocSettingsInfo)\
        ENTRY(AcpAttributeFlag,                     490,    FieldType_NumericU32,       AcpGeneralSettingsInfo)\
        ENTRY(AcpSupportedLanguageFlag,             491,    FieldType_NumericU32,       AcpGeneralSettingsInfo)\
        ENTRY(AcpParentalControlFlag,               492,    FieldType_NumericU32,       AcpGeneralSettingsInfo)\
        ENTRY(AcpScreenShot,                        493,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpVideoCapture,                      494,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpDataLossConfirmation,              495,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpPlayLogPolicy,                     496,    FieldType_NumericU8,        AcpPlayLogSettingsInfo)\
        ENTRY(AcpPresenceGroupId,                   497,    FieldType_NumericU64,       AcpGeneralSettingsInfo)\
        ENTRY(AcpRatingAge,                         498,    FieldType_I8Array,          AcpRatingSettingsInfo)\
        ENTRY(AcpAocBaseId,                         499,    FieldType_NumericU64,       AcpAocSettingsInfo)\
        ENTRY(AcpSaveDataOwnerId,                   500,    FieldType_NumericU64,       AcpStorageSettingsInfo)\
        ENTRY(AcpUserAccountSaveDataSize,           501,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpUserAccountSaveDataJournalSize,    502,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpDeviceSaveDataSize,                503,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpDeviceSaveDataJournalSize,         504,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpBcatDeliveryCacheStorageSize,      505,    FieldType_NumericI64,       AcpBcatSettingsInfo)\
        ENTRY(AcpApplicationErrorCodeCategory,      506,    FieldType_String,           AcpGeneralSettingsInfo)\
        ENTRY(AcpLocalCommunicationId,              507,    FieldType_U64Array,         AcpGeneralSettingsInfo)\
        ENTRY(AcpLogoType,                          508,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpLogoHandling,                      509,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpRuntimeAocInstall,                 510,    FieldType_NumericU8,        AcpAocSettingsInfo)\
        ENTRY(AcpCrashReport,                       511,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpHdcp,                              512,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(AcpSeedForPseudoDeviceId,             513,    FieldType_NumericU64,       AcpGeneralSettingsInfo)\
        ENTRY(AcpBcatPassphrase,                    514,    FieldType_String,           AcpBcatSettingsInfo)\
        ENTRY(AcpUserAccountSaveDataSizeMax,        515,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpUserAccountSaveDataJournalSizeMax, 516,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpDeviceSaveDataSizeMax,             517,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpDeviceSaveDataJournalSizeMax,      518,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpTemporaryStorageSize,              519,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpCacheStorageSize,                  520,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpCacheStorageJournalSize,           521,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpCacheStorageDataAndJournalSizeMax, 522,    FieldType_NumericI64,       AcpStorageSettingsInfo)\
        ENTRY(AcpCacheStorageIndexMax,              523,    FieldType_NumericI16,       AcpStorageSettingsInfo)\
        ENTRY(AcpPlayLogQueryableApplicationId,     524,    FieldType_U64Array,         AcpPlayLogSettingsInfo)\
        ENTRY(AcpPlayLogQueryCapability,            525,    FieldType_NumericU8,        AcpPlayLogSettingsInfo)\
        ENTRY(AcpRepairFlag,                        526,    FieldType_NumericU8,        AcpGeneralSettingsInfo)\
        ENTRY(RunningApplicationPatchStorageLocation,527,   FieldType_String,           RunningApplicationInfo)\
        ENTRY(RunningApplicationVersionNumber,      528,    FieldType_NumericU32,       RunningApplicationInfo)\
        ENTRY(FsRecoveredByInvalidateCacheCount,    529,    FieldType_NumericU32,       FsProxyErrorInfo)\
        ENTRY(FsSaveDataIndexCount,                 530,    FieldType_NumericU32,       FsProxyErrorInfo)\
        ENTRY(FsBufferManagerPeakTotalAllocatableSize, 531, FieldType_NumericU64,       FsMemoryInfo)\
        ENTRY(MonitorCurrentWidth,                  532,    FieldType_NumericU16,       MonitorSettings)\
        ENTRY(MonitorCurrentHeight,                 533,    FieldType_NumericU16,       MonitorSettings)\
        ENTRY(MonitorCurrentRefreshRate,            534,    FieldType_String,           MonitorSettings)\
        ENTRY(RebootlessSystemUpdateVersion,        535,    FieldType_String,           RebootlessSystemUpdateVersionInfo)\
        ENTRY(EncryptedExceptionInfo1,              536,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(EncryptedExceptionInfo2,              537,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(EncryptedExceptionInfo3,              538,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(EncryptedDyingMessage,                539,    FieldType_U8Array,          ErrorInfo)\
        ENTRY(DramId,                               540,    FieldType_NumericU32,       PowerClockInfo)\
        ENTRY(NifmConnectionTestRedirectUrl,        541,    FieldType_String,           NifmConnectionTestInfo)\
        ENTRY(AcpRequiredNetworkServiceLicenseOnLaunchFlag, 542, FieldType_NumericU8,   AcpGeneralSettingsInfo)\
        ENTRY(PciePort0Flags,                       543,    FieldType_NumericU32,       PcieLoggedStateInfo)\
        ENTRY(PciePort0Speed,                       544,    FieldType_NumericU8,        PcieLoggedStateInfo)\
        ENTRY(PciePort0ResetTimeInUs,               545,    FieldType_NumericU32,       PcieLoggedStateInfo)\
        ENTRY(PciePort0IrqCount,                    546,    FieldType_NumericU32,       PcieLoggedStateInfo)\
        ENTRY(PciePort0Statistics,                  547,    FieldType_U32Array,         PcieLoggedStateInfo)\
        ENTRY(PciePort1Flags,                       548,    FieldType_NumericU32,       PcieLoggedStateInfo)\
        ENTRY(PciePort1Speed,                       549,    FieldType_NumericU8,        PcieLoggedStateInfo)\
        ENTRY(PciePort1ResetTimeInUs,               550,    FieldType_NumericU32,       PcieLoggedStateInfo)\
        ENTRY(PciePort1IrqCount,                    551,    FieldType_NumericU32,       PcieLoggedStateInfo)\
        ENTRY(PciePort1Statistics,                  552,    FieldType_U32Array,         PcieLoggedStateInfo)\
        ENTRY(PcieFunction0VendorId,                553,    FieldType_NumericU16,       PcieLoggedStateInfo)\
        ENTRY(PcieFunction0DeviceId,                554,    FieldType_NumericU16,       PcieLoggedStateInfo)\
        ENTRY(PcieFunction0PmState,                 555,    FieldType_NumericU8,        PcieLoggedStateInfo)\
        ENTRY(PcieFunction0IsAcquired,              556,    FieldType_Bool,             PcieLoggedStateInfo)\
        ENTRY(PcieFunction1VendorId,                557,    FieldType_NumericU16,       PcieLoggedStateInfo)\
        ENTRY(PcieFunction1DeviceId,                558,    FieldType_NumericU16,       PcieLoggedStateInfo)\
        ENTRY(PcieFunction1PmState,                 559,    FieldType_NumericU8,        PcieLoggedStateInfo)\
        ENTRY(PcieFunction1IsAcquired,              560,    FieldType_Bool,             PcieLoggedStateInfo)\
        ENTRY(PcieGlobalRootComplexStatistics,      561,    FieldType_U32Array,         PcieLoggedStateInfo)\
        ENTRY(PciePllResistorCalibrationValue,      562,    FieldType_NumericU8,        PcieLoggedStateInfo)\
// ~FOREACH_ID(ENTRY)

