﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ECショップの機能を提供する本体機能向けAPI定義
 */

#pragma once

#include <nn/nn_StaticAssert.h>
#include <nn/applet/applet_Storage.h>
#include <nn/ec/ec_ShopTypesForSystem.h>
#include <nn/ec/ec_ResultShop.h>

namespace nn { namespace ec {

/**
 * @name ニンテンドーeショップの機能を利用するための本体機能向け関数
 * @{
 */

/**
 * @brief 1文字以上4文字以下の文字列を SourceId に変換します。
 * @tparam N 終端のNULL文字を含めた文字列の長さ
 * @param strId 1文字以上4文字以下の変換する文字列
 * @return 変換された SourceId の値
 * @pre
 *  - N >= 2 && N <= 5
 * @details
 *  - 文字列は終端のNULL文字を除いて1文字以上4文字以下である必要があります。
 *    N は終端のNULL文字を含めた長さとして扱っており、
 *    事前条件はそのNULL文字を含めた長さでチェックされます。
 *  - 「MakeSourceIdFromString("s_id")」のように文字列リテラルを
 *    SourceId に変換する際に使用します。
 */
template <size_t N>
inline SourceId MakeSourceIdFromString(const char (& strId)[N]) NN_NOEXCEPT
{
    NN_STATIC_ASSERT(N >= 2 && N <= 5);
    SourceId id = {{strId[0], strId[1], strId[2], strId[3]}};
    return id;
}

/**
 * @brief ニンテンドーeショップを起動します。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void LaunchEShop(const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief アプリケーションの詳細情報と購入画面を表示します。
 * @param[in] applicationId 情報を表示したいアプリケーションの ApplicationId。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - applicationId != nn::ApplicationId::GetInvalidId()
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopApplicationInformationForMenu(nn::ApplicationId applicationId,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief アプリケーションの詳細情報と購入画面を表示します。
 * @param[in] nsuidApplication 情報を表示したいアプリケーションの NsUid。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - nsuidApplication != NsUid::GetInvalidId()
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopApplicationInformationForMenu(NsUid nsuidApplication,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;
/**
 * @brief アプリケーションの詳細情報と購入画面を表示します。
 * @param[in] applicationId 情報を表示したいアプリケーションの ApplicationId。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @param[in] newsId 呼び出し元の情報として付加するニュースの識別子。
 *   不要、または存在しない場合は nullptr を指定してください。
 *   なお、末尾のNULL文字を除いて128文字以上の長さである場合は127文字に切り捨てられます。
 * @return なし
 * @pre
 *  - applicationId != nn::ApplicationId::GetInvalidId()
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopApplicationInformationForMenu(nn::ApplicationId applicationId,
    const nn::account::Uid& uid, SourceId sourceId, const char* newsId) NN_NOEXCEPT;

/**
 * @brief アプリケーションの詳細情報と購入画面を表示します。
 * @param[in] nsuidApplication 情報を表示したいアプリケーションの NsUid。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @param[in] newsId 呼び出し元の情報として付加するニュースの識別子。
 *   不要、または存在しない場合は nullptr を指定してください。
 *   なお、末尾のNULL文字を除いて128文字以上の長さである場合は127文字に切り捨てられます。
 * @return なし
 * @pre
 *  - nsuidApplication != NsUid::GetInvalidId()
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopApplicationInformationForMenu(NsUid nsuidApplication,
    const nn::account::Uid& uid, SourceId sourceId, const char* newsId) NN_NOEXCEPT;

/**
 * @brief 追加コンテンツの一覧と購入画面を表示します。
 * @param[in] applicationIdParent 一覧として表示する追加コンテンツに設定されたアプリケーションID。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - applicationIdParent != nn::ApplicationId::GetInvalidId()
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopAddOnContentListForMenu(nn::ApplicationId applicationIdParent,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief 追加コンテンツの一覧と購入画面を表示します。
 * @param[in] nsuidAddOnContentItem 詳細情報を表示したい追加コンテンツそのものを表す NsUid
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - nsuidAddOnContentItem != NsUid::GetInvalidId()
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopAddOnContentDetailsForMenu(NsUid nsuidAddOnContentItem,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief クーポンの一覧または詳細情報と購入画面を表示します。
 * @param[in] couponId 詳細情報を表示したいクーポンを表す CouponId。
 *   CouponId::GetInvalidId() を指定した場合はクーポンの一覧を表示します。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopCouponForMenu(CouponId couponId,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief 指定されたパラメーターに基づくショップのアクションを提供します。
 * @param[in] params 任意のパラメーター。2048バイト以下の長さ(NULL文字除く)にする必要があります。
 *   NULL文字までの長さがNULL文字を除いて2048文字を超える場合、その分は切り捨てられます。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - params != nullptr && strlen(params) > 0
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopInformationWithParams(const char* params,
    const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief アプリケーションの詳細情報と購入画面を表示します。
 * @param[in] applicationId 情報を表示したいアプリケーションの ApplicationId。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] isUnsearchableAllowed 検索非公開アプリケーションの詳細情報表示を許可する場合は true
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - applicationId != nn::ApplicationId::GetInvalidId()
 *
 * @details
 * - 本関数は isUnsearchableAllowed に true を指定することで、
 *   検索非公開のアプリケーションでも詳細情報画面にジャンプすることができます。
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopApplicationInformationForMenu(nn::ApplicationId applicationId,
    const nn::account::Uid& uid, bool isUnsearchableAllowed,
    SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief アプリケーションの詳細情報と購入画面を表示します。
 * @param[in] nsuidApplication 情報を表示したいアプリケーションの NsUid。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] isUnsearchableAllowed 検索非公開であるアプリケーションの詳細情報表示を許可する場合は true
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - nsuidApplication != NsUid::GetInvalidId()
 *
 * @details
 * - 本関数は isUnsearchableAllowed に true を指定することで、
 *   検索非公開のアプリケーションでも詳細情報画面にジャンプすることができます。
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopApplicationInformationForMenu(NsUid nsuidApplication,
    const nn::account::Uid& uid, bool isUnsearchableAllowed,
    SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief   アプリケーションの詳細情報と購入画面を表示します。
 *
 * @param[in] applicationId 情報を表示したいアプリケーションの ApplicationId
 *
 * @pre
 *  - 自身がライブラリアプレットである
 *
 * @details
 *  アプリケーションの詳細情報と購入画面のショップページを起動します。
 *
 *  本関数は内部で UI 表示の処理を行うため、呼出元スレッドをブロックします。
 *  また、内部で必要に応じてユーザ選択 UI を表示します。
 *
 */
void ShowShopApplicationInformationForApplet(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

/**
 * @brief 指定ユーザーに対するショップの設定項目を表示します。
 * @param[in] uid 設定対象のユーザーの Uid 値。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 * @pre
 *  - uid != nn::account::InvalidUid
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopSettingsForMenu(const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief 会員サービスの権利に関する情報と購入画面を表示します。
 * @param[in] uid 購入対象のユーザーの Uid 値。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return 処理の結果
 * @retval nn::ResultSuccess ユーザーによる購入処理が行われてそのまま画面表示を終了しました。
 * @retval nn::ec::ResultShopCanceled ユーザーによって操作がキャンセルされました。
 * @pre
 *  - uid != nn::account::InvalidUid
 *
 * @details
 * - 購入処理が行われた場合であっても、画面表示の終了方法によっては
 *   nn::ec::ResultShopCanceled が返る場合があります。
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - 本関数の戻り値は、適切なストレージハンドルを用いることで
 *   @ref GetShopResultFromStorage 経由で取得することができます。
 */
nn::Result ShowShopMembershipServiceInformation(const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;

/**
 * @brief 購入画面等の表示を行った際の処理の結果をストレージから取得します。
 * @param[in] handle ストレージを指すハンドル
 * @return 処理の結果
 *
 * @details
 * この関数は、本関数を利用できる旨の説明がある関数を呼び出した後でのみ利用できます。
 * それ以外の場合、戻り値は nn::ec::ResultShopCanceled が返ります。
 */
nn::Result GetShopResultFromStorage(const nn::applet::StorageHandle& handle) NN_NOEXCEPT;

/**
 * @brief プリインストールコンテンツの引き換え画面を表示します。
 * @param[in] uid ユーザーを明示的に指定する場合はその Uid 値。
 *   uid に @ref nn::account::InvalidUid を指定した場合は、
 *   本関数は内部でユーザー選択UIを表示してユーザーの選択を行います。
 * @param[in] sourceId 呼び出し元機能を表す SourceId_* の各値
 * @return なし
 *
 * @details
 * - 本関数は内部でUI表示を行うため、呼び出し元スレッドをブロックします。
 * - uid にネットワークサービスアカウントが紐付いているかどうかのチェックはここでは行いません。
 */
void ShowShopPreinstallInformationForMenu(const nn::account::Uid& uid, SourceId sourceId) NN_NOEXCEPT;
/**
 * @}
 */

// N = 2 の特殊化
template <> inline SourceId MakeSourceIdFromString<2>(const char(&strId)[2]) NN_NOEXCEPT
{
    SourceId id = { { strId[0], 0, 0, 0 } };
    return id;
}
// N = 3 の特殊化
template <> inline SourceId MakeSourceIdFromString<3>(const char(&strId)[3]) NN_NOEXCEPT
{
    SourceId id = { { strId[0], strId[1], 0, 0 } };
    return id;
}
// N = 4 の特殊化
template <> inline SourceId MakeSourceIdFromString<4>(const char(&strId)[4]) NN_NOEXCEPT
{
    SourceId id = { { strId[0], strId[1], strId[2], 0 } };
    return id;
}

}}
